package de.fzi.wim.trie.extractor;

import java.util.Vector;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

/**
 * Nomen est Omen
 *
 * @author <a href="zach@fzi.de">Valentin Zacharias</a>
 */
public class TrieNode implements java.io.Serializable{
    /**
     * If true, the Pointer objects are cloned before passed to the
     * satisfied method. If false they are not. Default is true
     * which increases the speed, can lead to wrong behavior when
     * data is stored in the pointer.*/
    public static final boolean CLONE_POINTER_SAT = true;

    /**
     * A small class to increase the usability of the HashMap - to allow more than one entry per key.
     */
    private static class Bag implements java.io.Serializable {
        TrieNode[] nodes;

        public Object clone(Set allNodes) {
            TrieNode[] newNodes = new TrieNode[nodes.length];
            for (int i=0;i<newNodes.length;i++) {
                newNodes[i] = (TrieNode) nodes[i].clone(allNodes);
            }
            return new Bag(newNodes);
        }

        Bag (TrieNode node) {
            nodes = new TrieNode[1];
            nodes[0] = node;
        }

        Bag (TrieNode[] nodes) {
            this.nodes = nodes;
        }

        Bag (Vector nodes) {
            this.nodes = new TrieNode[nodes.size()];
            for (int i=0;i<this.nodes.length;i++) {
                this.nodes[i] = (TrieNode) nodes.elementAt(i);
            }
        }

        TrieNode[] getNodes() {
            return nodes;
        }

        void addNode(TrieNode toAdd) {
            TrieNode[] temp = new TrieNode[nodes.length+1];
            for (int i=0;i<nodes.length;i++) temp[i] = nodes[i];
            temp[temp.length-1] = toAdd;
        }
    }

    private Vector listeners;
    private HashMap tokenTemplates = new HashMap();


    /**
     * Creates a TrieNode and places a reference in the given set.
     */
    public TrieNode(Set allNodes) {
        allNodes.add(this);
    }

    public Object clone(Set allNodes) {
        TrieNode newNode = new TrieNode(allNodes);
        if (listeners != null) {
            newNode.listeners = new Vector();
            for (int i=0;i<listeners.size();i++) {
                TrieListener current = (TrieListener) listeners.elementAt(i);
                newNode.listeners.addElement(current.clone());
            }
        }
        Iterator it = tokenTemplates.keySet().iterator();
        while (it.hasNext()) {
            TokenTemplate current = (TokenTemplate) it.next();
            newNode.tokenTemplates.put(current.clone(), ((Bag)tokenTemplates.get(current)).clone(allNodes));
        }
        return newNode;
    }

    public void addTrieListener (TrieListener toAdd) {
        if (listeners == null) listeners = new Vector();
        listeners.add(toAdd);
    }


    /**
     * Just a helping method for the trie to allow omitted characters
     */
    public void allowOmit() {
        Vector operations = new Vector();
        Iterator childs = getUniqueChilds().iterator();
        AnyTokenTemplate any = new AnyTokenTemplate();
        while (childs.hasNext()) {
            TrieNode current = (TrieNode) childs.next();
            Iterator it = current.tokenTemplates.keySet().iterator();
            while (it.hasNext()) {
                TokenTemplate currentTemplate = (TokenTemplate) it.next();
                if ((!currentTemplate.equals(any)) && (!(currentTemplate instanceof DecreaseConfidenceTokenTemplate))) {
                    //@xxx a little bit more code could speed the up the following lines
                    Bag bag = (Bag) current.tokenTemplates.get(currentTemplate);
                    TrieNode[] bagNodes = bag.getNodes();
                    DecreaseConfidenceTokenTemplate decreaser = new DecreaseConfidenceTokenTemplate(currentTemplate);
                    operations.addElement(new OmitStorer(decreaser,bagNodes));
                }
            }
        }
        for (int i=0;i<operations.size();i++) ((OmitStorer)operations.elementAt(i)).execute();
    }

    /**
     * stores information about yet to come omit operations
     */
    private class OmitStorer {
        TrieNode[] bagNodes;
        DecreaseConfidenceTokenTemplate decreaser;
        public OmitStorer(DecreaseConfidenceTokenTemplate decreaser, TrieNode[] bagNodes) {
            this.bagNodes = bagNodes;
            this.decreaser = decreaser;
        }

        public void execute() {
            for (int i=0;i<bagNodes.length;i++) addTokenTemplate(decreaser,bagNodes[i]);
        }
    }

    /**
     * @return          a HashSet with all unique childs.
     */
    public HashSet getUniqueChilds() {
        HashSet toReturn = new HashSet();
        Vector temp = getChilds();
        for (int i=0;i<temp.size();i++) {
            toReturn.add(temp.elementAt(i));
        }
        return toReturn;
    }


    /**
     * @return          a Vector with all childs of this element (not checked for equality).
     */
    public Vector getChilds() {
        Vector toReturn = new Vector();
        Iterator it = tokenTemplates.values().iterator();
        while (it.hasNext()) {
            Bag bag = (Bag) it.next();
            TrieNode[] nodes = bag.getNodes();
            for (int i=0;i<nodes.length;i++) {
                toReturn.addElement(nodes[i]);
            }
        }
        return toReturn;
    }



    /**
     * Inserts an "Any" Transition to all childs
     */
    public void allowWrongToken() {
        Vector childs = getChilds();
        Bag bag = new Bag(childs);
        tokenTemplates.put(new AnyTokenTemplate(),bag);
    }

    /**
     * Adds a token template and returns the node(s) referred by
     * this TokenTemplate (creates a new one if necessary)
     */
    public TrieNode[] addTokenTemplate(TokenTemplate template, Set allNodes) {
        TokenTemplate equalTemplate = getEqualTokenTemplate(template);
        if (equalTemplate != null) {
            Bag bag = (Bag)  tokenTemplates.get(equalTemplate);
            return bag.getNodes();
        }
        else {
            Bag bag = new Bag(new TrieNode(allNodes));
            tokenTemplates.put(template,bag);
            return bag.getNodes();
        }
    }

    /**
     * Notifies all TrieListeners that a pointer has reached this node.
     */
    public void notify(Pointer pointer, int index, String id) {
        if ((listeners != null) && (pointer.getConfidence() > Pointer.NOTIFY_CONFIDENCE)) {
            for (int i=0;i<listeners.size();i++) {
                TrieListener temp = (TrieListener) listeners.elementAt(i);
                temp.notify(pointer, index, id);
            }
        }
    }

    /**
     * Takes the TokenSet and the pointer and parses it (any new pointers are
     * stored in the given Vector)
     */
    public void parse(TokenSet tokenSet, Pointer pointer, Vector vector, String id) {
        Pointer newPointer = (Pointer) pointer.clone();

        Iterator it = tokenTemplates.keySet().iterator();
        while (it.hasNext()) {
            TokenTemplate temp =(TokenTemplate) it.next();
            if (temp.satisfied(tokenSet,newPointer)) {
                Bag bag = (Bag) tokenTemplates.get(temp);
                TrieNode[] nodes = bag.getNodes();
                for (int i=0;i<nodes.length;i++) {
                    newPointer.setTrieNode(nodes[i]);
                    if (newPointer.getConfidence() > Pointer.MINIMUM_CONFIDENCE) {
                        vector.addElement(newPointer);
                        nodes[i].notify(newPointer, tokenSet.getIndex(),id);
                    }
                    newPointer = (Pointer) newPointer.clone();
                }
            }
            if (CLONE_POINTER_SAT) newPointer = (Pointer) pointer.clone();
        }
    }

    public void addTokenTemplate(TokenTemplate template, TrieNode node) {
        TokenTemplate equalTemplate = getEqualTokenTemplate(template);
        if (equalTemplate != null) {
            Bag bag = (Bag) tokenTemplates.get(equalTemplate);
            bag.addNode(node);
        }
        else {
            Bag bag = new Bag(node);
            tokenTemplates.put(template,bag);
        }
    }

    public TokenTemplate getEqualTokenTemplate(TokenTemplate template) {
        Iterator it = tokenTemplates.keySet().iterator();
        while (it.hasNext()) {
            TokenTemplate temp =(TokenTemplate) it.next();
            if (temp.equals(template)) return temp;
        }
        return null;
    }

}
