package de.fzi.wim.trie.extractor;

import java.util.Vector;
import java.util.HashMap;

import de.fzi.wim.trie.extractor.character.*;
import de.fzi.wim.trie.EntryLocation;

/**
 * The real facade handling adding and stuff to the extractor trie.
 */
public class TrieImpl implements de.fzi.wim.trie.Trie {
    private Trie trie = new Trie();
    private Vector threadSaveListeners = new Vector();
    private boolean allowLevenstein = true;
    private boolean levenstein = false;

    private int idCreator = 0;


    /**
     * Adds an Entity to the trie. This Entity will be found, if
     * the given String is encountered.
     */
    public void addReference(String string,Object reference) {
        Vector templates = CharacterTokenTemplateFactory.createSimpleTemplates(string);
        ThreadSaveTrieListener tstl = new ThreadSaveTrieListener(reference);
        threadSaveListeners.addElement(tstl);
        trie.addTemplateVector(templates,tstl);
    }

    public boolean setAllowWrong(boolean allowWrong) {
        this.allowLevenstein = allowWrong;
        return true;
    }

    public synchronized String getUniqueId() {
        idCreator++;
        return new String(""+idCreator);
    }


    /**
     * parses the given string (using a trie allowing slightly wrong tokens)
     * @return              a Vector with the EntryLocations found in the text.
     */
    public Vector parse(String text) {
        String id = getUniqueId();
        Vector overlapTrieListeners = new Vector();
        Vector entryLocations = new Vector();

        for (int i=0;i<threadSaveListeners.size();i++) {
            ThreadSaveTrieListener tstl = (ThreadSaveTrieListener) threadSaveListeners.elementAt(i);
            OverlapTrieListener otl = new OverlapTrieListener(new StoringTrieListener(tstl.getReference(),entryLocations));
            overlapTrieListeners.addElement(otl);
            tstl.addListener(id,otl);
        }

        Vector textTokens = new Vector();
        CharacterToken.createCharacterTokens(textTokens,text);
        if (allowLevenstein && (!levenstein)) {
            trie.allowLevenstein();
            levenstein = true;
        }

        trie.parseTokenStore (new TokenTraverseStore(textTokens),id);

        for (int i=0;i<overlapTrieListeners.size();i++) {
            OverlapTrieListener listener = (OverlapTrieListener) overlapTrieListeners.elementAt(i);
            listener.end();
        }
        for (int i=0;i<threadSaveListeners.size();i++) {
            ThreadSaveTrieListener tstl = (ThreadSaveTrieListener) threadSaveListeners.elementAt(i);
            tstl.removeListener(id);
        }
        return entryLocations;
    }



    protected static class StoringTrieListener implements TrieListener {
        private Object reference;
        private Vector entryLocations;

        public StoringTrieListener(Object reference, Vector entryLocations) {
            this.reference = reference;
            this.entryLocations = entryLocations;
        }

        public void notify(Pointer pointer, int index, String threadID) {
            EntryLocation el = new EntryLocationImpl(pointer.getBeginIndex(), pointer.getBeginIndex()+index,reference,pointer.getConfidence());
            entryLocations.addElement(el);
        }

        public Object clone() {
            return this;
        }
    }



    /**
    * A class to make the trie thread save. It will only forward the notify method calls to the
    * previous registered TrieListener with the right thread id.
    */
    protected static class ThreadSaveTrieListener implements TrieListener {
        private HashMap listeners = new HashMap();
        private Object reference;

        public ThreadSaveTrieListener(Object reference) {
            this.reference = reference;
        }

        public Object getReference() {
            return reference;
        }

        public void addListener(String key, TrieListener list) {
            listeners.put(key,list);
        }

        public void removeListener(String key) {
            listeners.remove(key);
        }

        public Object clone() {
            return this;
        }

        public void notify(Pointer pointer, int index, String threadID) {
            TrieListener list = (TrieListener) listeners.get(threadID);
            list.notify(pointer,index,threadID);
        }
    }
}
