package de.fzi.wim.trie.extractor;

import java.util.Vector;
import java.util.HashSet;
import java.util.Iterator;

/**
 * The class really representing the Trie.
 *
 * @author <a href="zach@fzi.de">Valentin Zacharias</a>
 */
public class Trie implements java.io.Serializable {

    public static double DECREASE_CONFIDENCE = -0.2;
    public static double INCREASE_CONFIDENCE = 0.05;

    private HashSet nodes = new HashSet();
    private TrieNode rootNode;

    public Object clone() {
        Trie newTrie = new Trie();
        newTrie.rootNode = (TrieNode) rootNode.clone(newTrie.nodes);
        return newTrie;
    }

    public Trie() {
        rootNode = new TrieNode(nodes);
    }

    /**
     * Adds a vector of token templates. The token templates are assumed
     * to be the same order they should be recognized.
     * @param listener  The listeners notify will be notified as soon as
     *                  this vector of templates is matched in the token stream. Of course
     *                  it is not allowed to be null
     */
    public void addTemplateVector(Vector templates, TrieListener listener) {
        TrieNode next = rootNode;
        for (int i=0;i<templates.size();i++) {
            TokenTemplate temp = (TokenTemplate) templates.elementAt(i);
            TrieNode[] temps = next.addTokenTemplate(temp,nodes);
            next = temps[0];
        }
        if (next != null) next.addTrieListener(listener);
    }

    /**
     * allows fuzzy text recognition (a certain number of "levenstein" operations)
     */
    public void allowLevenstein() {
        allowInsertion();
        allowWrong();
        allowOmit();
    }


    /**
     * A part of the preparation for "fuzzy" text recognition. This part
     * insert transitions to allow (wrongly) inserted characters in strings
     * (actually it allows wrongly inserted tokens - not just characters)
     */
    public void allowInsertion () {
        Iterator it = nodes.iterator();
        while (it.hasNext()) {
            TrieNode current = (TrieNode) it.next();
            current.addTokenTemplate( new AnyTokenTemplate(),current);
        }
    }

    /**
     * A part of the preparation for "fuzzy" text recognition. This part
     * inserts transitions to allow wrong characters in strings (actually
     * it allows wrong tokens, not just wrong characters)
     */
    public void allowWrong () {
        Iterator it = nodes.iterator();
        while (it.hasNext()) {
            TrieNode current = (TrieNode) it.next();
            current.allowWrongToken();
        }
    }

    /**
     * A part of the preparation for fuzzy text recognition. This part
     * inserts transitions to allow omitting of characters in strings (actually
     * it allows omitting of tokens, not just characters)
     */
    public void allowOmit () {
        Iterator it = nodes.iterator();
        while (it.hasNext()) {
            TrieNode current = (TrieNode) it.next();
            current.allowOmit();
        }
    }

    /**
     * Parses a given TokenSet.
     */
    public void parse(TokenSet tokenSet, Vector pointer, Vector pointerTemp, String id) {
        pointer.addElement(new Pointer(rootNode, tokenSet.getIndex()));
        for (int i=0;i<pointer.size();i++) {
            TrieNode currentNode = ((Pointer)pointer.elementAt(i)).getTrieNode();
            currentNode.parse(tokenSet,((Pointer)pointer.elementAt(i)),pointerTemp, id);
        }
    }

    /**
     * Parses the entire TokenTraversStore
     */
    public void parseTokenStore(TokenTraverseStore store, String id) {
        Vector pointer = new Vector();
        Vector pointerTemp = new Vector();
        while(store.next()) {
            parse(store, pointer, pointerTemp, id);
            Vector shortTerm = pointer;
            pointer = pointerTemp;
            pointerTemp = shortTerm;
            pointerTemp.removeAllElements();
        }
    }

}
