package de.fzi.wim.trie.extractor;

import java.util.Vector;
import java.util.Comparator;
import java.util.Collections;


/**
 * A class specifically designed to supply the program with tokens while a string
 * is parsed. It's pretty fast at this specific task.<br>
 * <ul>
 *  <li>Beginning Tokens are the tokens starting at this position.</li>
 *  <li>Current Tokens are the tokens that started earlier and end later than
 *  the current position. Current Tokens to neither include the beginning
 *  tokens nor the ending tokens (although it could be argued that these are still
 *  in effect, it would be redundant information).</li>
 *  <li>Ending tokens are the tokens ending in this postion.</li>
 * </ul>
 * Tokens of length 0, i.e. begin = end, will never be current or ending tokens;
 * just beginning Tokens. Tokens of length 1, i.e. end = begin+1, will never be
 * current tokens, just beginning end ending.
 *
 * @author <a href="zach@fzi.de">Valentin Zacharias</a>
 */
public class TokenTraverseStore implements TokenSet {

    private Vector tokens;
    private Vector notBegun;
    private Vector beginning = new Vector(20);
    private Vector ending = new Vector(20);
    private Vector current = new Vector(40);
    private int index = -1;


    /**
     * Initializes the token store. Takes a while
     */
    public TokenTraverseStore(Vector tokens) {
        this.tokens = tokens;
        Collections.sort(this.tokens, new Comparator() {
            public int compare (Object o1, Object o2) {
                Token t1 = (Token) o1;
                Token t2 = (Token) o2;
                return t1.begin - t2.begin;
            }
        });
        reset();
    }

    /**
     * Resets the token store, it start's again with the first token (expensive
     * operation).
     */
    public void reset() {
        notBegun = (Vector) tokens.clone();
        beginning.removeAllElements();
        ending.removeAllElements();
        current.removeAllElements();
        index = -1;
    }

    public Vector getBeginningTokens() {
        return beginning;
    }
    public Vector getEndingTokens() {
        return ending;
    }
    public Vector getCurrentTokens() {
        return current;
    }
    public int getIndex() {
        return index;
    }

    /**
     * Increases the current Index by one.
     * @return      true if at lease one of the getXXXTokens methods will return at
     *              least one token OR will return at least one token in the future without a
     *              call to reset. It returns only false, if the end of all the tokens
     *              has been reached.
    */
    public boolean next() {
        index ++;

        //remove all tokens who endet
        ending.removeAllElements();

        //remove ending tokens from current array
        for (int i=0;i<current.size();i++) {
            Token currentToken = (Token) current.elementAt(i);
            if (currentToken.end == index) {
                ending.add(currentToken);
                current.set(i,null);
            }
        }
        int in = 0;
        while (in<current.size()) {
            if (current.elementAt(in) == null) {
                current.removeElementAt(in);
            }
            else {
                in++;
            }
        }

        //take care of beginning array
        // end == index -> ending,  end > index -> current
        for (int i=0;i<beginning.size();i++) {
            Token currentToken = (Token) beginning.elementAt(i);
            if (currentToken.end == index) ending.addElement(currentToken);
            else if (currentToken.end > index) current.addElement(currentToken);
        }
        beginning.removeAllElements();

        //get new beginning elements
        while(!notBegun.isEmpty()) {
            Token currentToken = (Token) notBegun.elementAt(0);
            if (currentToken.begin == index) {
                notBegun.removeElementAt(0);
                beginning.addElement(currentToken);
            }
            else {
                break;
            }
        }
        return !((notBegun.isEmpty()) && (current.isEmpty()) && (beginning.isEmpty()) && (ending.isEmpty()));

    }

}
