package de.fzi.wim.trie.deterministic;

import java.util.NoSuchElementException;
import java.util.Vector;
import java.util.HashMap;
import java.util.Set;
import java.util.Iterator;
import java.util.HashSet;
import java.util.Collection;

import de.fzi.wim.trie.*;

/**
 * An implementation of the LexiconStructure interface. For any information on the
 * methods, please refer to the Trie interface.
 *
 * @see Trie
 * @author <a href="mailto:zach@fzi.de">Valentin Zacharias</a>
 * @version 19.12.2001
 */
public class LexiconStructureImpl implements LexiconStructure {

    /** The private class representing a node. */
    private class Node {
        private int length=0; //the distance to the root node.
        private Vector references=new Vector(3); //stores the reference objects for this node
        private String entry=null; //the entry represented through this node
        private HashMap edges = new HashMap(15); //stores the edges with the corresponding character.

        Node(int depth) {
            length = depth;
        }

        Node travers(Character character) {
           return (Node) edges.get(character);
        }

        void addNoteChar(Character c, Node node) {
            edges.put(c,node);
        }

        boolean foundReference() {
            return (entry != null);
        }

        Node removeNoteChar(char character) {
            return (Node) edges.remove(new Character(character));
        }

        Collection getChilds() {
            return edges.values();
        }

        //Returns true if there's still some entity done this way ....
        boolean deleteDeserted() {
            Iterator it = edges.keySet().iterator();
            boolean populated = false;
            if (entry != null) populated = true;
            while (it.hasNext()) {
                Object key  = it.next();
                Node reference = (Node) edges.get(key);
                boolean temp = reference.deleteDeserted();
                if (temp == false) {
                    edges.remove(key);
                    it = edges.keySet().iterator();
                }
                populated = temp | populated;
            }
            return populated;
        }



        void addReferenceEntry(String entry, Object reference) {
            if (this.entry != null) {
            	if (references.indexOf(reference) == -1) {
                	references.addElement(reference);
             	}
            }
            else {
                this.entry = entry;
                references.addElement(reference);
            }
        }

        int removeEntryReference (String entry, Object ref) {
            if (this.entry.equals(entry)) {
                int i = references.indexOf(ref);
                if (i == -1) return 0;
                else {
                    references.removeElementAt(i);
                    if (references.size() == 0) {
                        this.entry = null;
                        return 1;
                    }
                    else {
                        return 2;
                    }
                }
            }
            else {
                return 0;
            }
        }

        boolean changeReference (Object oldRef, Object newRef) {
            int i = references.indexOf(oldRef);
            if (i!= -1) {
                references.setElementAt(newRef, i);
                return true;
            }
            return false;
        }

        boolean removeReference(Object ref) {
            int i= references.indexOf(ref);
            if (i!=-1) {
                references.removeElementAt(i);
                if (references.size() == 0) entry = null;
                return true;
            }
            else {
                return false;
            }
        }

        boolean removeEntry (String entry) {
            if (this.entry.equals(entry)) {
                this.entry = null;
                references.removeAllElements();
                return true;
            }
            else {
                return false;
            }
        }
    }


	/** The local class implementing the TrieChecker interface */
    public class TrieCheckerImpl implements LexiconChecker {
           private LexiconStructureImpl myTrie = null;
           private Vector pointer = new Vector(20);
           private Vector pointerTemp = new Vector(20);
           private Vector listener = new Vector(5);
           private Vector entryLocations = new Vector(20);
           private int entryLocationsIndex = 0;
           private int textIndex = 0;

        private TrieCheckerImpl (LexiconStructureImpl myTrie) {
               this.myTrie = myTrie;
        }

	    /**
         * Adds an event Listener that is notified anytime a lexicon
         * entry is found.
         */
	    public void addLexiconCheckerListener(LexiconCheckerListener tcl){
               listener.add(tcl);
        }

		public void removeLexiconCheckerListener(LexiconCheckerListener tcl){
               listener.remove(tcl);
        }

	    /**
         * Use this method to travers the text.
         *
	     * @return      Returns true if at least one reference has been found. Use getEntryLocation and hasMoreEntryLocations to find out more.
         */
	    public boolean nextChar(char c){
            entryLocations.clear(); entryLocationsIndex = 0;
            boolean foundOne = false;
            Character currentCharacter;
            if (Character.isWhitespace(c)) currentCharacter = new Character(' ');
            else if ((c =='.') || (c==',') || (c==';') || (c=='(') || (c==')') ) currentCharacter = new Character(' ');
            else currentCharacter = new Character(c);
           	pointer.add(myTrie.root);
            textIndex++;
           	for (int i= pointer.size()-1;i>=0;i--) {
            	Node current = (Node)pointer.elementAt(i);
                Node newNode = current.travers(currentCharacter);
                if (newNode != null) {
                    if (newNode.foundReference()) {
                        foundEntryLocation(newNode);
                        foundOne = true;
                    }
                    pointerTemp.add(newNode);
                }
           }
           pointer.clear();
           Vector swap = pointer;
           pointer = pointerTemp;
           pointerTemp = swap;
           return foundOne;
        }

        private void foundEntryLocation(Node node) {
            for (int i=0;i<node.references.size();i++) {
                EntryLocationImpl location = new EntryLocationImpl(node.references.elementAt(i), textIndex - node.length, textIndex);
                entryLocations.add(location);
                for (int j=0; j<listener.size();j++) {
                    LexiconCheckerListener cur = (LexiconCheckerListener)listener.elementAt(j);
                    cur.EntryLocationFound(location);
                }
            }
        }


	    /**
         * Returns a EntryLocation found by the last call to nextChar or throws
	     * an exception if no entry was found. Please note that more than one
	     * Lexicon entry can be found at any given time (if the lexicon contains
	     * "bubab" and "bab" the last char of the input text "bubab" will result
	     * in two created EntryLocations.
	     */
	    public EntryLocation getEntryLocation() throws NoSuchElementException {
            if (entryLocationsIndex < entryLocations.size()) {
                 entryLocationsIndex ++;
                 return (EntryLocation) entryLocations.elementAt(entryLocationsIndex-1);
            }
            else {
                throw new NoSuchElementException();
            }
        }

	    /**
         * Returns true if and only if at least one more object is returned by getReference.
         */
	    public boolean hasMoreEntryLocations(){
	        return (entryLocationsIndex < entryLocations.size());
	    }
	}

	private Vector listerner;
	private Node root = new Node(0);

	public LexiconStructureImpl() {
	}

    /**
     * Adds a entry from the lexicon to the trie. If the "entry" is already
     * contained in the trie, the reference is added to to this entry (any entry
     * may have more than one refernced object).
     *
     * @param entry The string from the object
     * @param reference The refernced object.
     */
    public void addEntry(String entry, Object reference) {
       addEntry(transformToCharacter(entry), reference,root,0,entry);
    }

    private Character[] transformToCharacter(String str) {
        char[] entryChars = str.toLowerCase().toCharArray();
        Character[] entryCharObjects = new Character[entryChars.length];
        for (int i=(entryChars.length -1);i>=0;i--) {
            if (Character.isWhitespace(entryChars[i])) entryCharObjects[i] = new Character(' ');
            else entryCharObjects[i] = new Character(entryChars[i]);
        }
        return entryCharObjects;
    }

    private void addEntry(Character[] entryC, Object reference, Node current, int depth, String entryS) {
        if (depth == entryC.length) {
            current.addReferenceEntry(entryS,reference);
        }
        else {
            Node temp = current.travers(entryC[depth]);
            if (temp == null) {
                temp = new Node(depth+1);
                current.addNoteChar(entryC[depth],temp);
            }
            addEntry(entryC,reference,temp,depth+1,entryS);
        }
    }

    private Node traversEntry(Character[] entry, int i, Node current) {
        if (current == null) return null;
        else if (i==entry.length) return current;
        else {
	        Node temp = current.travers(entry[i]);
            return traversEntry(entry,i+1,temp);
        }
    }


    /**
     * Removes an en entry from the lexicon, any references to this entry
     * is deleted from the Trie.
     *
     * @return          true, if the entry was contained in the Trie, false otherwise.
     */
    public boolean removeEntry(String entry) {
        Node temp = traversEntry(transformToCharacter(entry),0,root);
        if (temp == null) return false;
        else {
            return temp.removeEntry(entry);
        }
    }

    /**
     * Removes a entry-reference relation from the Trie. If there are other
     * references for this entry, the entry will remain in the Trie, otherwise
     * it's deleted. The Object.equals method is used to compare reference
     * objects.
     *
     * @return          0 if the entry-reference relation was not contained in the Trie,
     *                  1 if it was and the entry was deleted,
     *                  2 if it was contained but the entry is still in the Trie (because it has other references).
     */
    public int removeEntryReference(String entry, Object reference){
    	Node temp = traversEntry(transformToCharacter(entry),0,root);
        if (temp == null) return 0;
        else {
            return temp.removeEntryReference(entry,reference);
        }
    }

    /**
     * Removes all occurrences of objects equal to the supplied "reference"
     * object from the trie. This may result in entry Strings beeing deleted.
     * Takes a while to complete!
     *
     * @return          The number of objects equal to the reference object found.
     */
    public int removeReference(Object reference){
        Set workingSet = new HashSet();
        workingSet.addAll(root.getChilds());
        int referenceCounter = 0;
        Object[] currentSet;
        if (root.removeReference(reference)) referenceCounter++;
        while (!workingSet.isEmpty()) {
            currentSet = workingSet.toArray();
            workingSet.clear();
            for (int i=(currentSet.length-1);i>=0;i--) {
                Node current = (Node) currentSet[i];
                if (current.removeReference(reference)) referenceCounter++;
				workingSet.addAll(current.getChilds());
            }
        }
        return referenceCounter;
    }

    /**
    * Replaces all ocurrences of objects equal to the supplied object "oldR"
    * and replaces them with "newR". Depending on the implementation,
    * this method may take a while.
    * @return           the number of replacements made.
    */
    public int replaceReference(Object oldR, Object newR){
        Set workingSet = new HashSet();
        workingSet.addAll(root.getChilds());
        int referenceCounter = 0;
        Object[] currentSet;
        if (root.changeReference(oldR,newR)) referenceCounter++;
        while (!workingSet.isEmpty()) {
            currentSet = workingSet.toArray();
            workingSet.clear();
            for (int i=(currentSet.length-1);i>=0;i--) {
                Node current = (Node) currentSet[i];
                if (current.changeReference(oldR,newR)) referenceCounter++;
				workingSet.addAll(current.getChilds());
            }
        }
        return referenceCounter;
    }

    public LexiconChecker getLexiconChecker(){
    	return new TrieCheckerImpl(this);
    }

    public void optimize() {
        root.deleteDeserted();
    }
}
