package de.fzi.wim.similarity;

import java.io.File;
import java.io.IOException;
import java.io.FileOutputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.FileInputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.util.Set;
import java.util.Iterator;
import java.util.HashMap;
import java.util.Vector;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.KAONVocabularyAdaptor;

/**
* A class used to store and calculate the similarity matrix
* of a set of instances.
* @author <a href="mailto:zach@fzi.de">Valentin Zacharias</a>
*/
public class SimilarityMatrix {

    private Instance[] instances;
    private double[][] simMatrix;
    private Similarity sim;
    private HashMap instanceHash = new HashMap();


    public SimilarityMatrix(Similarity sim, Set instances) {
        this.sim = sim;
        this.instances = new Instance[instances.size()];
        Iterator it = instances.iterator();
        int i=0;
        while (it.hasNext()) {
            this.instances[i] = (Instance) it.next();
            instanceHash.put(this.instances[i],new Integer(i));
            i++;
        }
        simMatrix = new double[instances.size()][instances.size()];
    }

    /**
    * Instantiates a SimilarityMatrix from a previous saved similarity matrix object.
    */
    public SimilarityMatrix(File file, OIModel oimodel) throws IOException, KAONException, ClassNotFoundException {
        ObjectInputStream in = new ObjectInputStream(new BufferedInputStream(new FileInputStream(file)));
        String[] uris = (String[]) in.readObject();
        instances = new Instance[uris.length];
        for (int i=0;i<uris.length;i++) {
            instances[i] = oimodel.getInstance(uris[i]);
            instanceHash.put(instances[i],new Integer(i));
        }
        simMatrix = (double[][]) in.readObject();
    }


    /**
    * Saves a SimilarityMatrix to a file.
    */
    public void save(File file) throws IOException, KAONException {
        ObjectOutputStream out = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(file)));
        String[] uris = new String[instances.length];
        for (int i=0;i<uris.length;i++) uris[i] = instances[i].getURI();
        out.writeObject(uris);
        out.writeObject(simMatrix);
        out.close();
    }

    public double getSimilarity(Instance inst1, Instance inst2) {
        int inst1Int = ((Integer)instanceHash.get(inst1)).intValue();
        int inst2Int = ((Integer)instanceHash.get(inst2)).intValue();
        return simMatrix[inst1Int][inst2Int];
    }


    /**
    * Allows to set the objet used to compute similarites. Please note that this
    * is necessary before you can call "calculate" on a SimilarityMatrix that
    * was loadet from disk.
    */
    public void setSimilarity(Similarity sim) {
        this.sim = sim;
    }

    public void calculate() throws KAONException{
        for (int x=0;x<instances.length;x++) {
            Instance inst1 = instances[x];
            simMatrix[x][x] = 1.0;
            for (int y=x+1;y<instances.length;y++) {
                Instance inst2 = instances[y];
                simMatrix[x][y] = sim.calculateSimilarity(inst1,inst2);
            }
        }
        for (int x=0;x<instances.length;x++) {
            for (int y=0; y<x;y++) {
                simMatrix[x][y] = simMatrix[y][x];
            }
        }
    }

    public double[][] getSimMatrix() {
        return simMatrix;
    }

    public Instance[] getInstances() {
        return instances;
    }

    public Vector getMostSimilar(Instance inst, double excludeThreshold, double includeThreshold, int normalNumber, int maxNumber) {
        double[] similarities = new double[maxNumber];
        Instance[] instances = new Instance[maxNumber];

        int current = ((Integer) instanceHash.get(inst)).intValue();
        for (int i=0;i<instances.length;i++) {
            if (current != i) {
                double sim = simMatrix[current][i];
                addComparison(instances[current],sim,similarities,instances);
            }
        }

        Vector toReturn = new Vector(maxNumber);
        for (int i=0;i<similarities.length;i++) {
            if ((i<normalNumber) && (similarities[i] > excludeThreshold)) {
                toReturn.addElement(instances[i]);
            }
            else if ((i<maxNumber) && (similarities[i] > includeThreshold)) {
                toReturn.addElement(instances[i]);
            }
            else {
                break;
            }
        }
        return toReturn;
    }




    private static void addComparison(Instance current, double sim, double[] similarities, Instance[] instances) {
        if (sim > similarities[similarities.length-1]) {
            for (int i=0;i<similarities.length;i++) {
                if (sim > similarities[i]) {
                    moveBack(similarities,instances,i);
                    similarities[i] = sim;
                    instances[i] = current;
                    break;
                }
            }
        }
    }

    /**
    * Moves all elements of the two arrays after index one position towards the end
    * of the array. The last element is discarded. This way the position "index" becomes
    * empty.
    */
    private static void moveBack(double[] similarities, Instance[] instances, int index) {
        for (int i=similarities.length-1;i>index;i--) {
            similarities[i] = similarities[i-1];
            instances[i] = instances[i-1];
        }
    }







    /**
    * Returns a HashMap where associating instances with their index in the instances array (
    * and thereby also their index in the simMatrix array). The index is stored as Integer object.
    */
    public HashMap getInstanceHash() {
        return instanceHash;
    }

    public void print() throws KAONException{
        for (int x=0;x<instances.length;x++) {
            String label1 = instances[x].getLexicalAttribute(KAONVocabularyAdaptor.INSTANCE.getKAONLabel(),KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en"),KAONVocabularyAdaptor.INSTANCE.getValue());
            for (int y=x+1;y<instances.length;y++) {
                String label2 = instances[y].getLexicalAttribute(KAONVocabularyAdaptor.INSTANCE.getKAONLabel(),KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en"),KAONVocabularyAdaptor.INSTANCE.getValue());
                System.out.println(label1 + " --- " + label2 + " === "+ simMatrix[x][y]);
            }
        }
    }


}
