package de.fzi.wim.similarity;

import java.util.HashMap;
import java.util.Vector;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.KAONException;
import edu.unika.aifb.kaon.api.vocabulary.VocabularyAdaptor;
import edu.unika.aifb.kaon.api.vocabulary.KAONVocabularyAdaptor;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
* Combines the different similarity measures and offers
* convenient methods to calculate similarities.
* @author <a href="zach@fzi.de">Valentin Zacharias</a>
*/
public class SimilarityImpl extends Similarity {
    protected static final VocabularyAdaptor vocabularyAdaptor = KAONVocabularyAdaptor.INSTANCE;

    private HashMap[] caches;
    private SemanticSimilarity semSim;
    private PropertySimilarity propSim;
    private OIModel oimodel;

    public SimilarityImpl (OIModel oimodel) {
        this.oimodel = oimodel;
        semSim = new SemanticSimilarity();
        propSim = new PropertySimilarity(this);
        caches = new HashMap[maxDepth+1];
        for (int i=0;i<caches.length;i++) caches[i] = new HashMap();
    }

    /**
    * Returns the most similar instances.
    * @param excludeThreshold Instances with a similarity less then "excludeThreshold" are never returned.
    * @param includeThreshold Instance with a similarity higher than "includeThreshold" are always included (but
    *   never more than "maxNumber".
    * @param normalNumber If "includeThreshold" and "excludeThreshold" do not determine the number of returned instances,
    *   the number of returned instance is "normalNumber".
    * @param maxNumber The method is guaranteed to never return more than maxNumber instances.
    */
    public Vector getMostSimilar(Instance inst, double excludeThreshold, double includeThreshold, int normalNumber, int maxNumber) throws KAONException {
        double[] similarities = new double[maxNumber];
        Instance[] instances = new Instance[maxNumber];

        Iterator it = oimodel.getRootConcept().getAllInstances().iterator();
        while (it.hasNext()) {
            Instance current = (Instance) it.next();
            if (!current.equals(inst)) {
                double sim = calculateSimilarity(inst,current);
                addComparison(current,sim,similarities,instances);
            }
        }

        Vector toReturn = new Vector(maxNumber);
        for (int i=0;i<similarities.length;i++) {
            if ((i<normalNumber) && (similarities[i] > excludeThreshold)) {
                toReturn.addElement(instances[i]);
            }
            else if ((i<maxNumber) && (similarities[i] > includeThreshold)) {
                toReturn.addElement(instances[i]);
            }
            else {
                break;
            }
        }
        return toReturn;
    }




    private static void addComparison(Instance current, double sim, double[] similarities, Instance[] instances) {
        if (sim > similarities[similarities.length-1]) {
            for (int i=0;i<similarities.length;i++) {
                if (sim > similarities[i]) {
                    moveBack(similarities,instances,i);
                    similarities[i] = sim;
                    instances[i] = current;
                    break;
                }
            }
        }
    }

    /**
    * Moves all elements of the two arrays after index one position towards the end
    * of the array. The last element is discarded. This way the position "index" becomes
    * empty.
    */
    private static void moveBack(double[] similarities, Instance[] instances, int index) {
        for (int i=similarities.length-1;i>index;i--) {
            similarities[i] = similarities[i-1];
            instances[i] = instances[i-1];
        }
    }



    public double calculateSimilarity(Instance inst1, Instance inst2) throws KAONException {
        return calculateSimilarity(new InstanceTupel(inst1,inst2),0);
    }

    protected double calculateSimilarity(InstanceTupel current, int depth) throws KAONException {
        if (current.inst1.equals(current.inst2)) return 1.0;
        else {
            Double result = (Double) caches[depth].get(current);
            if (result == null) {
                double divider = 0;
                double sum = 0;
                sum += (semSim.getSimilarity(current) * semSimWeight);
                divider +=semSimWeight;
                if (depth < maxDepth) {
                    sum += (propSim.getSimilarity(current,depth));
                    divider += propSimWeight;
                }
                result = new Double (sum/divider);
                caches[depth].put(current,result);
            }
            return result.doubleValue();
        }
    }


}
