package de.fzi.wim.similarity;

import java.util.Set;
import java.util.Iterator;
import java.util.HashSet;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
* The class to compute the relational similarity between
* two instances.
* @author <a href="mailto:zach@fzi.de">Valentin Zacharias</a>
*/
public class PropertySimilarity {

    private SimilarityImpl sim;


    public PropertySimilarity(SimilarityImpl sim) {
        this.sim = sim;
    }

    public double getSimilarity(InstanceTupel instt, int depth) throws KAONException{
        Set rangeProperties  = getRangeProperties(instt.inst1, instt.inst2);
        Set domainProperties = getDomainProperties(instt.inst1, instt.inst2);
        double sum = 0;
        double counter = 0;
        Iterator it = rangeProperties.iterator();
        while (it.hasNext()) {
            Property current = (Property)it.next();
            double currentResult  = calculateSimilarityToProperty(instt,current,depth);
            if (currentResult != Double.NaN) {
                sum += currentResult;
                counter++;
            }
        }
        it = domainProperties.iterator();
        while (it.hasNext()) {
            Property current = (Property) it.next();
            double currentResult = calculateSimilarityFromProperty(instt,current,depth);
            if (currentResult != Double.NaN) {
                sum += currentResult;
                counter++;
            }
        }
        double result=0.0;
        if ((rangeProperties.size() + domainProperties.size()) !=  0) {
            result = sum / (rangeProperties.size() + domainProperties.size());
        }
        return result;
    }

    private double calculateSimilarityToProperty(InstanceTupel instt, Property prop, int depth) throws KAONException {
        Set asocRelInst1 = instt.inst1.getToPropertyInstances();
        Iterator it = asocRelInst1.iterator();
        Set asoc1 = new HashSet();
        while (it.hasNext()) {
            PropertyInstance current = (PropertyInstance) it.next();
            asoc1.add(current.getSourceInstance());
        }
        Set asocRelInst2 = instt.inst2.getToPropertyInstances();
        it = asocRelInst2.iterator();
        Set asoc2 = new HashSet();
        while (it.hasNext()) {
            PropertyInstance current = (PropertyInstance) it.next();
            asoc2.add(current.getSourceInstance());
        }
        return calculateSimilarityForProperty(asoc1,asoc2,depth);
    }

    private double calculateSimilarityFromProperty(InstanceTupel instt, Property prop, int depth) throws KAONException {
        Set asoc1 = instt.inst1.getFromPropertyValues(prop);
        Set asoc2 = instt.inst2.getFromPropertyValues(prop);
        return calculateSimilarityForProperty(asoc1,asoc2,depth);
    }

    private double calculateSimilarityForProperty(Set asoc1, Set asoc2, int depth) throws KAONException{
        if (asoc2.size() > asoc1.size()) {
            Set temp = asoc1;
            asoc1 = asoc2;
            asoc2 = temp;
        }
        double sum = 0;
        Iterator it1 = asoc1.iterator();
        while (it1.hasNext()) {
            Instance currentInst1 = (Instance) it1.next();
            double maxSim = 0;
            Iterator it2 = asoc2.iterator();
            while(it2.hasNext()) {
                Instance currentInst2 = (Instance) it2.next();
                maxSim = Math.max(maxSim,sim.calculateSimilarity(new InstanceTupel(currentInst1,currentInst2), depth+1));
            }
            sum+=maxSim;
        }
        if (asoc1.size() == 0) return 0;
        else return sum / ((double) asoc1.size());
    }


    private Set getDomainProperties(Instance inst1, Instance inst2) throws KAONException {
        Set range1 = getDomainProperties(inst1);
        Set range2 = getDomainProperties(inst2);
        range1.retainAll(range2);
        return range1;
    }

    private Set getDomainProperties(Instance inst) throws KAONException{
        Set parents = inst.getParentConcepts();
        Set result = new HashSet();
        Iterator it = parents.iterator();
        while (it.hasNext()) {
            Concept current = (Concept) it.next();
            result.addAll(current.getPropertiesFromConcept());
        }
        it = result.iterator();
        HashSet toRemove = new HashSet();
        while(it.hasNext()) {
            Property current = (Property) it.next();
            if (current.isAttribute()) {
                toRemove.add(current);
            }
        }
        result.removeAll(toRemove);
        return result;
    }

    private Set getRangeProperties(Instance inst1, Instance inst2) throws KAONException {
        Set range1 = getRangeProperties(inst1);
        Set range2 = getRangeProperties(inst2);
        range1.retainAll(range2);
        return range1;
    }

    private Set getRangeProperties(Instance inst) throws KAONException{
        Set parents = inst.getParentConcepts();
        Set result = new HashSet();
        Iterator it = parents.iterator();
        while (it.hasNext()) {
            Concept current = (Concept) it.next();
            result.addAll(current.getPropertiesToConcept());
        }
        return result;
    }

}
