package de.fzi.wim.registry.gui.ui;

import java.util.List;
import java.util.Collections;
import java.text.Collator;
import java.util.Comparator;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import javax.swing.table.AbstractTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.registry.api.*;

/**
 * A table model containing Term objects from the registry.
 */
public class TermsTableModel extends AbstractTableModel {
    /** The localization manager. */
    protected LocalizationManager m_localizationManager;
    /** The list of elements. */
    protected List m_elements;

    /**
     * Creates an instance of this class.
     *
     * @param localizationManager               the localization manager
     */
    public TermsTableModel(LocalizationManager localizationManager) {
        m_localizationManager=localizationManager;
        m_elements=new ArrayList();
    }
    /**
     * Refreshes the model.
     *
     * @param elements                          elements the model should display
     * @throws KAONException                    thrown in case of error
     */
    public void setElements(Collection elements) throws KAONException {
        m_elements.clear();
        try {
            Iterator iterator=elements.iterator();
            while (iterator.hasNext()) {
                Term curterm=(Term)iterator.next();
                m_elements.add(new TableRow(curterm));
            }
        }
        finally {
            fireTableDataChanged();
        }
    }
    /**
     * Returns the class of the given column.
     *
     * @param columnIndex                       the index of the column
     * @return                                  the class of supplied column
     */
    public Class getColumnClass(int columnIndex) {
        switch (columnIndex) {
        case 0:
        default:
            return String.class;
        }
    }
    /**
     * Returns the number of columns.
     *
     * @return                                  the number of columns
     */
    public int getColumnCount() {
        return 2;
    }
    /**
     * Returns the name of given column.
     *
     * @param columnIndex                       the index of the column
     * @return                                  the name of the given column
     */
    public String getColumnName(int columnIndex) {
        return m_localizationManager.getPhrase("registry.termTableModel."+columnIndex);
    }
    /**
     * Returns the number of rows.
     *
     * @return                                  the number of rows
     */
    public int getRowCount() {
        return m_elements.size();
    }

    /**
     * Returns a value at given coordinates.
     *
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     * @return                                  the value
     */
    public Object getValueAt(int rowIndex,int columnIndex) {
        TableRow tableRow=(TableRow)m_elements.get(rowIndex);
		return tableRow.getColumnText(columnIndex);
    }
    /**
     * Checks whether the cell is editable.
     *
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     * @return                                  <code>true</code> if the cell is editable
     */
    public boolean isCellEditable(int rowIndex,int columnIndex) {
        return false;
    }
    /**
     * Sets the value at given coordinates.
     *
     * @param aValue                            the value
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     */
    public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
    }
    /**
     * Sorts the model.
     *
     * @param columnIndex                       the index of the column that is sorted
     * @param ascending                         if <code>true</code>, the column is sorted in the ascending order
     */
    public void sortModel(int columnIndex,boolean ascending) {
		Comparator comparator=new TableRowComparator(columnIndex,ascending);

		Collections.sort(m_elements,comparator);

		fireTableDataChanged();
    }
    /**
     * Returns the Term object at the given coordinates.
     *
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     * @return                                  the OIModelInfo object
     */
    public Term getTermAt(int rowIndex,int columnIndex) {
        TableRow tableRow=(TableRow)m_elements.get(rowIndex);
        switch (columnIndex) {
        case 0:
            return tableRow.m_term;
        default:
            return null;
        }
    }

	/**
	 * Class to compare two table row elements in the specified column index.
	 */
    protected static class TableRowComparator implements Comparator {
		protected int m_columnindex;
		protected boolean m_ascending;

        public TableRowComparator( int columnindex,boolean ascending ) {
			m_columnindex = columnindex;
			m_ascending=ascending;
		}

		 public int compare(Object o1,Object o2) {
			// just convert both objects to strings
			String s1=((TableRow)o1).getColumnText(m_columnindex);
			String s2=((TableRow)o2).getColumnText(m_columnindex);

			// and use the collator with the current locale to compare them
			if ( m_ascending )
				return Collator.getInstance().compare(s1,s2);
			else
				return -Collator.getInstance().compare(s1,s2);
        }
    }

    /**
     * The element of the table model.
     */
    protected class TableRow {
        /** The term. */
        protected Term m_term;
        /** The cached synonyms of the term. */
        protected String m_synonyms;
        /** The cached term definition. */
        protected String m_termDefinition;

		/**
		 * Create a new instance of this class
		 *
		 * @param term                      the term
         * @throws KAONException            thrown if there is an error
		 */
        public TableRow(Term term) throws KAONException {
            m_term=term;
            m_termDefinition=m_term.getTermDefinition();
            StringBuffer buffer=new StringBuffer();
            Iterator iterator=m_term.getTermSynonyms(KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en")).iterator();
            while (iterator.hasNext()) {
                String synonym=(String)iterator.next();
                buffer.append(synonym);
                if (iterator.hasNext())
                    buffer.append(", ");
            }
            m_synonyms=buffer.toString();
        }

		/**
		 * Returns the text of the specified column index.
		 *
		 * @param columnindex               zero based column index
		 * @return                          the string that can be displayed on the screen
		 */
		public String getColumnText( int columnindex ) {
            switch (columnindex) {
                case 0:
                    return m_synonyms;
                case 1:
                default:
                    return m_termDefinition;
            }
		}
    }
}
