package de.fzi.wim.registry.gui.ui;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collection;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.awt.Cursor;
import java.awt.Insets;
import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.BorderFactory;
import javax.swing.SwingUtilities;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JSplitPane;
import javax.swing.JScrollPane;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.table.JTableHeader;
import javax.swing.ListSelectionModel;
import javax.swing.event.TableModelListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.ListSelectionEvent;

import edu.unika.aifb.kaon.api.*;

import de.fzi.wim.guibase.appdriver.*;
import de.fzi.wim.guibase.configuration.*;
import de.fzi.wim.guibase.tables.*;
import de.fzi.wim.guibase.viewableselector.*;

import de.fzi.wim.oimodeler.OIModelerModule;
import de.fzi.wim.oimodeler.ui.OIModelerViewable;
import edu.unika.aifb.kaon.api.oimodel.OIModel;

import de.fzi.wim.registry.api.*;
import de.fzi.wim.registry.gui.*;

/**
 * Displays information in a registry.
 */
public class RegistryViewable extends AbstractViewable {
    /** The registry. */
    protected Registry m_registry;
    /** Set to <code>true</code> if the registry is being processed. */
    protected boolean m_registryProcessed;
    /** The table model displaying the results of the search. */
    protected OIModelInfoTableModel m_oimodelInfoTableModel;
    /** The search parameters pane. */
    protected OIModelInfoPane m_searchParametersPane;
    /** The pane for registering new ontologies. */
    protected OIModelInfoPane m_registerOIModelPane;
    /** The table displaying the results of the search. */
    protected JTable m_oimodelInfoTable;
    /** this variable holds the standard row height of the table */
    protected int m_originalrowheight;
    /** this variable holds the maximum of lines currently in the table*/
    protected int m_maxlines=1;
    /** The selector for the OI-model on the new OIModel pane. */
    protected ViewableSelector m_oimodelChooser;
    /** The left tab pane. */
    protected JTabbedPane m_leftTabbedPane;

    /**
     * Creates an intance of this class.
     *
     * @param registryModule                    the module for the registry
     * @param registry                          the registry to be shown in this pane
     */
    public RegistryViewable(RegistryModule registryModule,Registry registry) {
        super(registryModule);
        m_registry=registry;

        // create the search pane
        m_searchParametersPane=new OIModelInfoPane(this,OIModelInfoPane.PANE_SEARCH);
        JButton searchbutton=registryModule.getAppDriver().getLocalizationManager().getButton("registry.searchbutton");
        searchbutton.setIcon(registryModule.getAppDriver().getLocalizationManager().getImageIcon("registry.magnifier"));
        searchbutton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                doSearch();
            }
        });

        // create the new OIModel pane
        m_registerOIModelPane=new OIModelInfoPane(this,OIModelInfoPane.PANE_CREATE);
        JButton registerNewOIModelButton=registryModule.getAppDriver().getLocalizationManager().getButton("registry.registernewoimodelbutton");
        registerNewOIModelButton.setIcon(registryModule.getAppDriver().getLocalizationManager().getImageIcon("registry.asterisk"));
        registerNewOIModelButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                registerNewOIModel();
            }
        });

        // create the manage elements pane
        ManageElementsPane manageElementsPane=new ManageElementsPane(this);
        manageElementsPane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

        // create the list for the ontologies
        m_oimodelInfoTableModel=new OIModelInfoTableModel(registryModule);
        m_oimodelInfoTableModel.addTableModelListener(new TableModelListener() {
            public void tableChanged(TableModelEvent e) {
                // detect the insertion of elements to see if they are password fields
                for (int currow=0; currow<=m_oimodelInfoTableModel.getRowCount(); currow++) {
                    for (int curcol=0; curcol<m_oimodelInfoTableModel.getColumnCount(); curcol++){
                        try {
                            String data = (String)m_oimodelInfoTableModel.getValueAt(currow, curcol);
                            String[] lines=data.split("\n");
                            if ( lines.length > m_maxlines ) {
                                m_maxlines = lines.length;
                                if ( m_oimodelInfoTable != null ) {
                                    m_oimodelInfoTable.setRowHeight( m_originalrowheight * m_maxlines);
                                }
                            }
                        }
                        catch(Exception ex) {
                        }
                    }
                }
            }
        });

        m_oimodelInfoTable=new SmartTable(m_oimodelInfoTableModel) {
            protected JTableHeader createDefaultTableHeader() {
                return new SortableTableHeader(columnModel);
            }
        };
        m_originalrowheight=m_oimodelInfoTable.getRowHeight();
        m_oimodelInfoTable.setRowHeight( m_originalrowheight * m_maxlines);
        m_oimodelInfoTable.getTableHeader().addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent event) {
                if (SortableTableHeader.SORTED_COLUMN_PROPERTY.equals(event.getPropertyName()) || SortableTableHeader.SORT_ASCENDING_PROPERTY.equals(event.getPropertyName()))
                    updateSorting();
            }
        });
        // listen to changes of the selection status
        m_oimodelInfoTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            public void valueChanged(ListSelectionEvent e) {
                RegistryViewable.this.updateActions();
            }
        });

        // create the search panel
        JPanel searchpanel=new JPanel(new GridBagLayout());
        GridBagConstraints gbc=new GridBagConstraints();
        gbc.insets = new Insets(0,0,0,0);
        gbc.ipadx = 0; gbc.ipady = 0;
        gbc.gridwidth = 1;gbc.gridheight = 1;
        gbc.weightx = 100;gbc.weighty = 100;
        //add search parameters pane
        gbc.weighty = 95;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.BOTH;
        searchpanel.add(m_searchParametersPane,gbc);
        //add search button
        gbc.weighty = 5;
        gbc.gridy = 1;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        searchpanel.add(searchbutton,gbc);
        searchpanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

        // create the new OIModel panel
        JPanel oimodelchooserpanel=new JPanel(new BorderLayout(0,3));
        m_oimodelChooser=new ViewableSelector(registryModule.getAppDriver().getModule(OIModelerModule.NAME));
        m_oimodelChooser.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                m_registerOIModelPane.OIModelChanged();
            }
        });
        oimodelchooserpanel.add(registryModule.getAppDriver().getLocalizationManager().getLabel("registry.selectOIModel"),BorderLayout.NORTH);
        oimodelchooserpanel.add(m_oimodelChooser,BorderLayout.CENTER);

        JPanel newOIModelpanel=new JPanel(new GridBagLayout());
        //add the chooser panel
        gbc.gridy = 0;
        gbc.weighty = 5;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        newOIModelpanel.add(oimodelchooserpanel,gbc);
        //add register OI model pane
        gbc.weighty = 90;
        gbc.gridy = 1;
        gbc.fill = GridBagConstraints.BOTH;
        newOIModelpanel.add(m_registerOIModelPane,gbc);
        //add the button
        gbc.weighty = 5;
        gbc.gridy = 2;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        newOIModelpanel.add(registerNewOIModelButton,gbc);
        newOIModelpanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

        // create the panel for the result list
        m_oimodelInfoTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        m_oimodelInfoTable.setDefaultRenderer(String.class,new SmartTable.MultiLineCellRenderer());
        JScrollPane listscrollPane=new JScrollPane(m_oimodelInfoTable);
        listscrollPane.getViewport().setBackground(m_oimodelInfoTable.getBackground());

        JPanel listpanel=new JPanel(new BorderLayout());
        listpanel.add(listscrollPane,BorderLayout.CENTER);
        // create the splitter pane for the search
        JSplitPane searchSplit=new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,searchpanel,listpanel);
        searchSplit.setDividerLocation(0.20);
        searchSplit.setResizeWeight(0.4);

        // create the main tabbed pane for the left side
        m_leftTabbedPane = new JTabbedPane();

        m_leftTabbedPane.addTab(registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.tab.searchLabel"), null, searchSplit, registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.tab.searchTip") );
        m_leftTabbedPane.addTab(registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.tab.registerOIModelLabel"), null, newOIModelpanel, registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.tab.registerOIModelTip"));
        m_leftTabbedPane.addTab(registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.tab.manageElementsLabel"), null, manageElementsPane, registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.tab.manageElementsTip"));

        m_leftTabbedPane.setSelectedIndex(0);

        m_component=new JPanel(new BorderLayout());
        m_component.add(m_leftTabbedPane,BorderLayout.CENTER);
    }
    /**
     * Returns the registry.
     *
     * @return                                  the registry in this pane
     */
    public Registry getRegistry() {
        return m_registry;
    }
    /**
     * Overridden to set the name of the registry.
     *
     * @param anchor                            the viewable anchor
     */
    public void setViewableAnchor(ViewableAnchor anchor) {
        super.setViewableAnchor(anchor);
        try {
            m_viewableAnchor.setTitle(m_module.getAppDriver().getLocalizationManager().format("registry.viewerTitleURI",new Object[] { m_registry.getRegistryOIModelInstance().getPhysicalURI() }));
        }
        catch (KAONException e) {
            m_viewableAnchor.setTitle(m_module.getAppDriver().getLocalizationManager().getPhrase("registry.viewerTitle"));
        }
    }
    /**
     * Applies the search parameters.
     */
    public void doSearch() {
        try {
            try {
                m_module.getAppDriver().startWaitState();

                String oimodelName=m_searchParametersPane.getOIModelName();
                if (oimodelName!=null && oimodelName.length()==0)
                    oimodelName=null;
                Collection appliedInFields=m_searchParametersPane.getApplicationFields();
                if (appliedInFields.isEmpty())
                    appliedInFields=null;
                Collection projects=m_searchParametersPane.getProjects();
                if (projects.isEmpty())
                    projects=null;
                Collection creators=m_searchParametersPane.getCreators();
                if (creators.isEmpty())
                    creators=null;
                Collection terms=m_searchParametersPane.getTerms();
                if (terms.isEmpty())
                    terms=null;
                Collection oimodelInfos=m_registry.search(oimodelName,m_searchParametersPane.getStartDate(),m_searchParametersPane.getEndDate(),appliedInFields,projects,creators,terms);
                m_oimodelInfoTableModel.setElements(oimodelInfos);
            }
            finally {
                m_module.getAppDriver().endWaitState();
            }
        }
        catch (KAONException e) {
            m_viewableAnchor.setTitle(m_module.getAppDriver().getLocalizationManager().getPhrase("registry.viewerTitle"));
        }
        updateActions();
    }
    /**
     * Returns the currently selected OIModel in the new oimodel pane
     *
     * @return            the currently selected OIModel
     */
    protected OIModel getOIModel() {
        OIModelerViewable viewable=(OIModelerViewable)m_oimodelChooser.getSelectedViewable();
        if (viewable==null)
            return null;
        else
            return viewable.getOIModel();
    }

    /**
     * Displays the register oi model tab
     */
    public void registerOIModel() {
        m_leftTabbedPane.setSelectedIndex(1);
    }
    /**
     * Registers a new OIModel with the entered parameters
     */
    protected void registerNewOIModel() {
        Registry registry=getRegistry();

        try {
            OIModel oimodel=getOIModel();
            if (oimodel!=null) {
                OIModelInfo oimodelInfo=registry.locateByLogicalURI(oimodel.getLogicalURI());
                if (oimodelInfo!=null) {
                    String text=getModule().getAppDriver().getLocalizationManager().format("registry.oimodelAlreadyRegistered",new Object[] { oimodel.getLogicalURI() });
                    JOptionPane.showMessageDialog(getModule().getAppDriver().getMainFrameWindow(),text,getModule().getAppDriver().getLocalizationManager().getPhrase("registry.error"),JOptionPane.ERROR_MESSAGE);
                }
                else {
                    oimodelInfo=registry.newOIModelInfo();
                    oimodelInfo.create();
                    oimodelInfo.setPhysicalURI(oimodel.getPhysicalURI());
                    oimodelInfo.setLogicalURI(oimodel.getLogicalURI());
                    oimodelInfo.setOIModelName(m_registerOIModelPane.getOIModelName());
                    oimodelInfo.setCreationDate( m_registerOIModelPane.getStartDate() );
                    Iterator iterator=m_registerOIModelPane.getApplicationFields().iterator();
                    while (iterator.hasNext()) {
                        ApplicationField applicationField=(ApplicationField)iterator.next();
                        oimodelInfo.addAppliedInField(applicationField);
                    }
                    iterator=m_registerOIModelPane.getProjects().iterator();
                    while (iterator.hasNext()) {
                        Project project=(Project)iterator.next();
                        oimodelInfo.addProject(project);
                    }
                    iterator=m_registerOIModelPane.getCreators().iterator();
                    while (iterator.hasNext()) {
                        Party party=(Party)iterator.next();
                        oimodelInfo.addCreator(party);
                    }
                    iterator=m_registerOIModelPane.getTerms().iterator();
                    while (iterator.hasNext()) {
                        Term term=(Term)iterator.next();
                        oimodelInfo.addAboutTerm(term);
                    }
                    registry.commit();

                    JOptionPane.showMessageDialog(getModule().getAppDriver().getMainFrameWindow(),getModule().getAppDriver().getLocalizationManager().getPhrase("registry.registernewoimodel.success.message"),getModule().getAppDriver().getLocalizationManager().getPhrase("registry.registernewoimodel.success.title"),JOptionPane.INFORMATION_MESSAGE);
                }
            }
        }
        catch (KAONException error) {
            registry.rollback();
            getModule().getAppDriver().displayErrorNotification(error);
        }
    }
    /**
     * Disposes of this viewable.
     */
    public void dispose() {
        m_oimodelChooser.dispose();
    }
    /**
     * Called when the viewable anchor is closing.
     */
    public void closing() {
        if (!m_registryProcessed)
            m_viewableAnchor.dispose();
    }
    /**
     * Starts the processing of the registry.
     */
    public void startRegistryProcessing() {
        m_registryProcessed=true;
        m_component.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    }
    /**
     * Finishes the processing of the registry.
     */
    public void finishRegistryProcessing() {
        if (SwingUtilities.isEventDispatchThread())
            finishRegistryProcessingMainThread();
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    finishRegistryProcessingMainThread();
                }
            });
    }
    /**
     * Finishes reigstry processing on the main thread.
     */
    protected void finishRegistryProcessingMainThread() {
        m_registryProcessed=false;
        m_component.setCursor(null);
    }
    /**
     * Returns <code>true</code> if the commands for this registry are enabled.
     *
     * @return                                  <code>true</code> if the commands can be executed on the registry
    */
    public boolean commandsAreEnabled() {
        return true;
    }
    /**
     * Called to save the UI state of the viewable.
     *
     * @param configuration                 the configuration
     * @return                              <code>true</code> if the configuration saved
     */
    public boolean saveUIState(Configuration configuration) {
        return true;
    }
    /**
     * Updates the sorting of the model.
     */
    protected void updateSorting() {
        SortableTableHeader header=(SortableTableHeader)m_oimodelInfoTable.getTableHeader();
        if (header.getSortedColumn()!=null)
            m_oimodelInfoTableModel.sortModel(header.getSortedColumn().getModelIndex(),header.getSortAscending());
        updateActions();
    }
    /**
     * Returns the currently selected OIModelInfos.
     *
     * @return                               iterator of the currently selected OIModelInfos
     */
    public Iterator getCurrentOIModelInfos() {
        ArrayList oimodelInfos=new ArrayList();
        ListSelectionModel selectionModel=m_oimodelInfoTable.getSelectionModel();
        int startIndex=selectionModel.getMinSelectionIndex();
        int endIndex=selectionModel.getMaxSelectionIndex();
        if (startIndex>-1) {
            for (int i=startIndex;i<=endIndex;i++) {
                OIModelInfo curOIModelInfo=m_oimodelInfoTableModel.getOIModelInfoAt(i,0);
                if (curOIModelInfo!=null)
                    oimodelInfos.add(curOIModelInfo);
            }
        }
        return oimodelInfos.iterator();
    }
}
