package de.fzi.wim.registry.gui.ui;

import java.util.List;
import java.util.Collections;
import java.text.Collator;
import java.util.Comparator;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Date;
import java.text.DateFormat;
import javax.swing.table.AbstractTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import de.fzi.wim.registry.api.*;
import de.fzi.wim.registry.gui.*;

/**
 * A table model containing OIModelInfo objects from the registry.
 */
public class OIModelInfoTableModel extends AbstractTableModel {
    /** The date format for the model. */
    protected DateFormat m_dateFormat;
    /** The reigstry module. */
    protected RegistryModule m_registryModule;
    /** The list of elements. */
    protected List m_elements;

    /**
     * Creates an instance of this class.
     *
     * @param registryModule                    the registry module
     */
    public OIModelInfoTableModel(RegistryModule registryModule) {
        m_dateFormat=DateFormat.getDateInstance(DateFormat.SHORT);
        m_registryModule=registryModule;
        m_elements=new ArrayList();
    }
    /**
     * Refreshes the model with supplied elements.
     *
     * @param oimodelInfos                      the OI-model infos
     * @throws KAONException                    thrown if there is a problem
     */
    public void setElements(Collection oimodelInfos) throws KAONException {
        m_elements.clear();
        Iterator iterator=oimodelInfos.iterator();
        while (iterator.hasNext()) {
            OIModelInfo oimodelInfo=(OIModelInfo)iterator.next();
            m_elements.add(new TableRow(oimodelInfo));
        }
        fireTableDataChanged();
    }
    /**
     * Returns the class of the given column.
     *
     * @param columnIndex                       the index of the column
     * @return                                  the class of supplied column
     */
    public Class getColumnClass(int columnIndex) {
        switch (columnIndex) {
        case 0:
        default:
            return String.class;
        }
    }
    /**
     * Returns the number of columns.
     *
     * @return                                  the number of columns
     */
    public int getColumnCount() {
        return 6;
    }
    /**
     * Returns the name of given column.
     *
     * @param columnIndex                       the index of the column
     * @return                                  the name of the given column
     */
    public String getColumnName(int columnIndex) {
        return m_registryModule.getAppDriver().getLocalizationManager().getPhrase("registry.oimodelInfoTableModel."+columnIndex);
    }
    /**
     * Returns the number of rows.
     *
     * @return                                  the number of rows
     */
    public int getRowCount() {
        return m_elements.size();
    }

    protected String copyIteratorToMultilineString(Iterator iterator) {
        StringBuffer result=new StringBuffer();
        while (iterator.hasNext()) {
            if (result.length() > 0) {
                result.append(new String("\n"));
            }
            result.append(iterator.next().toString());
        }
        return result.toString();
    }

    protected Iterator getApplicationFieldNames(Iterator applicationsfields) {
        List results=new ArrayList();

        while (applicationsfields.hasNext()) {
            try {
                results.add(((ApplicationField)applicationsfields.next()).getApplicationFieldName());
            }
            catch (KAONException e) {
            }
        }
        return results.iterator();
    }

    protected Iterator getAboutTermDefinitions(Iterator aboutterms) {
        List results=new ArrayList();

        while (aboutterms.hasNext()) {
            try {
                //create the list of synonyms for the term
                StringBuffer buffer=new StringBuffer();
                Term term=(Term)aboutterms.next();
                Iterator iterator=term.getTermSynonyms(KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en")).iterator();
                while (iterator.hasNext()) {
                    String synonym=(String)iterator.next();
                    buffer.append(synonym);
                    if (iterator.hasNext())
                        buffer.append(", ");
                }
                //append the term definition
                String termdefinition=term.getTermDefinition();
                if (termdefinition!=null) {
                    if (buffer.length() > 0)
                        buffer.append(" (");
                    buffer.append(termdefinition);
                    if (buffer.length() > termdefinition.length())
                        buffer.append(")");
                }
                results.add(buffer.toString());
            }
            catch (KAONException e) {
            }
        }
        return results.iterator();
    }

    protected Iterator getCreatedInProjectNames(Iterator projects) {
        List results=new ArrayList();

        while (projects.hasNext()) {
            try {
                results.add(((Project)projects.next()).getProjectName());
            }
            catch (KAONException e) {
            }
        }
        return results.iterator();
    }

    protected Iterator getPartyNames(Iterator parties) {
        List results=new ArrayList();

        while (parties.hasNext()) {
            try {
                Object current=parties.next();
                if (current instanceof Person) {
                    Person person=(Person)current;
                    results.add(person.getPersonName());
                }
                else if (current instanceof Organization) {
                    Organization organization=(Organization)current;
                    results.add(organization.getOrganizationName());
                }
            }
            catch (KAONException e) {
            }
        }
        return results.iterator();
    }

    /**
     * Returns a value at given coordinates.
     *
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     * @return                                  the value
     */
    public Object getValueAt(int rowIndex,int columnIndex) {
        TableRow tableRow=(TableRow)m_elements.get(rowIndex);
		return tableRow.getColumnText(columnIndex);
    }
    /**
     * Checks whether the cell is editable.
     *
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     * @return                                  <code>true</code> if the cell is editable
     */
    public boolean isCellEditable(int rowIndex,int columnIndex) {
        return false;
    }
    /**
     * Sets the value at given coordinates.
     *
     * @param aValue                            the value
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     */
    public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
    }
    /**
     * Sorts the model.
     *
     * @param columnIndex                       the index of the column that is sorted
     * @param ascending                         if <code>true</code>, the column is sorted in the ascending order
     */
    public void sortModel(int columnIndex,boolean ascending) {
		Comparator comparator=new TableRowComparator(columnIndex,ascending);

		Collections.sort(m_elements,comparator);

		fireTableDataChanged();
    }

	/**
	 * Class to compare two table row elements in the specified column index.
	 */
    protected static class TableRowComparator implements Comparator {
		protected int m_columnindex;
		protected boolean m_ascending;

        public TableRowComparator( int columnindex,boolean ascending ) {
			m_columnindex = columnindex;
			m_ascending=ascending;
		}

		 public int compare(Object o1,Object o2) {
			// just convert both objects to strings
			String s1=((TableRow)o1).getColumnText(m_columnindex);
			String s2=((TableRow)o2).getColumnText(m_columnindex);

			// and use the collator with the current locale to compare them
			if ( m_ascending )
				return Collator.getInstance().compare(s1,s2);
			else
				return -Collator.getInstance().compare(s1,s2);
        }
    }

    /**
     * The element of the table model.
     */
    protected class TableRow {
        /** The OI-model info. */
        protected OIModelInfo m_oimodelInfo;
        /** OI-models name. */
        protected String m_oimodelName;
        /** cached values */
        protected String m_date;
        protected String m_terms;
        protected String m_fields;
        protected String m_projects;
        protected String m_parties;

		/**
		 * Create a new instance of this class
		 *
		 * @param oimodelInfo        the infos about the oimodel
		 * @throws KAONAException    thrown if some error occures
		 */
        public TableRow(OIModelInfo oimodelInfo) throws KAONException {
            m_oimodelInfo=oimodelInfo;
            m_oimodelName=oimodelInfo.getOIModelName();

            Date creationDate=m_oimodelInfo.getCreationDate();
            if (creationDate!=null)
                m_date=m_dateFormat.format(creationDate);
            m_terms=copyIteratorToMultilineString(getAboutTermDefinitions(this.m_oimodelInfo.getAboutTerms().iterator()));
            m_fields=copyIteratorToMultilineString(getApplicationFieldNames(this.m_oimodelInfo.getAppliedInField().iterator()));
            m_projects=copyIteratorToMultilineString(getCreatedInProjectNames(this.m_oimodelInfo.getCreatedInProject().iterator()));
            m_parties=copyIteratorToMultilineString(getPartyNames(this.m_oimodelInfo.getCreators().iterator()));
        }

		/**
		 * Returns the text of the specified column index.
		 *
		 * @param columnindex        zero based column index
		 * @return                   the string that can be displayed on the screen
		 */
		public String getColumnText( int columnindex ) {
            switch (columnindex) {
                case 0:
                    return m_oimodelName;
                case 1:
                    return m_date;
                case 2:
                    return m_terms;
                case 3:
                    return m_fields;
                case 4:
                    return m_projects;
                case 5:
                    return m_parties;
                default:
                    return null;
            }
		}
    }

     /**
     * Returns the OIModel object at the given coordinates.
     *
     * @param rowIndex                          the row
     * @param columnIndex                       the column
     * @return                                  the OIModelInfo object
     */
    public OIModelInfo getOIModelInfoAt(int rowIndex,int columnIndex) {
        TableRow tableRow=(TableRow)m_elements.get(rowIndex);
        switch (columnIndex) {
        case 0:
            return tableRow.m_oimodelInfo;
        default:
            return null;
        }
    }
}
