package de.fzi.wim.registry.gui.ui;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.HashSet;
import java.util.HashMap;
import java.util.Enumeration;
import java.util.Date;
import java.awt.Component;
import java.awt.Insets;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.JPanel;
import javax.swing.JButton;
import javax.swing.JTextField;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.TableColumn;
import javax.swing.table.JTableHeader;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import de.fzi.wim.guibase.tables.*;
import de.fzi.wim.guibase.localization.*;

import de.fzi.wim.registry.api.*;
import de.fzi.wim.registry.gui.*;

import de.fzi.wim.guibase.appdriver.AbstractModule;
import de.fzi.wim.oimodeler.ui.ProgressDlg;

/**
 * The pane that can be used to present information about an OI-model in the registry.
 */
public class OIModelInfoPane extends JPanel {
    /** The registry viewable. */
    protected RegistryViewable m_registryViewable;
    /** The OI-model name. */
    protected JTextField m_oimodelName;
    /** The searching dates of OI-model creation. */
    protected DateSelectorPane m_oimodelSearchDates;
    /** The selector for the fields, projects, creators and termn. */
    protected ExtElementSelector m_selector;
    /** The fields in which the OI-model was applied. */
    protected String m_appliedInFieldsName;
    /** The projects in which the OI-model was applied. */
    protected String m_appliedInProjectsName;
    /** The creators of the OI-model. */
    protected String m_creatorsName;
    /** The terms of the OI-model. */
    protected String m_termsName;
    /** The automatically extracted terms of the OI-model. */
    protected String m_extractedTermsName;
    /** The source for the automatically extracted terms of the OI-model. */
    protected ExtractedTermsElementSource m_extractedTermsSource;

    /** the different pane types */

    /** With this constant the pane is created as a search pane.
     * Only a single date field is used to enter. Either <code>getStartDate()</code> or <code>getEndDate()</code> return this date value.
     */
    public static int PANE_SEARCH = 0;
    /** With this constant the pane is created as a create pane.
     * Two date fields are used to enter a date range for a search. <code>getStartDate()</code> and <code>getEndDate()</code> return those date values.
     */
    public static int PANE_CREATE = 1;

    /**
     * Creates an instance of this pane.
     *
     * @param registryViewable                  the viewable
     * @param type                              type of the pane that should be created
     */
    public OIModelInfoPane(RegistryViewable registryViewable, int type) {
        super(new GridBagLayout());
        m_registryViewable=registryViewable;

        LocalizationManager localizationManager=m_registryViewable.getModule().getAppDriver().getLocalizationManager();
        m_oimodelName=new JTextField();
        if (type == PANE_SEARCH) {
            m_oimodelSearchDates=new DateSelectorPane(localizationManager.getCheckBox("registry.search.startdatecheckbox"),localizationManager.getCheckBox("registry.search.enddatecheckbox"),null,null);
        }
        else {
            m_oimodelSearchDates=new DateSelectorPane(localizationManager.getCheckBox("registry.search.creationdatecheckbox"),null,null,null);
        }
        JButton addButton = localizationManager.getButton("registry.search.addButton");
        JButton removeButton = localizationManager.getButton("registry.search.removeButton");
        addButton.setIcon(localizationManager.getImageIcon("registry.arrow_down"));
        removeButton.setIcon(localizationManager.getImageIcon("registry.arrow_up"));

        m_appliedInFieldsName=localizationManager.getPhrase("registry.search.Fields");
        m_appliedInProjectsName=localizationManager.getPhrase("registry.search.Projects");
        m_creatorsName=localizationManager.getPhrase("registry.search.Creators");
        m_termsName=localizationManager.getPhrase("registry.search.Terms");
        m_extractedTermsName=localizationManager.getPhrase("registry.search.ExtractedTerms");

        m_selector=new ExtElementSelector(addButton , removeButton, localizationManager.getPhrase("registry.search.categoryName"),localizationManager.getPhrase("registry.search.valueName"));
        m_selector.addElementSource( new PartyElementSource(m_registryViewable,m_creatorsName,m_creatorsName),localizationManager.getLabel("registry.search.searchLabel"),getButton(localizationManager,"registry.search.searchButton","registry.magnifier"),null);
        m_selector.addElementSource( new ProjectsElementSource(m_registryViewable,m_appliedInProjectsName,m_appliedInProjectsName),localizationManager.getLabel("registry.search.searchLabel"),getButton(localizationManager,"registry.search.searchButton","registry.magnifier"),null);
        m_selector.addElementSource( new ApplicationFieldsElementSource(m_registryViewable,m_appliedInFieldsName,m_appliedInFieldsName),localizationManager.getLabel("registry.search.searchLabel"),getButton(localizationManager,"registry.search.searchButton","registry.magnifier"),null);
        m_selector.addElementSource( new TermsElementSource(m_registryViewable,m_termsName,m_termsName),localizationManager.getLabel("registry.search.searchLabel"),getButton(localizationManager,"registry.search.searchButton","registry.magnifier"),null);
        if (type == PANE_CREATE) {
            m_extractedTermsSource=new ExtractedTermsElementSource(m_registryViewable,m_extractedTermsName,m_extractedTermsName);
            m_selector.addElementSource( m_extractedTermsSource,localizationManager.getLabel("registry.search.searchLabel"),getButton(localizationManager,"registry.search.searchButton","registry.magnifier"),null);
        }

        GridBagConstraints gbc=new GridBagConstraints();
        gbc.insets = new Insets(0,0,0,0);
        gbc.ipadx = 0; gbc.ipady = 0;
        gbc.gridwidth = 1;
        gbc.gridheight = 1;
        gbc.weightx = 100;
        // add a label
        gbc.weighty = 2;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.fill = GridBagConstraints.NONE;
        gbc.anchor = GridBagConstraints.SOUTHWEST;
        add( localizationManager.getLabel("registry.oimodelName"), gbc );
        // add the text field for the ontology name
        gbc.gridy = 1;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        add( m_oimodelName, gbc );
        // add another label
        gbc.gridy = 2;
        gbc.fill = GridBagConstraints.NONE;
        gbc.anchor = GridBagConstraints.SOUTHWEST;
        add( localizationManager.getLabel("registry.oimodelCreationDate"), gbc );
        // add the creation date selector
        gbc.weightx = 70;
        gbc.weighty = 5;
        gbc.gridy = 3;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        add( m_oimodelSearchDates, gbc );
        //add the element selector
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weighty = 93;
        gbc.gridy = 4;
        add( m_selector, gbc );
    }
    /**
     * Returns a <code>JButton</code> with the specified text and icon.
     *
     * @param localizationManager               current Localization manager
     * @param buttonID                          ID to the text of the button
     * @param iconID                            ID to the icon of the button
     * @return                                  the name of the OI-model
     */
    protected JButton getButton(LocalizationManager localizationManager,String buttonID,String iconID) {
        JButton button = localizationManager.getButton(buttonID);
        button.setIcon(localizationManager.getImageIcon(iconID));
        return button;
    }
    /**
     * Returns the name of the OI-model.
     *
     * @return                                  the name of the OI-model
     */
    public String getOIModelName() {
        return m_oimodelName.getText();
    }
    /**
     * Returns the entered starting date.
     *
     * @return                                  the starting date ( <code>null</code> if it should not be included in the search)
     */
    public Date getStartDate() {
        return m_oimodelSearchDates.getStartDate();
    }
    /**
     * Returns the entered ending date.
     *
     * @return                                  the ending date ( <code>null</code> if it should not be included in the search)
     */
    public Date getEndDate() {
        return m_oimodelSearchDates.getEndDate();
    }
    /**
     * Returns the collection of all selected application fields.
     *
     * @return                                  the collection of selected application fields
     */
    public List getApplicationFields() {
        return m_selector.getSelectedElements( m_appliedInFieldsName );
    }
    /**
     * Returns the collection of all selected projects.
     *
     * @return                                  the collection of selected projects
     */
    public List getProjects() {
        return m_selector.getSelectedElements( m_appliedInProjectsName );
    }
    /**
     * Returns the collection of all selected creators.
     *
     * @return                                  the collection of selected creators
     */
    public List getCreators() {
        return m_selector.getSelectedElements( m_creatorsName );
    }
    /**
     * Returns the collection of all selected terms.
     *
     * @return                                  the collection of selected terms
     */
    public List getTerms() {
        HashMap termMap=new HashMap();
        Iterator iterator=m_selector.getSelectedElements(m_termsName).iterator();
        while(iterator.hasNext()) {
            Term term=(Term)iterator.next();
            try {
                termMap.put(term.getEntity().getURI(),term);
            }
            catch(KAONException ignored) {
            }
        }
        if(m_extractedTermsSource!=null) {
            iterator=m_selector.getSelectedElements(m_extractedTermsName).iterator();
            while(iterator.hasNext()) {
                Term term=(Term)iterator.next();
                try {
                    termMap.put(term.getEntity().getURI(),term);
                }
                catch(KAONException ignored) {
                }
            }
        }

        return new ArrayList(termMap.values());
    }
    /**
     * This method should be called when the selected OIModel has changed in the <code>RegistryViewable</code>.
     */
    public void OIModelChanged() {
        if(m_extractedTermsSource!=null)
            m_extractedTermsSource.clear();
    }


   /**
     * The element source for the application fields.
     */
    protected static abstract class JElementSource implements ExtElementSelector.ElementSource {
        protected RegistryViewable m_registryViewable;
        protected String m_Name;
        protected String m_Filter;
        protected String m_displayName;

        /**
         * Returns the name of this element source that should be displayed in the list.
         *
         * @param registryViewable       the attached viewable
         * @param name                   the name of the element source
         * @param displayName            the displayable name of the element source
         */
        public JElementSource(RegistryViewable registryViewable, String name, String displayName) {
             m_registryViewable=registryViewable;
             m_Name = name;
             m_displayName = displayName;
        }
        /**
         * Returns the name of this element source.
         *
         * @return    name of the element source
         */
        public String getName() {
            return m_Name;
        };
        /**
         * Returns the name of this element source that should be displayed in the list.
         *
         * @return    display name of the element source
         */
        public String getDisplayName() {
            return m_displayName;
        }

        /**
         * Returns the text for the given object.
         *
         * @param object                the object
         * @return                      the text for given object
         */
        public String getObjectText(Object object) {
            try {
                return getKAONObjectText( object );
            }
            catch (KAONException e) {
                return "---";
            }
        }
        /**
         * Returns the component of this ElementSource. If <code>null</code> is returned, <code>ExtElementSelector</code> will manage
         * the display and selection of the elements (with a <code>JList</code>).
         *
         * @return                      the component may be <code>null</code>
         */
        public Component getComponent() {
            return null;
        }
        /**
         * Returns the selected elements of this ElementSource. Only used if <code>getComponent()</code> does not
         * return <code>null</code>
         *
         * @return                      the collection of selected elements of this ElementSource (never <code>null</code>)
         */
        public Collection getSelectedElements(){
            return null;
        }
        /**
         * Sets the filter for the elements of this ElementSource.
         *
         * @param filter                the filter string for the elements
         */
        public void setFilter(String filter){
            m_Filter=filter;
        }
        /**
         * Returns the elements of this ElementSource.
         *
         * @return                      the collection of elements of this ElementSource (never <code>null</code>)
         */
        public Collection getElements() {
            Collection result=new ArrayList();
            try {
                if (! "".equals(m_Filter)) {
                    try {
                        this.m_registryViewable.getModule().getAppDriver().startWaitState();

                        result=getKAONElements(m_Filter);
                    }
                    finally{
                        this.m_registryViewable.getModule().getAppDriver().endWaitState();
                    }
                }
            }
            catch (KAONException e) {
            }
            return result;
        }
        /**
         * Called when the user requests a new element to be added.
         *
         * @param newElementText                  a proposal for the text of the new element
         */
        public void addNewElement(String newElementText) {
            NewRegistryEntityDlg newRegistryEntityDlg=new NewRegistryEntityDlg((RegistryModule)this.m_registryViewable.getModule(),getAddNewElementID());
            newRegistryEntityDlg.setEntityName( newElementText );
            newRegistryEntityDlg.setVisible(true);
            if (newRegistryEntityDlg.isConfirmed()) {
                Registry registry=this.m_registryViewable.getRegistry();
                try {
                    registry.commit();
                    JOptionPane.showMessageDialog(m_registryViewable.getModule().getAppDriver().getMainFrameWindow(),m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase(getAddNewElementID()+".success.message"),m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase(getAddNewElementID()+".success.title"),JOptionPane.INFORMATION_MESSAGE);
                }
                catch (KAONException e) {
                    registry.rollback();
                    m_registryViewable.getModule().getAppDriver().displayErrorNotification(e);
                }
            }
        }
        /**
         * Called when an id for a new element is requested
         *
         * @return                      the id for the new element
         */
        protected abstract String getAddNewElementID();

        /**
         * Called when an id for a new element is requested
         *
         * @param registry              the registry object that the new object should be created in
         * @param entityName            the name of the new entity
         * @return                      the new element
         */
        protected abstract Object createObject( Registry registry, String entityName ) throws KAONException;

        /**
         * Returns the text for the given KAON object.
         *
         * @param object                the object
         * @return                      the text for given object
         * @throws KAONException        thrown if an error occures
         */
        public abstract String getKAONObjectText( Object object ) throws KAONException;
        /**
         * Returns the LAON elements of this ElementSource.
         *
         * @param filter                the filter string for the elements
         * @return                      the collection of elements of this ElementSource (never <code>null</code>)
         * @throws KAONException        thrown if an error occures
         */
        public abstract Collection getKAONElements( String filter ) throws KAONException;
    }

    /**
     * The element source for the application fields.
     */
    protected static class ApplicationFieldsElementSource extends JElementSource {
        public ApplicationFieldsElementSource(RegistryViewable registryViewable, String name, String displayName) {
             super( registryViewable, name, displayName );
        }
       public String getKAONObjectText(Object object) throws KAONException {
            ApplicationField applicationField=(ApplicationField)object;
            return applicationField.getApplicationFieldName();
        }
        public Collection getKAONElements(String filter) throws KAONException {
            Registry registry=this.m_registryViewable.getRegistry();
            Collection result=new HashSet();

            Iterator iterator=registry.getApplicationFields(filter);
            while (iterator.hasNext()) {
                result.add(iterator.next());
            }
            /*
            Iterator iterator=registry.getRegistryOIModelInstance().getConcept(RegistryConstants.OMO_ApplicationField).getAllInstances().iterator();
            while (iterator.hasNext()) {
                Object curfield=registry.getApplicationField(((Instance)iterator.next()).getURI());
                String fieldName=getObjectText(curfield);
                if ( fieldName != null && fieldName.indexOf(filter) != -1 ) {
                    result.add(curfield);
                }
            }
            */
            return result;
        }
        protected String getAddNewElementID() {
            return "registry.newApplicationField";
        }
        protected Object createObject(Registry registry,String entityName) throws KAONException {
            ApplicationField applicationField=registry.getApplicationField(registry.getRegistryOIModelInstance().createNewURI());
            applicationField.create();
            applicationField.setApplicationFieldName(entityName);
            return applicationField;
        }
    }

    /**
     * The element source for the projects.
     */
    protected static class ProjectsElementSource extends JElementSource {
        public ProjectsElementSource(RegistryViewable registryViewable, String name, String displayName) {
             super( registryViewable, name, displayName );
        }
        public String getKAONObjectText(Object object) throws KAONException {
            Project project=(Project)object;
            return project.getProjectName();
        }
        public Collection getKAONElements(String filter) throws KAONException {
            Registry registry=this.m_registryViewable.getRegistry();
            Collection result=new HashSet();
            Iterator iterator=registry.getProjects(filter);//registry.getRegistryOIModelInstance().getConcept(RegistryConstants.OMO_Project).getAllInstances().iterator();
            while (iterator.hasNext()) {
                result.add(iterator.next());
            }
            /*
            Iterator iterator=registry.getRegistryOIModelInstance().getConcept(RegistryConstants.OMO_Project).getAllInstances().iterator();
            while (iterator.hasNext()) {
                Object curproject=registry.getProject(((Instance)iterator.next()).getURI());
                String projectName=getObjectText(curproject);
                if ( projectName != null && projectName.indexOf(filter) != -1 ) {
                    result.add(curproject);
                }
            }
            */
            return result;
        }
        protected String getAddNewElementID() {
            return "registry.newProject";
        }
        protected Object createObject(Registry registry,String entityName) throws KAONException {
            Project project=registry.getProject(registry.getRegistryOIModelInstance().createNewURI());
            project.create();
            project.setProjectName(entityName);
            return project;
        }
    }

    /**
     * The element source for parties.
     */
    protected static class PartyElementSource extends JElementSource {
        public PartyElementSource(RegistryViewable registryViewable, String name, String displayName) {
             super( registryViewable, name, displayName );
        }
        public String getKAONObjectText(Object object) throws KAONException {
            if (object instanceof Person) {
                Person person=(Person)object;
                return person.getPersonName();
            }
            else if (object instanceof Organization) {
                Organization organization=(Organization)object;
                return organization.getOrganizationName();
            }
            return "---";
        }
        public Collection getKAONElements(String filter) throws KAONException {
            Registry registry=this.m_registryViewable.getRegistry();
            Collection result=new HashSet();
            Iterator iterator=registry.getParties(filter);
            while (iterator.hasNext()) {
                result.add(iterator.next());
            }
            return result;
        }
        protected String getAddNewElementID() {
            return "registry.newParty";
        }
        /**
         * Called when the user requests a new element to be added.
         *
         * @param newElementText                  a proposal for the text of the new element
         */
        public void addNewElement(String newElementText) {
            NewPartyEntityDlg newPartyEntityDlg=new NewPartyEntityDlg((RegistryModule)this.m_registryViewable.getModule(),getAddNewElementID());
            newPartyEntityDlg.setEntityName(newElementText);
            newPartyEntityDlg.setVisible(true);
            if (newPartyEntityDlg.isConfirmed()) {
                Registry registry=this.m_registryViewable.getRegistry();
                try {
                    Party newparty;
                    // create the desired entity
                    if (newPartyEntityDlg.isPerson()==true) {
                        newparty=registry.getPerson(registry.getRegistryOIModelInstance().createNewURI());
                        newparty.create();
                        ((Person)newparty).setPersonName(newPartyEntityDlg.getEntityName());
                    }
                    else {
                        newparty=registry.getOrganization(registry.getRegistryOIModelInstance().createNewURI());
                        newparty.create();
                        ((Organization)newparty).setOrganizationName(newPartyEntityDlg.getEntityName());
                    }
                    // and set the address information
                    String value=newPartyEntityDlg.getStreetAndNumber();
                    if (value!=null && !"".equals(value))
                        newparty.setStreetAndNumber(value);
                    value=newPartyEntityDlg.getCountryName();
                    if (value!=null && !"".equals(value))
                        newparty.setCountry(value);
                    value=newPartyEntityDlg.getStateName();
                    if (value!=null && !"".equals(value))
                        newparty.setState(value);
                    value=newPartyEntityDlg.getZipCode();
                    if (value!=null && !"".equals(value))
                        newparty.setZIP(value);

                    registry.commit();
                    JOptionPane.showMessageDialog(this.m_registryViewable.getModule().getAppDriver().getMainFrameWindow(),this.m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("registry.newParty.success.message"),this.m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("registry.newParty.success.title"),JOptionPane.INFORMATION_MESSAGE);
                }
                catch (KAONException e) {
                    registry.rollback();
                    m_registryViewable.getModule().getAppDriver().displayErrorNotification(e);
                }
            }
        }
       protected Object createObject(Registry registry,String entityName) {
            return null;
        }
     }

    /**
     * The element source for the terms.
     */
    protected static class TermsElementSource extends JElementSource {
        /** The table model displaying the results of the search. */
        protected TermsTableModel m_termsTableModel;
        /** The table displaying the elements filtered by <code>m_Filter</code>.*/
        protected JTable m_termsTable;
        /** The cache for the synonyms and their descriptions. */
        protected HashMap m_termTexts;

        public TermsElementSource(RegistryViewable registryViewable, String name, String displayName) {
             super( registryViewable, name, displayName );

             m_termTexts=new HashMap(17);
             // create the list for the ontologies
             m_termsTableModel=new TermsTableModel(registryViewable.getModule().getAppDriver().getLocalizationManager());
             m_termsTable=new SmartTable(m_termsTableModel) {
                 protected JTableHeader createDefaultTableHeader() {
                     return new SortableTableHeader(columnModel);
                 }
             };
             m_termsTable.getTableHeader().addPropertyChangeListener(new PropertyChangeListener() {
                 public void propertyChange(PropertyChangeEvent event) {
                     if (SortableTableHeader.SORTED_COLUMN_PROPERTY.equals(event.getPropertyName())||SortableTableHeader.SORT_ASCENDING_PROPERTY.equals(event.getPropertyName()))
                         updateSorting();
                 }
             });
             m_termsTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
             for (Enumeration e = m_termsTable.getColumnModel().getColumns();e.hasMoreElements();) {
                 ((TableColumn)e.nextElement()).setPreferredWidth(150);
             }
             m_termsTable.doLayout();
        }
        /**
        * Updates the sorting of the model.
        */
        protected void updateSorting() {
            SortableTableHeader header=(SortableTableHeader)m_termsTable.getTableHeader();
            if (header.getSortedColumn()!=null)
                m_termsTableModel.sortModel(header.getSortedColumn().getModelIndex(),header.getSortAscending());
        }
        /**
         * Returns the component of this ElementSource. If <code>null</code> is returned, <code>ExtElementSelector</code> will manage
         * the display and selection of the elements (with a <code>JList</code>).
         *
         * @return                      the component may be <code>null</code>
         */
        public Component getComponent() {
            return m_termsTable;
        }
        /**
         * Returns the selected elements of this ElementSource. Only used if <code>getComponent()</code> does not
         * return <code>null</code>
         *
         * @return                      the collection of selected elements of this ElementSource (never <code>null</code>)
         */
        public Collection getSelectedElements(){
            ArrayList selectedterms=new ArrayList();
            ListSelectionModel selectionModel=m_termsTable.getSelectionModel();
            int startIndex=selectionModel.getMinSelectionIndex();
            int endIndex=selectionModel.getMaxSelectionIndex();
            if (startIndex>-1) {
                for (int i=startIndex;i<=endIndex;i++) {
                    Term curTerm=m_termsTableModel.getTermAt(i,0);
                    if (curTerm!=null)
                        selectedterms.add(curTerm);
                }
            }
            return selectedterms;
        }
        /**
         * Sets the filter for the elements of this ElementSource.
         *
         * @param filter                the filter string for the elements
         */
        public void setFilter(String filter) {
            try {
                m_Filter=filter;
                try {
                    m_registryViewable.getModule().getAppDriver().startWaitState();
                    Collection elements=getElements();
                    m_registryViewable.getModule().getAppDriver().startWaitState();
                    m_termsTableModel.setElements(elements);
                }
                finally {
                    this.m_registryViewable.getModule().getAppDriver().endWaitState();
                }
            }
            catch (KAONException error) {
                this.m_registryViewable.getModule().getAppDriver().displayErrorNotification(error);
            }
        }
        public String getKAONObjectText(Object object) throws KAONException {
            Term term=(Term)object;
            String termtext=(String)m_termTexts.get(term);

            if (termtext==null) {
                //create the list of synonyms for the term
                StringBuffer buffer=new StringBuffer();
                Iterator iterator=term.getTermSynonyms(KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en")).iterator();
                while (iterator.hasNext()) {
                    String synonym=(String)iterator.next();
                    buffer.append(synonym);
                    if (iterator.hasNext())
                        buffer.append(", ");
                }
                //append the term definition
                String termdefinition=term.getTermDefinition();
                if (termdefinition!=null) {
                    if (buffer.length() > 0)
                        buffer.append(" (");
                    buffer.append(termdefinition);
                    if (buffer.length() > termdefinition.length())
                        buffer.append(")");
                }
                termtext=buffer.toString();
                m_termTexts.put(term,termtext);
            }
            return termtext;
        }
        public Collection getKAONElements(String filter) throws KAONException {
            Registry registry=this.m_registryViewable.getRegistry();
            Collection result=new HashSet();
            Iterator iterator=registry.getTerms(filter);
            while (iterator.hasNext()) {
                Term curterm=(Term)iterator.next();
                result.add(curterm);
            }
            return result;
        }
        protected String getAddNewElementID() {
            return "registry.newTerm";
        }
        /**
         * Called when the user requests a new element to be added.
         *
         * @param newElementText                  a proposal for the text of the new element
         */
        public void addNewElement(String newElementText) {
            NewTermEntityDlg newTermEntityDlg=new NewTermEntityDlg((RegistryModule)this.m_registryViewable.getModule(),getAddNewElementID());
            newTermEntityDlg.setSynonymLabel(newElementText);
            newTermEntityDlg.setVisible(true);
            if (newTermEntityDlg.isConfirmed()) {
                Registry registry=this.m_registryViewable.getRegistry();
                try {
                    Term term=registry.getTerm(registry.getRegistryOIModelInstance().createNewURI());
                    term.create();
                    String value=newTermEntityDlg.getTermDefinition();
                    if (value!=null && !"".equals(value))
                        term.setTermDefinition(value);

                    HashMap existingsynonyms=new HashMap(7);
                    Iterator iterator=registry.getExistingSynonyms(newTermEntityDlg.getSynonymLabels().iterator());
                    while (iterator.hasNext()) {
                        Synonym synonym=(Synonym)iterator.next();
                        existingsynonyms.put(synonym.getValue(),synonym);
                    }

                    iterator=newTermEntityDlg.getSynonymLabels().iterator();
                    while (iterator.hasNext()) {
                        String synonymlabel=(String)iterator.next();
                        Synonym synonym=(Synonym)existingsynonyms.get(synonymlabel);
                        if (synonym==null) {
                            synonym=registry.getSynonym(registry.getRegistryOIModelInstance().createNewURI());
                            synonym.create();
                            synonym.setValue(synonymlabel);
                            synonym.setLanguageURI(KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en"));
                        }
                        synonym.addReferencedTerm(term);
                    }
                    registry.commit();

                    JOptionPane.showMessageDialog(this.m_registryViewable.getModule().getAppDriver().getMainFrameWindow(),this.m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("registry.newTerm.success.message"),this.m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("registry.newTerm.success.title"),JOptionPane.INFORMATION_MESSAGE);
                }
                catch (KAONException e) {
                    registry.rollback();
                    this.m_registryViewable.getModule().getAppDriver().displayErrorNotification(e);
                }
            }
        }
        protected Object createObject(Registry registry,String entityName) {
            return null;
        }
    }
    /**
     * The element source for the automatically extracted terms.
     */
    protected static class ExtractedTermsElementSource extends JElementSource {
        /** The table model displaying the results of the search. */
        protected TermsTableModel m_termsTableModel;
        /** The table displaying the elements filtered by <code>m_Filter</code>.*/
        protected JTable m_termsTable;
        /** The cache for the synonyms and their descriptions. */
        protected HashMap m_termTexts;
        /** the list of the extracted terms */
        protected ArrayList m_extractedTerms;
        /** the currently used OIModel */
        protected OIModel m_oimodel;

        public ExtractedTermsElementSource(RegistryViewable registryViewable, String name, String displayName) {
             super( registryViewable, name, displayName );

             m_termTexts=new HashMap(17);
             // create the list for the ontologies
             m_termsTableModel=new TermsTableModel(registryViewable.getModule().getAppDriver().getLocalizationManager());
             m_termsTable=new SmartTable(m_termsTableModel) {
                 protected JTableHeader createDefaultTableHeader() {
                     return new SortableTableHeader(columnModel);
                 }
             };
             m_termsTable.getTableHeader().addPropertyChangeListener(new PropertyChangeListener() {
                 public void propertyChange(PropertyChangeEvent event) {
                     if (SortableTableHeader.SORTED_COLUMN_PROPERTY.equals(event.getPropertyName())||SortableTableHeader.SORT_ASCENDING_PROPERTY.equals(event.getPropertyName()))
                         updateSorting();
                 }
             });
             m_termsTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
             for (Enumeration e = m_termsTable.getColumnModel().getColumns();e.hasMoreElements();) {
                 ((TableColumn)e.nextElement()).setPreferredWidth(150);
             }
             m_termsTable.doLayout();
        }
        /**
        * Updates the sorting of the model.
        */
        protected void updateSorting() {
            SortableTableHeader header=(SortableTableHeader)m_termsTable.getTableHeader();
            if (header.getSortedColumn()!=null)
                m_termsTableModel.sortModel(header.getSortedColumn().getModelIndex(),header.getSortAscending());
        }
        /**
         * Returns the component of this ElementSource. If <code>null</code> is returned, <code>ExtElementSelector</code> will manage
         * the display and selection of the elements (with a <code>JList</code>).
         *
         * @return                      the component may be <code>null</code>
         */
        public Component getComponent() {
            return m_termsTable;
        }
        /**
         * Returns the selected elements of this ElementSource. Only used if <code>getComponent()</code> does not
         * return <code>null</code>
         *
         * @return                      the collection of selected elements of this ElementSource (never <code>null</code>)
         */
        public Collection getSelectedElements(){
            ArrayList selectedterms=new ArrayList();
            ListSelectionModel selectionModel=m_termsTable.getSelectionModel();
            int startIndex=selectionModel.getMinSelectionIndex();
            int endIndex=selectionModel.getMaxSelectionIndex();
            if (startIndex>-1) {
                for (int i=startIndex;i<=endIndex;i++) {
                    Term curTerm=m_termsTableModel.getTermAt(i,0);
                    if (curTerm!=null)
                        selectedterms.add(curTerm);
                }
            }
            return selectedterms;
        }
        /**
         * Sets the filter for the elements of this ElementSource.
         *
         * @param filter                the filter string for the elements
         */
        public void setFilter(String filter) {
            try {
                this.m_registryViewable.getModule().getAppDriver().startWaitState();

                // set the filter and initiate the thread reading all elements
                m_Filter=filter;

                if (m_oimodel!=this.m_registryViewable.getOIModel() || m_extractedTerms==null) {
                    OIModel oimodel=this.m_registryViewable.getOIModel();
                    if (oimodel!=null) {
                        // warn that this operation might take long if it has not been initiated yet
                        String message=this.m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("registry.OIModelInfo.ExtractTerms.ConfirmMessage");
                        String title=this.m_registryViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("registry.OIModelInfo.ExtractTerms.ConfirmTitle");
                        int result=JOptionPane.showConfirmDialog(this.m_registryViewable.getModule().getAppDriver().getMainFrameWindow(),message,title,JOptionPane.YES_NO_OPTION);

                        if (result==JOptionPane.YES_OPTION) {
                            try {
                                // load all concepts to match them against the synonyms
                                Concept concept=oimodel.getConcept(KAONVocabularyAdaptor.INSTANCE.getRoot());
                                matchingTerms(concept.getAllSubConcepts(),oimodel);
                            }
                            catch (KAONException e) {
                                this.m_registryViewable.getModule().getAppDriver().displayErrorNotification(e);
                            }
                        }
                    }
                }
                else if (m_oimodel!=null && m_extractedTerms!=null) {
                    setElements(m_extractedTerms,m_oimodel);
                }
                else
                    clear();
            }
            finally {
                this.m_registryViewable.getModule().getAppDriver().endWaitState();
            }
        }
        /**
         * Match the terms in the registry against the lsexical layer of the give set of concepts.
         *
         * @param concepts              set of concepts to match against the terms (should never be <code>null</code>)
         * @param matchedTerms          list to add the found terms to
         */
        void matchingTerms(final Set concepts, final OIModel oimodel) {
            // do the matching within a new thread
            final RegistryViewable viewable=this.m_registryViewable;
            final ArrayList matchedTerms=new ArrayList();

            ProgressDlg progressDlg=new ProgressDlg((AbstractModule )viewable.getModule(),"registry.OIModelInfo.ExtractTerms.matchTerms",true);
            progressDlg.executeAction(new ProgressDlg.AsynchronousAction() {
                public void run(final ProgressDlg progressDlg) throws Exception {
                    int stepsperformed=1;
                    int totalsteps=concepts.size();
                    HashMap stringentries=new HashMap(27);
                    oimodel.loadObjects(concepts,OIModel.LOAD_CONCEPT_ALL|OIModel.LOAD_LEXICON);
                    Iterator conceptsiterator=concepts.iterator();
                    while(conceptsiterator.hasNext()) {
                        Concept concept=(Concept)conceptsiterator.next();
                        Iterator lexicalEntries=concept.getLexicalEntries().iterator();
                        while(lexicalEntries.hasNext()) {
                            Iterator attributeValues=((LexicalEntry)lexicalEntries.next()).getAttributeValues().values().iterator();
                            while(attributeValues.hasNext()) {
                                Object object=attributeValues.next();
                                if (object instanceof String){
                                    stringentries.put(((String)object).toLowerCase(),((String)object).toLowerCase());
                                }

                            }
                        }
                        progressDlg.updateProgress(1,2,stepsperformed++,totalsteps);
                    }

                    if (!stringentries.isEmpty()) {
                        Iterator iterator=viewable.getRegistry().getMatchingTerms(stringentries.values().iterator());
                        while(iterator.hasNext())
                            matchedTerms.add(iterator.next());
                    }
                    progressDlg.updateProgress(2,2,0,1);
                    setElements(matchedTerms,oimodel);
                }
                public void completion(ProgressDlg progressDlg,boolean success) {
                    if (success) {
                        viewable.finishRegistryProcessing();
                    }
                    else {
                        clear();
                        viewable.finishRegistryProcessing();
                    }
                }
            });
        }
        /**
         * Clears the current values of this element source.
         */
         public void clear() {
            m_extractedTerms=null;
            m_oimodel=null;
            try {
                m_termsTableModel.setElements(m_extractedTerms);
            }
            catch(Exception ignore) {
            }
        }
        /**
         * Sets the new elements to this source.
         *
         * @param elements                    list of new elements for this source
         */
        protected void setElements(ArrayList elements,OIModel oimodel) {

            try {
                this.m_registryViewable.getModule().getAppDriver().startWaitState();

                // filter the elements
                ArrayList filteredElements=new ArrayList();
                Iterator terms=elements.iterator();
                while(terms.hasNext()) {
                    Term term=(Term)terms.next();
                    if ("".equals(m_Filter))
                        filteredElements.add(term);
                    else {
                        try {
                            Iterator synonyms=term.getTermSynonyms(KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en")).iterator();
                            while (synonyms.hasNext()) {
                                String synonym=(String)synonyms.next();
                                if (synonym.equals(m_Filter))
                                    filteredElements.add(term);
                            }
                        }
                        catch (KAONException e) {
                        }
                    }
                }

                try {
                    m_termsTableModel.setElements(filteredElements);
                    m_oimodel=oimodel;
                    m_extractedTerms=elements;
                }
                catch (KAONException e) {
                    this.m_registryViewable.getModule().getAppDriver().displayErrorNotification(e);
                }
            }
            finally {
                this.m_registryViewable.getModule().getAppDriver().endWaitState();
            }
        }
         /**
         * Returns the elements of this ElementSource.
         *
         * @return                      the collection of elements of this ElementSource (never <code>null</code>)
         */
        public Collection getElements() {
            return m_extractedTerms;
        }

        public String getKAONObjectText(Object object) throws KAONException {
            Term term=(Term)object;
            String termtext=(String)m_termTexts.get(term);

            if (termtext==null) {
                //create the list of synonyms for the term
                StringBuffer buffer=new StringBuffer();
                Iterator iterator=term.getTermSynonyms(KAONVocabularyAdaptor.INSTANCE.getLanguageURI("en")).iterator();
                while (iterator.hasNext()) {
                    String synonym=(String)iterator.next();
                    buffer.append(synonym);
                    if (iterator.hasNext())
                        buffer.append(", ");
                }
                //append the term definition
                String termdefinition=term.getTermDefinition();
                if (termdefinition!=null) {
                    if (buffer.length() > 0)
                        buffer.append(" (");
                    buffer.append(termdefinition);
                    if (buffer.length() > termdefinition.length())
                        buffer.append(")");
                }
                termtext=buffer.toString();
                m_termTexts.put(term,termtext);
            }
            return termtext;
        }
        public Collection getKAONElements(String filter) {
            return new ArrayList();
        }
        protected String getAddNewElementID() {
            return "registry.newTerm";
        }
        /**
         * Called when the user requests a new element to be added.
         *
         * @param newElementText                  a proposal for the text of the new element
         */
        public void addNewElement(String newElementText) {
            return;
        }
        protected Object createObject(Registry registry,String entityName) {
            return null;
        }
    }
}
