package de.fzi.wim.registry.gui.ui;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.awt.Insets;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.Component;
import javax.swing.JTabbedPane;
import javax.swing.JPanel;
import javax.swing.JList;
import javax.swing.JTextField;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JScrollPane;
import javax.swing.AbstractListModel;

import javax.swing.ListSelectionModel;
import de.fzi.wim.guibase.tables.*;
import javax.swing.table.AbstractTableModel;

/**
 * A control that allows selecting elements from another component.
 */
public class ExtElementSelector extends JPanel {
    /** The tabbed pane displaying the sources for the elements. */
    protected JTabbedPane m_sourcesTabbedPane;
    /** The list of selected elements. */
    protected List m_selectedElements;
    /** The model for the selected elements. */
    protected SelectedElementsTableModel m_selectedElementsModel;
    /** The list showing selected elements. */
    protected SmartTable m_selectedElementsList;

    /**
     * Creates an instance of this class.
     *
     * @param addSelectionButton        the button for adding elements to selection
     * @param removeSelectionButton     the button for removing elements from selection
	 * @param selectionCategoryName     the name displayed in the table header for the category
	 * @param selectionValueName        the name displayed in the table header for the value
     */
    public ExtElementSelector(JButton addSelectionButton, JButton removeSelectionButton, String selectionCategoryName, String selectionValueName) {
        super(new GridBagLayout());
        m_selectedElements=new ArrayList();
        m_selectedElementsModel=new SelectedElementsTableModel( selectionCategoryName, selectionValueName );
        m_selectedElementsList=new SmartTable(m_selectedElementsModel);
		m_sourcesTabbedPane = new JTabbedPane();

        addSelectionButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (m_sourcesTabbedPane.getTabCount()>0) {
                    addSelectedElements(((ElementSourcePanel)m_sourcesTabbedPane.getSelectedComponent()).getSelectedElements());
                }
            }
        });
        removeSelectionButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
				ListSelectionModel selectionModel=m_selectedElementsList.getSelectionModel();
				int startIndex=selectionModel.getMinSelectionIndex();
				int endIndex=selectionModel.getMaxSelectionIndex();
				if (startIndex>-1) {
					List selection=new ArrayList();
					for (int i=startIndex;i<=endIndex;i++) {
						selection.add( m_selectedElements.get( i ) );
					}
					removeSelectedElements( selection );
				}
            }
        });

        GridBagConstraints gbc=new GridBagConstraints();
        gbc.insets = new Insets(2,2,2,2);
        gbc.ipadx = 0; gbc.ipady = 0;
		gbc.gridwidth = 1;gbc.gridheight = 1;
        gbc.weightx = 100;gbc.weighty = 100;

		//add the tabbed pane
        gbc.weightx = 100;gbc.weighty = 20;
		gbc.gridy = 1;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.BOTH;
        add( m_sourcesTabbedPane, gbc );
		//add the buttons
		JPanel buttonpanel=new JPanel( new FlowLayout( FlowLayout.LEFT, 10, 0 ) );
		//add the add button
        buttonpanel.add( addSelectionButton );
		//add the remove button
        buttonpanel.add( removeSelectionButton );
		gbc.weightx = 100;gbc.weighty = 10;
        gbc.gridy = 2;
        gbc.anchor = GridBagConstraints.CENTER;
		gbc.fill = GridBagConstraints.NONE;
        add( buttonpanel, gbc );
		//add the selected elements list
		gbc.weightx = 100;gbc.weighty = 70;
        gbc.gridy = 3;
        gbc.anchor = GridBagConstraints.CENTER;
		gbc.fill = GridBagConstraints.BOTH;
        JScrollPane selectedscrollPane=new JScrollPane(m_selectedElementsList);
        selectedscrollPane.getViewport().setBackground(m_selectedElementsList.getBackground());
        selectedscrollPane.getViewport().setPreferredSize(new Dimension(100,50));
        add( selectedscrollPane, gbc );
    }

    /**
     * Adds a new element source to the selector.
     *
     * @param elementSource           an ElementSource object
     * @param searchLabel             the label displayed on top of the selection list
     * @param searchButton            the button for searching elements (may be <code>null</code>)
     * @param addButton               the button for adding elements (may be <code>null</code>)
     */
    public void addElementSource(ElementSource elementSource,JLabel searchLabel,JButton searchButton,JButton addButton) {
		m_sourcesTabbedPane.addTab(elementSource.getDisplayName(), null, new ElementSourcePanel(elementSource,searchLabel,searchButton,addButton), elementSource.getDisplayName());
		if ( m_sourcesTabbedPane.getSelectedIndex() == -1 )
			m_sourcesTabbedPane.setSelectedIndex(0);
    }

    /**
     * Sets the set of selected elements.
     *
     * @param objects                   the selected elements
     */
    protected void setSelectedElements(Object[] objects) {
        int startSize=m_selectedElements.size();
        m_selectedElements.clear();
        m_selectedElementsModel.fireTableRowsDeleted(0,startSize-1);
        for (int i=0;i<objects.length;i++)
            m_selectedElements.add(objects[i]);
        m_selectedElementsModel.fireTableRowsInserted(0,m_selectedElements.size()-1);
    }
    /**
     * The array of selected elements.
     *
	 * @param sourceName                name of the source
     * @return                          the array of selected elements (never <code>null</code>)
     */
    public List getSelectedElements( String sourceName ) {
		List returnlist=new ArrayList();

		Iterator iterator=m_selectedElements.iterator();
		while ( iterator.hasNext() ) {
			ListElement element=(ListElement)iterator.next();

			if ( element.getSource().getName().equals(sourceName) ) {
				returnlist.add( element.getElement()  );
			}
		}

        return returnlist;
    }
    /**
     * Adds the objects to the selection.
     *
     * @param objects                   objects added to the selection
     */
    protected void addSelectedElements(Collection objects) {
        int startIndex=m_selectedElements.size();
		Iterator iterator=objects.iterator();
		while ( iterator.hasNext() ) {
			Object curObject=iterator.next();
			if ( !m_selectedElements.contains( curObject ) )
				m_selectedElements.add( curObject );
		}
        if ( m_selectedElements.size() > 0 ) {
			m_selectedElementsModel.fireTableRowsInserted(startIndex,m_selectedElements.size()-1);
		}
    }
    /**
     * Removes the objects from the selection.
     *
     * @param objects                   objects removed from the selection
     */
    public void removeSelectedElements(List objects) {
		Iterator iterator=objects.iterator();
		while ( iterator.hasNext() ) {
            int index=m_selectedElements.indexOf( iterator.next() );
            if ( index!=-1 ) {
                m_selectedElements.remove( index );
                m_selectedElementsModel.fireTableRowsDeleted( index, index );
            }
		}
    }

    /**
     * The source for the element selector.
     */
    public interface ElementSource {
        /**
         * Returns the name of this element source.
         *
         * @return    name of the element source
         */
        String getName();
        /**
         * Returns the name of this element source that should be displayed in the list.
         *
         * @return    display name of the element source
         */
        String getDisplayName();
        /**
         * Returns the text for the given object.
         *
         * @param object                the object
         * @return                      the text for given object
         */
        String getObjectText(Object object);
        /**
         * Sets the filter for the elements of this ElementSource.
         *
         * @param filter                the filter string for the elements
         */
        void setFilter(String filter);
        /**
         * Returns the elements of this ElementSource.
         *
         * @return                      the collection of elements of this ElementSource (never <code>null</code>)
         */
        Collection getElements();
        /**
         * Returns the selected elements of this ElementSource. Only used if <code>getComponent()</code> does not
         * return <code>null</code>
         *
         * @return                      the collection of selected elements of this ElementSource (never <code>null</code>)
         */
        Collection getSelectedElements();
        /**
         * Called when the user requests a new element to be added.
		 *
		 * @param newElementText                  a proposal for the text of the new element
         */
        void addNewElement(String newElementText);
        /**
         * Returns the component of this ElementSource. If <code>null</code> is returned, <code>ExtElementSelector</code> will manage
         * the display and selection of the elements (with a <code>JList</code>).
         *
         * @return                      the component may be <code>null</code>
         */
        Component getComponent();
    }

    /**
     * Class to display the ElementSource, a search field and a search button
     */
    protected static class ElementSourcePanel extends JPanel {
        /** The list of the current source elements. */
        protected List m_sourceElements;
        /** The model for the source elements. */
        protected ListElementsModel m_sourceElementsModel;
        /** The component for the source elements */
        protected Component m_sourceElementsComponent;

        /** The textfield for entering the filter string */
        protected JTextField m_filterField;
        /** The attached element source */
		protected ElementSource m_source;

		/**
		 * Creates a new instance.
		 *
		 * @param source                source of the element
         * @param searchLabel           the label displayed on top of the selection list
         * @param searchButton          the button for searching elements (may be <code>null</code>)
         * @param addButton             the button for adding elements (may be <code>null</code>)
		 */
		public ElementSourcePanel(ElementSource source,JLabel searchLabel,final JButton searchButton,JButton addButton) {
            super(new GridBagLayout());

			m_source = source;

            m_filterField=new JTextField();
            m_filterField.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent evt) {
                    searchButton.doClick();
                }
            });

            if (searchButton!=null) {
                searchButton.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        m_source.setFilter(ElementSourcePanel.this.m_filterField.getText());
                        fillSourceList();
                    }
                });
            }

            if (addButton!=null) {
                addButton.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        m_source.setFilter(ElementSourcePanel.this.m_filterField.getText());
                        fillSourceList();
                        if (m_source.getElements().size()==0)
                            m_source.addNewElement(ElementSourcePanel.this.m_filterField.getText());
                        else
                            ElementSourcePanel.this.m_source.addNewElement("");
                    }
                });
            }

            //create the list only if the source doesn't provide any component
            m_sourceElementsComponent=source.getComponent();
            if (m_sourceElementsComponent==null) {
                m_sourceElements=new ArrayList();
                m_sourceElementsModel=new ExtElementSelector.ListElementsModel( m_sourceElements );

                m_sourceElementsComponent=new JList(m_sourceElementsModel);
            }

            GridBagConstraints gbc=new GridBagConstraints();
            gbc.insets = new Insets(5,5,5,5);
            gbc.ipadx = 0;gbc.ipady = 0;
            gbc.gridwidth = 1;gbc.gridheight = 1;
            gbc.weightx = 100;gbc.weighty = 100;
            gbc.gridx = 0;gbc.gridy = 0;
            //add search label
            if (searchButton!=null || addButton!=null) {
                gbc.gridwidth = 2;
            }
            gbc.weighty = 1;
            gbc.anchor = GridBagConstraints.SOUTHWEST;
            gbc.fill = GridBagConstraints.NONE;
            add(searchLabel,gbc);
            //add the filter field
            gbc.gridwidth = 1;
            gbc.weighty = 2;
            gbc.weightx = 100;
            if (searchButton!=null || addButton!=null) {
                gbc.weightx = 98;
            }
            gbc.gridy = 1;
            gbc.anchor = GridBagConstraints.CENTER;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            add(m_filterField,gbc);
            //add the search button
            if (searchButton!=null) {
                gbc.weightx = 2;
                gbc.gridx = 1;
                gbc.anchor = GridBagConstraints.EAST;
                gbc.fill = GridBagConstraints.HORIZONTAL;
                JPanel searchbuttonpanel=new JPanel(new GridBagLayout());
                add(searchbuttonpanel,gbc);

                GridBagConstraints buttongbc=new GridBagConstraints();
                buttongbc.insets = new Insets(0,2,0,0);
                buttongbc.ipadx = 0;gbc.ipady = 0;
                buttongbc.gridwidth = 1;gbc.gridheight = 1;
                buttongbc.weightx = 100;gbc.weighty = 100;
                buttongbc.gridx = 0;gbc.gridy = 0;
                buttongbc.anchor = GridBagConstraints.CENTER;
                buttongbc.fill = GridBagConstraints.HORIZONTAL;
                searchbuttonpanel.add(searchButton,buttongbc);
            }
            //add the SourceElementsComponent
            gbc.weighty = 97;
            gbc.weightx = 100;
            if (searchButton!=null || addButton!=null) {
                gbc.weightx = 98;
            }
            gbc.gridwidth = 1;
            if (addButton==null) {
                gbc.gridwidth = 2;
            }
            gbc.gridy = 2;
            gbc.gridx = 0;
            gbc.anchor = GridBagConstraints.CENTER;
            gbc.fill = GridBagConstraints.BOTH;
            if (m_sourceElementsComponent.getClass()!=JScrollPane.class) {
                JScrollPane sourcescrollPane=new JScrollPane(m_sourceElementsComponent);
                sourcescrollPane.getViewport().setBackground(m_sourceElementsComponent.getBackground());
                sourcescrollPane.getViewport().setPreferredSize(new Dimension(100,50));
                add(sourcescrollPane,gbc);
            }
            else
                add(m_sourceElementsComponent,gbc);
            //add the add button
            if (addButton!=null) {
                gbc.weightx = 2;
                gbc.gridx = 1;
                gbc.anchor = GridBagConstraints.NORTHEAST;
                gbc.fill = GridBagConstraints.HORIZONTAL;
                JPanel addbuttonpanel=new JPanel(new GridBagLayout());
                add(addbuttonpanel,gbc);

                GridBagConstraints buttongbc=new GridBagConstraints();
                buttongbc.insets = new Insets(0,2,0,0);
                buttongbc.ipadx = 0;gbc.ipady = 0;
                buttongbc.gridwidth = 1;gbc.gridheight = 1;
                buttongbc.weightx = 100;gbc.weighty = 100;
                buttongbc.gridx = 0;gbc.gridy = 0;
                buttongbc.anchor = GridBagConstraints.CENTER;
                buttongbc.fill = GridBagConstraints.HORIZONTAL;
                addbuttonpanel.add(addButton,buttongbc);
            }
		}

        /**
         * Fills the source list with elements from the supplied ElementSource.
         */
         protected void fillSourceList() {
             if (m_source.getComponent()==null) {
                 int startSize=m_sourceElements.size();
                 if (startSize > 0) {
                     m_sourceElements.clear();
                     m_sourceElementsModel.fireIntervalRemoved(0,startSize-1);
                 }

                 Iterator iterator=m_source.getElements().iterator();
                 while (iterator.hasNext()) {
                     m_sourceElements.add(new ExtElementSelector.ListElement(iterator.next(),m_source));
                 }

                 if (m_sourceElements.size() > 0) {
                     m_sourceElementsModel.fireIntervalAdded(0,m_sourceElements.size()-1);
                 }
             }
         }

        /**
         * Get the currently selected elements.
         *
         * @return                collection of selected elements
         */
         public Collection getSelectedElements() {
             List selection=new ArrayList();
             if (m_source.getComponent()!=null) {
                 Iterator iterator=m_source.getSelectedElements().iterator();

                 while (iterator.hasNext()) {
                     selection.add(new ExtElementSelector.ListElement(iterator.next(),m_source));
                 }
             }
             else {
                 int[] selectedIndices=((JList)m_sourceElementsComponent).getSelectedIndices();
                 for (int i=0;i<selectedIndices.length;i++) {
                     selection.add(m_sourceElements.get(selectedIndices[i]));
                 }
             }

             return selection;
         }
	}

    /**
     * Class to store an element and its source.
     */
    protected static class ListElement {
		protected Object m_Element;
		protected ElementSource m_source;

		/**
		 * Creates a new instance.
		 *
		 * @param element       the element
		 * @param source        source of the element
		 */
		public ListElement( Object element, ElementSource source ) {
			m_Element = element;
			m_source = source;
		}

		public String getObjectText() {
			return m_source.getObjectText( m_Element );
		}

		public Object getElement() {
			return m_Element;
		}

		public ElementSource getSource() {
			return m_source;
		}
	}

	/**
     * The table model for the list of source elements.
     */
    protected static class ListElementsModel extends AbstractListModel {
		protected List m_sourceList;
		public ListElementsModel( List sourceList ) {
			m_sourceList = sourceList;
		}
        public Object getElementAt(int index) {
			ListElement element=(ListElement)m_sourceList.get(index);
			return element.getObjectText( );
         }
        public int getSize() {
            return m_sourceList.size();
        }
        public void fireIntervalAdded(int index0,int index1) {
            super.fireIntervalAdded(this,index0,index1);
        }
        public void fireIntervalRemoved(int index0,int index1) {
            super.fireIntervalRemoved(this,index0,index1);
        }
    }

	/**
	 * A table model for the selection list
	 */
	 public class SelectedElementsTableModel extends AbstractTableModel {
		 String m_selectionCategoryName;
		 String m_selectionValueName;
		 /**
		  * Creates an instance of this class.
		  *
		  * @param selectionCategoryName     the name displayed in the table header for the category
		  * @param selectionValueName        the name displayed in the table header for the value
		  */
		 public SelectedElementsTableModel( String selectionCategoryName, String selectionValueName ) {
			 m_selectionCategoryName=selectionCategoryName;
			 m_selectionValueName=selectionValueName;
		 }
		 /**
		  * Refreshes the model with supplied elements.
		  *
		  * @param elements                      elements to add to the model
		  */
		 public void setElements(Collection elements) {
			 m_selectedElements.clear();
			 Iterator iterator=elements.iterator();
			 while (iterator.hasNext()) {
				 m_selectedElements.add((ListElement)iterator.next());
			 }
			 fireTableDataChanged();
		 }
		 /**
		  * Returns the class of the given column.
		  *
		  * @param columnIndex                       the index of the column
		  * @return                                  the class of supplied column
		  */
		 public Class getColumnClass(int columnIndex) {
			 switch (columnIndex) {
				 case 0:
				 default:
				 return String.class;
			 }
		 }
		 /**
		  * Returns the number of columns.
		  *
		  * @return                                  the number of columns
		  */
		 public int getColumnCount() {
			 return 2;
		 }

		 /**
		  * Returns the name of given column.
		  *
		  * @param columnIndex                       the index of the column
		  * @return                                  the name of the given column
		  */
		 public String getColumnName(int columnIndex) {
			 switch (columnIndex) {
				 case 0:
				     return m_selectionCategoryName;
				 case 1:
				     return m_selectionValueName;
			 }
			 return "";
		 }

		 /**
		  * Returns the number of rows.
		  *
		  * @return                                  the number of rows
		  */
		 public int getRowCount() {
			 return m_selectedElements.size();
		 }

		 /**
		  * Returns a value at given coordinates.
		  *
		  * @param rowIndex                          the row
		  * @param columnIndex                       the column
		  * @return                                  the value
		  */
		 public Object getValueAt(int rowIndex,int columnIndex) {
			 ListElement element=(ListElement)m_selectedElements.get( rowIndex );

			 switch (columnIndex) {
                case 0:
                    return element.getSource().getDisplayName();
                case 1:
                    return element.getObjectText();
                default:
                    return "";
			 }
		 }

		 /**
		  * Checks whether the cell is editable.
		  *
		  * @param rowIndex                          the row
		  * @param columnIndex                       the column
		  * @return                                  <code>true</code> if the cell is editable
		  */
		 public boolean isCellEditable(int rowIndex,int columnIndex) {
			 return false;
		 }
		 /**
		  * Sets the value at given coordinates.
		  *
		  * @param aValue                            the value
		  * @param rowIndex                          the row
		  * @param columnIndex                       the column
		  */
		 public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
		 }
		 /**
		  * Sorts the model.
		  *
		  * @param columnIndex                       the index of the column that is sorted
		  * @param ascending                         if <code>true</code>, the column is sorted in the ascending order
		  */
		 public void sortModel(int columnIndex,boolean ascending) {
		 }
	 }
}