package de.fzi.wim.registry.gui.ui;

import java.util.List;
import java.util.ArrayList;
import java.awt.Insets;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JList;
import javax.swing.JButton;
import javax.swing.JScrollPane;
import javax.swing.AbstractListModel;

/**
 * A control that allows selecting elements from another component.
 */
public class ElementSelector extends JPanel {
    /** The source for the elements. */
    protected ElementSource m_elementSource;
    /** The list of selected elements. */
    protected List m_selectedElements;
    /** The model for the selected elements. */
    protected SelectedElementsModel m_selectedElementsModel;
    /** The list showing selected elements. */
    protected JList m_selectedElementsList;

    /**
     * Creates an instance of this class.
     *
     * @param elementSource             the source for the elements
     * @param addSelectionIcon          the icon for adding elements to selection
     * @param removeSeletionIcon        the icon for removing elements from selection
     * @param newElementIcon            the icon for adding a new element
     */
    public ElementSelector(ElementSource elementSource,Icon addSelectionIcon,Icon removeSeletionIcon,Icon newElementIcon) {
        super(new GridBagLayout());
        m_elementSource=elementSource;
        m_selectedElements=new ArrayList();
        m_selectedElementsModel=new SelectedElementsModel();
        m_selectedElementsList=new JList(m_selectedElementsModel);
        JButton addSelection=new JButton(addSelectionIcon);
        addSelection.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                Object[] selection=m_elementSource.getSelectedObjects();
                addSelectedElements(selection);
            }
        });
        JButton removeSelection=new JButton(removeSeletionIcon);
        removeSelection.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                int[] selectedIndices=m_selectedElementsList.getSelectedIndices();
                List selection=new ArrayList();
                for (int i=0;i<selectedIndices.length;i++) {
                    selection.add(m_selectedElements.get(selectedIndices[i]));
                    removeSelectedElements(selection.toArray());
                }
            }
        });
        JButton newElement=new JButton(newElementIcon);
        newElement.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                m_elementSource.addNewElement();
            }
        });
        GridBagConstraints gbc=new GridBagConstraints(0,0,3,1,1.0,0.75f,GridBagConstraints.CENTER,GridBagConstraints.BOTH,new Insets(0,0,0,0),-1,0);
        add(m_elementSource.getComponent(),gbc);
        gbc.gridy=1;
        gbc.weightx=0.33f;
        gbc.weighty=0.0f;
        gbc.gridwidth=1;
        add(addSelection,gbc);
        gbc.gridx=1;
        add(removeSelection,gbc);
        gbc.gridx=2;
        add(newElement,gbc);
        gbc.gridx=0;
        gbc.gridy=2;
        gbc.weightx=1.0f;
        gbc.weighty=0.25f;
        gbc.gridwidth=3;
        JScrollPane scrollPane=new JScrollPane(m_selectedElementsList);
        scrollPane.getViewport().setPreferredSize(new Dimension(100,50));
        add(scrollPane,gbc);
    }
    /**
     * Sets the set of selected elements.
     *
     * @param objects                   the selected elements
     */
    public void setSelectedElements(Object[] objects) {
        int startSize=m_selectedElements.size();
        m_selectedElements.clear();
        m_selectedElementsModel.fireIntervalRemoved(0,startSize-1);
        for (int i=0;i<objects.length;i++)
            m_selectedElements.add(objects[i]);
        m_selectedElementsModel.fireIntervalAdded(0,m_selectedElements.size()-1);
    }
    /**
     * The array of selected elements.
     *
     * @return                          the array of selected elements
     */
    public Object[] getSelectedElements() {
        return m_selectedElements.toArray();
    }
    /**
     * Adds the objects to the selection.
     *
     * @param objects                   objects added to the selection
     */
    public void addSelectedElements(Object[] objects) {
        int startIndex=m_selectedElements.size();
        for (int i=0;i<objects.length;i++)
            if (!m_selectedElements.contains(objects[i]))
                m_selectedElements.add(objects[i]);
        m_selectedElementsModel.fireIntervalAdded(startIndex,m_selectedElements.size()-1);
    }
    /**
     * Removes the objects from the selection.
     *
     * @param objects                   objects removed from the selection
     */
    public void removeSelectedElements(Object[] objects) {
        for (int i=0;i<objects.length;i++) {
            int index=m_selectedElements.indexOf(objects[i]);
            if (index!=-1) {
                m_selectedElements.remove(index);
                m_selectedElementsModel.fireIntervalRemoved(index,index);
            }
        }
    }

    /**
     * The source for the element selector.
     */
    public interface ElementSource {
        /**
         * Returns the current selection of the element source.
         *
         * @return                      currently selected objects (never <code>null</code>)
         */
        Object[] getSelectedObjects();
        /**
         * Returns the text for the given object.
         *
         * @param object                the object
         * @return                      the text for given object
         */
        String getObjectText(Object object);
        /**
         * Returns the component for the source.
         *
         * @return                      the component for the source
         */
        JComponent getComponent();
        /**
         * Called when the user requests a new element to be added.
         */
        void addNewElement();
    }

    /**
     * The table model displaying the selected elements.
     */
    protected class SelectedElementsModel extends AbstractListModel {
         public Object getElementAt(int index) {
            Object object=m_selectedElements.get(index);
            return m_elementSource.getObjectText(object);
         }
        public int getSize() {
            return m_selectedElements.size();
        }
        public void fireIntervalAdded(int index0,int index1) {
            super.fireIntervalAdded(this,index0,index1);
        }
        public void fireIntervalRemoved(int index0,int index1) {
            super.fireIntervalRemoved(this,index0,index1);
        }
    }
}