package de.fzi.wim.registry.gui.ui;

import java.util.Date;
import java.util.Calendar;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import javax.swing.JPanel;
import javax.swing.JCheckBox;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import javax.swing.event.ChangeListener;
import javax.swing.event.ChangeEvent;

/**
 * This pane that can be used to present and enter a date range or a single date.
 */
public class DateSelectorPane extends JPanel {
    /** The checkbox for the start date. */
    protected JCheckBox m_startCheckBox;
    /** The spinner component of the start date. */
    protected JSpinner m_startDateSpinner;
    /** The data model of the start date. */
    protected SpinnerDateModel m_startDateModel;
    /** The checkbox for the end date. */
    protected JCheckBox m_endCheckBox;
    /** The spinner component of the end date. */
    protected JSpinner m_endDateSpinner;
    /** The data model of the end date. */
    protected SpinnerDateModel m_endDateModel;

    /**
     * Initializes the layout of this pane.
     */
    protected void initLayout( ) {
        // the starting date spinner with its label
        GridBagConstraints gbc=new GridBagConstraints();
        gbc.gridx = 0;gbc.gridy = 0;
        gbc.gridwidth = 1; gbc.gridheight = 1;
        gbc.weightx = 10; gbc.weighty = 100;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.insets = new Insets(0,0,0,0);
        gbc.ipadx = 0; gbc.ipady = 0;

        // add the label
        gbc.weightx = 10;
		gbc.anchor = GridBagConstraints.EAST;
        add( m_startCheckBox, gbc );
        // and the spinner
        gbc.weightx = 90;
		gbc.gridx = 1;
        gbc.anchor = GridBagConstraints.CENTER;
        add( m_startDateSpinner, gbc );

        if ( m_endDateModel != null ) {
			// the ending date spinner with its label
			gbc.weightx = 10;
			gbc.gridx = 0;gbc.gridy = 1;
			gbc.anchor = GridBagConstraints.EAST;
			add( m_endCheckBox, gbc );
			// and the spinner
			gbc.weightx = 90;
			gbc.gridx = 1;
			gbc.anchor = GridBagConstraints.CENTER;
			add( m_endDateSpinner, gbc );
		}
    }

    /**
     * Adjusts the dates according to their selection states, so that the starting date is always before the ending date.
     */
    protected void adjustDates() {
		if ( m_endDateModel != null ) {
			Date endDate = getEndDate();
			Date startDate = getStartDate();

			if ( null != endDate && null != startDate ) {
				if ( startDate.after( endDate ) ) {
					setStartDate( (Date)m_startDateModel.getPreviousValue() );
				}
				else if ( endDate.before( startDate ) ) {
					setEndDate( (Date)m_endDateModel.getNextValue() );
				}
			}
        }
    }

    /**
     * Creates an instance of this pane.
     *
     * @param startCheckBox            the checkbox for the starting date
     * @param endCheckBox              the checkbox for the ending date ( may be <code>null</code> if only a single date should be entered)
     * @param startDate                the starting date to be shown initially
     * @param endDate                  the ending date to be shown initially
     */
    protected void createPane(JCheckBox startCheckBox, JCheckBox endCheckBox, Date startDate, Date endDate) {
        // create the date models
        m_startDateModel = new SpinnerDateModel();
        m_startDateModel.setCalendarField( Calendar.DAY_OF_WEEK );

        if ( endCheckBox != null ) {
			m_endDateModel = new SpinnerDateModel();
			m_endDateModel.setCalendarField( Calendar.DAY_OF_WEEK );
		}

        // create the spinners
        m_startDateSpinner = new JSpinner();
        m_startDateSpinner.setModel( m_startDateModel );
        DateFormat dateFormat=DateFormat.getDateInstance(DateFormat.SHORT);
        SimpleDateFormat defaultDateFormat;
        if (dateFormat instanceof SimpleDateFormat)
            defaultDateFormat=(SimpleDateFormat)dateFormat;
        else
            defaultDateFormat=new SimpleDateFormat();
		JSpinner.DateEditor spinnereditor = new JSpinner.DateEditor(m_startDateSpinner, defaultDateFormat.toPattern());
        m_startDateSpinner.setEditor(spinnereditor);
        m_startDateSpinner.setEnabled(null == startDate ? false : true);

		if ( m_endDateModel != null ) {
			m_endDateSpinner = new JSpinner();
			m_endDateSpinner.setModel( m_endDateModel );
			spinnereditor = new JSpinner.DateEditor(m_endDateSpinner, defaultDateFormat.toPattern());
			m_endDateSpinner.setEditor(spinnereditor);
			m_endDateSpinner.setEnabled(null == endDate ? false : true);
		}

        // handle the checkboxes
        m_startCheckBox = startCheckBox;
        m_endCheckBox = endCheckBox;

        m_startCheckBox.addChangeListener( new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                DateSelectorPane.this.m_startDateSpinner.setEnabled( DateSelectorPane.this.getStartDateSelected() );
                adjustDates();
            }
        } );

		if ( m_endCheckBox != null ) {
			m_endCheckBox.addChangeListener( new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					DateSelectorPane.this.m_endDateSpinner.setEnabled( DateSelectorPane.this.getEndDateSelected() );
					adjustDates();
				}
			} );
		}

        // fill the date models
        m_startDateModel.addChangeListener( new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
                adjustDates();
            }
        } );

		if ( m_endDateModel != null ) {
			m_endDateModel.addChangeListener( new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					adjustDates();
				}
			} );
		}

        setStartDate( startDate );
        setEndDate( endDate );

        initLayout( );
    }

    /**
     * Creates an instance of this pane.
     *
     * @param startCheckBox            the checkbox for the starting date
     * @param endCheckBox              the checkbox for the ending date  ( may be <code>null</code> if only a single date should be entered)
     * @param startDate                the starting date to be shown initially
     * @param endDate                  the ending date to be shown initially
     */
    public DateSelectorPane(JCheckBox startCheckBox, JCheckBox endCheckBox, Date startDate, Date endDate) {
        super(new GridBagLayout());
        createPane( startCheckBox, endCheckBox, startDate, endDate );
    }

    /**
     * Creates an instance of this pane.
     *
     * @param startLabel               the string to be displayed as label for the starting date
     * @param endLabel                 the string to be displayed as label for the ending date  ( may be <code>null</code> if only a single date should be entered)
     * @param startDate                the starting date to be shown initially
     * @param endDate                  the ending date to be shown initially
     */
    public DateSelectorPane(String startLabel, String endLabel, Date startDate, Date endDate) {
        super(new GridBagLayout());
		if ( endLabel != null ) {
			createPane( new JCheckBox(startLabel), new JCheckBox(endLabel), startDate, endDate );
		}
		else {
			createPane( new JCheckBox(startLabel), null, startDate, endDate );
		}
    }

    /**
     * Returns the current starting date
     *
     * @return                     the current starting date
     */
    public Date getStartDate() {
        if ( getStartDateSelected() == false )
            return null;

        return m_startDateModel.getDate();
    }

    /**
     * Returns the current ending date.
	 * If only a single date field is displayed, this value will be always the same as <code>getStartDate()</code>.
     *
     * @return                     the current ending date
     */
    public Date getEndDate() {
		if ( m_endDateModel == null ) {
			return getStartDate();
		}

        if ( getEndDateSelected() == false )
            return null;

        return m_endDateModel.getDate();
    }

    /**
     * Sets the starting date
     *
     * @param     value             the new starting date
     */
    public void setStartDate(Date value) {
        if ( null == value ) {
            m_startCheckBox.setSelected( false );
        }
        else {
            m_startDateModel.setValue( value );
        }
    }

    /**
     * Sets the ending date
     *
     * @param     value             the new ending date
     */
    public void setEndDate(Date value) {
		if ( m_endDateModel != null ) {
			if ( null == value ) {
				m_endCheckBox.setSelected( false );
			}
			else {
				m_endDateModel.setValue( value );
			}
		}
    }

    /**
     * Sets the selection state of the starting date
     *
     * @param     bselected             the new selection state for the starting date
     */
    public void setStartDateSelected(boolean bselected) {
        m_startCheckBox.setSelected( bselected );
    }

    /**
     * Sets the selection state of the ending date
     *
     * @param     bselected             the new selection state for the ending date
     */
    public void setEndDateSelected(boolean bselected) {
		if ( m_endCheckBox != null ) {
			m_endCheckBox.setSelected( bselected );
		}
    }

    /**
     * Gets the selection state of the starting date
     *
     * @return                     the current selection state for the starting date
     */
    public boolean getStartDateSelected() {
        return m_startCheckBox.isSelected();
    }

    /**
     * Gets the selection state of the ending date
     *
     * @return                     the current selection state for the ending date
     */
    public boolean getEndDateSelected() {
		if ( m_endCheckBox == null )
			return false;
        return m_endCheckBox.isSelected();
    }
}
