package de.fzi.wim.registry.api;

import java.util.Set;
import java.util.HashSet;
import java.util.Collection;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Encapsulates information about a term in the registry.
 */
public class Term extends AbstractRegistryEntity {

    /**
     * Creates an instance of this class.
     *
     * @param registry                  the registry
     * @param term                      the instance representing the term
     */
    public Term(Registry registry,Instance term) {
        super(registry,term);
    }
    /**
     * Returns the URI of the type of this object.
     *
     * @return                                      the URI of the object type
     */
    protected String getTypeURI() {
        return OMO_Term;
    }
    /**
     * Returns the synonyms to a term.
     *
     * @param languageURI                           the URI of the synonym's language
     * @return                                      the gloss of the term
     * @throws KAONException                        thrown if there is a problem
     */
    public Collection getTermSynonyms(String languageURI) throws KAONException {
        Iterator iterator=m_entity.getLexicalEntries(KAONVocabularyAdaptor.INSTANCE.getSynonym()).iterator();
        Set result=new HashSet();
        while (iterator.hasNext()) {
            LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
            if (languageURI.equals(lexicalEntry.getInLanguage()))
                result.add(lexicalEntry.getValue());
        }
        return result;
    }
    /**
     * Returns the definition of a term.
     *
     * @return                          the definition of the term
     * @throws KAONException            thrown if there is a problem
     */
    public String getTermDefinition() throws KAONException {
        Property termDefinition=m_entity.getOIModel().getProperty(OMO_termDefinition);
        Set result=m_entity.getAllFromPropertyValues(termDefinition);
        if (!result.isEmpty()) {
            Object object=result.iterator().next();
            if (object instanceof String)
                return (String)object;
        }
        return null;
    }
    /**
     * Sets the definition of the term.
     *
     * @param termDefinition            the definition of the term
     * @throws KAONException            thrown if there is a problem
     */
    public void setTermDefinition(String termDefinition) throws KAONException {
        setPropertyValue(OMO_termDefinition,termDefinition);
    }
    /**
     * Returns the set of hypernym terms of this term.
     *
     * @return                          the set of hypernym terms
     * @throws KAONException            thrown if there is a problem
     */
    public Set getHypernymTerms() throws KAONException {
        Set result=new HashSet();
        Property property=m_entity.getOIModel().getProperty(OMO_hyponymOf);
        Iterator hypernymTermInstances=m_entity.getFromPropertyValues(property).iterator();
        while (hypernymTermInstances.hasNext()) {
            Instance instance=(Instance)hypernymTermInstances.next();
            result.add(m_registry.getTerm(instance.getURI()));
        }
        return result;
    }
    /**
     * Makes this term the hyponym term of given term.
     *
     * @param hypernymTerm              the hypernym term
     * @throws KAONException            thrown if there is a problem
     */
    public void addHyponymOf(Term hypernymTerm) throws KAONException {
        addPropertyValue(OMO_hyponymOf,hypernymTerm.getEntity());
    }
    /**
     * Removes this term from the hyponym term set of given term.
     *
     * @param hypernymTerm              the hypernym term
     * @throws KAONException            thrown if there is a problem
     */
    public void removeHyponymOf(Term hypernymTerm) throws KAONException {
        removePropertyValue(OMO_hyponymOf,hypernymTerm.getEntity());
    }
}
