package de.fzi.wim.registry.api;

import java.util.Date;
import java.util.Map;
import java.util.List;
import java.util.LinkedList;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Collection;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;
import edu.unika.aifb.kaon.defaultevolution.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Implements access to the registry and manipulation of the OMO.
 */
public class Registry implements RegistryConstants {

    static {
        try {
            KAONManager.registerWellKnownRDFOIModel(Registry.class.getResource("res/kaon-omo.xml").toString());
        }
        catch (KAONException shouldntHappen) {
            throw new RuntimeException(shouldntHappen);
        }
    }

    /** OI-model for registry. */
    protected OIModel m_registryOIModelInstance;
    /** The evoltion strategy. */
    protected EvolutionStrategy m_evolutionStrategy;
    /** List of pending changes to the registry. */
    protected List m_pendingChanges;
    /** The map of the OIModelInfo objects. */
    protected Map m_oimodelInfos;
    /** The map of the ApplicationField objects. */
    protected Map m_applicationFields;
    /** The map of the Project objects. */
    protected Map m_projects;
    /** The map of the Term objects. */
    protected Map m_terms;
    /** The map of the Synonym objects. */
    protected Map m_synonyms;
    /** The map of the Person objects. */
    protected Map m_persons;
    /** The map of the Organization objects. */
    protected Map m_organizations;

    /**
     * Creates an instance of this class.
     *
     * @param registryOIModelInstance               the OI-model that is the instance of the registry OI-model
     * @throws KAONException                        thrown if there is a problem
     */
    public Registry(OIModel registryOIModelInstance) throws KAONException {
        if (registryOIModelInstance.getIncludedOIModel(OMO_URI)==null) {
            OIModel registryOIModel=registryOIModelInstance.getKAONConnection().openOIModelLogical(OMO_URI);
            registryOIModelInstance.applyChanges(Collections.singletonList(new AddIncludedOIModel(registryOIModel)));
        }
        m_pendingChanges=new LinkedList();
        m_registryOIModelInstance=registryOIModelInstance;
        m_evolutionStrategy=new EvolutionStrategyImpl(m_registryOIModelInstance);
        m_oimodelInfos=new WeakValueHashMap();
        m_applicationFields=new WeakValueHashMap();
        m_projects=new WeakValueHashMap();
        m_terms=new WeakValueHashMap();
        m_persons=new WeakValueHashMap();
        m_organizations=new WeakValueHashMap();
        m_synonyms=new WeakValueHashMap();
    }
    /**
     * Returns the OI-model that holds OMO instances (the registry contents).
     *
     * @return                                      the OI-model instance of the reigstry
     */
    public OIModel getRegistryOIModelInstance() {
        return m_registryOIModelInstance;
    }
    /**
     * Adds a pending change.
     *
     * @param changeEvent                           the event
     */
    public void applyChange(ChangeEvent changeEvent) {
        m_pendingChanges.add(changeEvent);
    }
    /**
     * Commits all pending changes.
     *
     * @throws KAONException                        thrown if there is an error commiting the changes
     */
    public void commit() throws KAONException {
        if (!m_pendingChanges.isEmpty()) {
            List requestedChanges=m_evolutionStrategy.computeRequestedChanges(m_pendingChanges);
            m_registryOIModelInstance.applyChanges(requestedChanges);
            m_pendingChanges.clear();
            m_registryOIModelInstance.save();
        }
    }
    /**
     * Undoes all pending changes.
     */
    public void rollback() {
        m_pendingChanges.clear();
    }
    /**
     * Returns the OIModelInfo object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the OI-model into object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public OIModelInfo getOIModelInfo(String uri) throws KAONException {
        OIModelInfo oimodelInfo=(OIModelInfo)m_oimodelInfos.get(uri);
        if (oimodelInfo==null) {
            Instance oimodelInstance=m_registryOIModelInstance.getInstance(uri);
            oimodelInfo=new OIModelInfo(this,oimodelInstance);
            m_oimodelInfos.put(uri,oimodelInfo);
        }
        return oimodelInfo;
    }
    /**
     * Returns the OIModelInfo object for given Instance.
     *
     * @param oimodelInstance                       the OI-model for given instance
     * @return                                      the OI-model into object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    protected OIModelInfo getOIModelInfo(Instance oimodelInstance) throws KAONException {
        OIModelInfo oimodelInfo=(OIModelInfo)m_oimodelInfos.get(oimodelInstance.getURI());
        if (oimodelInfo==null) {
            oimodelInfo=new OIModelInfo(this,oimodelInstance);
            m_oimodelInfos.put(oimodelInstance.getURI(),oimodelInfo);
        }
        return oimodelInfo;
    }
    /**
     * Returns a new OI-model info.
     *
     * @return                                      a new OI-model info
     * @throws KAONException                        thrown if there is an error
     */
    public OIModelInfo newOIModelInfo() throws KAONException {
        return getOIModelInfo(m_registryOIModelInstance.createNewURI());
    }
    /**
     * Returns all ApplicationFields that match the given filter string.
     *
     * @param filter                                the filter used to search for ApplicationFields
     * @return                                      iterator of all found ApplicationFields
     * @throws KAONException                        thrown if there is an error
     */
    public Iterator getApplicationFields(String filter) throws KAONException {
        Set result=new HashSet();
        Iterator iterator=filterEntitiesOnSingleProperty(OMO_ApplicationField,OMO_applicationFieldName,filter);
        while (iterator.hasNext()) {
            ApplicationField applicationField=getApplicationField((String)iterator.next());
            result.add(applicationField);
        }
        return result.iterator();
    }
    /**
     * Returns the ApplicationField object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the application field object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public ApplicationField getApplicationField(String uri) throws KAONException {
        ApplicationField applicationField=(ApplicationField)m_applicationFields.get(uri);
        if (applicationField==null) {
            Instance applicationFieldInstance=m_registryOIModelInstance.getInstance(uri);
            applicationField=new ApplicationField(this,applicationFieldInstance);
            m_applicationFields.put(uri,applicationField);
        }
        return applicationField;
    }
    /**
     * Returns all projects that match the given filter string.
     *
     * @param filter                                the filter used to search for projects
     * @return                                      iterator of all found Projects
     * @throws KAONException                        thrown if there is an error
     */
    public Iterator getProjects(String filter) throws KAONException {
        Set result=new HashSet();
        Iterator iterator=filterEntitiesOnSingleProperty(OMO_Project,OMO_projectName,filter);
        while (iterator.hasNext()) {
            Project project=getProject((String)iterator.next());
            result.add(project);
        }
        return result.iterator();
    }
    /**
     * Returns the Project object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the project object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public Project getProject(String uri) throws KAONException {
        Project project=(Project)m_projects.get(uri);
        if (project==null) {
            Instance projectInstance=m_registryOIModelInstance.getInstance(uri);
            project=new Project(this,projectInstance);
            m_projects.put(uri,project);
        }
        return project;
    }
    /**
     * Returns the terms for given synonym.
     *
     * @param synonym                               the synonym to search for
     * @return                                      iterator of all Terms with synonym equal to the specified word
     * @throws KAONException                        thrown if there is an error
     */
    public Iterator getTerms(String synonym) throws KAONException {
        Set result=new HashSet();
        Iterator iterator=m_registryOIModelInstance.executeQuery("HAS_LEXICON('Synonym','"+synonym+"')").iterator();
        while (iterator.hasNext()) {
            Instance termInstance=(Instance)iterator.next();
            Term term=getTerm(termInstance.getURI());
            result.add(term);
        }
        return result.iterator();
    }
    /**
     * Returns the Term object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the term object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public Term getTerm(String uri) throws KAONException {
        Term term=(Term)m_terms.get(uri);
        if (term==null) {
            Instance termInstance=m_registryOIModelInstance.getInstance(uri);
            term=new Term(this,termInstance);
            m_terms.put(uri,term);
        }
        return term;
    }
    /**
     * Returns a all terms having one of the given values as synonym.
     *
     * @param synonymNames                          names to search synonyms for
     * @return                                      iterator of all existing synonyms
     * @throws KAONException                        thrown if there is an error
     */
    public Iterator getMatchingTerms(Iterator synonymNames) throws KAONException {
        Set result=new HashSet();
        if (synonymNames.hasNext()) {
            StringBuffer queryBuffer=new StringBuffer();
            while (synonymNames.hasNext()) {
                queryBuffer.append("HAS_LEXICON('Synonym','");
                queryBuffer.append((String)synonymNames.next());
                queryBuffer.append("')");
                if (synonymNames.hasNext())
                    queryBuffer.append(" OR ");
            }
            Iterator iterator=m_registryOIModelInstance.executeQuery(queryBuffer.toString()).iterator();
            while (iterator.hasNext()) {
                Instance termInstance=(Instance)iterator.next();
                Term term=getTerm(termInstance.getURI());
                result.add(term);
            }
        }
        return result.iterator();
    }
    /**
     * Returns the existing synonyms for a given set of synonym names.
     *
     * @param synonymNames                          names to search synonyms for
     * @return                                      iterator of all existing synonyms
     * @throws KAONException                        thrown if there is an error
     */
    public Iterator getExistingSynonyms(Iterator synonymNames) throws KAONException {
        Set result=new HashSet();
        if (synonymNames.hasNext()) {
            StringBuffer queryBuffer=new StringBuffer();
            queryBuffer.append("[");
            queryBuffer.append(KAONVocabularyAdaptor.INSTANCE.getSynonym());
            queryBuffer.append("] AND SOME(<");
            queryBuffer.append(KAONVocabularyAdaptor.INSTANCE.getValue());
            queryBuffer.append(">={");
            while (synonymNames.hasNext()) {
                queryBuffer.append("'");
                queryBuffer.append((String)synonymNames.next());
                queryBuffer.append("'");
                if (synonymNames.hasNext())
                    queryBuffer.append(",");
            }
            queryBuffer.append("})");
            Iterator iterator=m_registryOIModelInstance.executeQuery(queryBuffer.toString()).iterator();
            while (iterator.hasNext()) {
                Instance synonymInstance=(Instance)iterator.next();
                Synonym synonym=getSynonym(synonymInstance.getURI());
                result.add(synonym);
            }
        }
        return result.iterator();
    }
    /**
     * Returns the Synonym object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the synonym object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public Synonym getSynonym(String uri) throws KAONException {
        Synonym synonym=(Synonym)m_synonyms.get(uri);
        if (synonym==null) {
            Instance synonymInstance=m_registryOIModelInstance.getInstance(uri);
            synonym=new Synonym(this,synonymInstance);
            m_synonyms.put(uri,synonym);
        }
        return synonym;
    }
    /**
     * Returns all Parties (Persons or Organizations) that match the given filter string.
     *
     * @param filter                                the filter used to search for Parties
     * @return                                      iterator of all found Parties
     * @throws KAONException                        thrown if there is an error
     */
    public Iterator getParties(String filter) throws KAONException {
        Set result=new HashSet();
        Iterator iterator=m_registryOIModelInstance.executeQuery("["+OMO_Party+"] AND (SOME(<"+OMO_personName+">='"+filter+"') OR SOME(<"+OMO_organizationName+">='"+filter+"'))").iterator();
        while (iterator.hasNext()) {
            Instance partyInstance=(Instance)iterator.next();
            Party party=getParty(partyInstance.getURI());
            result.add(party);
        }
        return result.iterator();
    }
    /**
     * Returns the Party object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the party object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public Party getParty(String uri) throws KAONException {
        Person person=(Person)m_persons.get(uri);
        if (person!=null)
            return person;
        Organization organization=(Organization)m_organizations.get(uri);
        if (organization!=null)
            return organization;
        Instance partyInstance=m_registryOIModelInstance.getInstance(uri);
        Iterator parentConcepts=partyInstance.getParentConcepts().iterator();
        if (parentConcepts.hasNext()) {
            Concept parentConcept=(Concept)parentConcepts.next();
            if (OMO_Organization.equals(parentConcept.getURI()))
                return getOrganization(uri);
            if (OMO_Person.equals(parentConcept.getURI()))
                return getPerson(uri);
        }
        return null;
    }
    /**
     * Returns the Person object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the person object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public Person getPerson(String uri) throws KAONException {
        Person person=(Person)m_persons.get(uri);
        if (person==null) {
            Instance personInstance=m_registryOIModelInstance.getInstance(uri);
            person=new Person(this,personInstance);
            m_persons.put(uri,person);
        }
        return person;
    }
    /**
     * Returns the Organization object for given URI.
     *
     * @param uri                                   the URI
     * @return                                      the organization object for given URI
     * @throws KAONException                        thrown if there is an error
     */
    public Organization getOrganization(String uri) throws KAONException {
        Organization organization=(Organization)m_organizations.get(uri);
        if (organization==null) {
            Instance organizationInstance=m_registryOIModelInstance.getInstance(uri);
            organization=new Organization(this,organizationInstance);
            m_organizations.put(uri,organization);
        }
        return organization;
    }
    /**
     * Creates a new entry for the OI-model.
     *
     * @param oimodel                               the OI-model
     * @return                                      the info object for the OI-model
     * @throws KAONException                        thrown if there is an error
     */
    public OIModelInfo addOIModel(OIModel oimodel) throws KAONException {
        String newURI=m_registryOIModelInstance.createNewURI();
        OIModelInfo oimodelInfo=getOIModelInfo(newURI);
        oimodelInfo.setLogicalURI(oimodel.getLogicalURI());
        oimodelInfo.setPhysicalURI(oimodel.getPhysicalURI());
        oimodelInfo.create();
        return oimodelInfo;
    }
    /**
     * Searches the registry and returns a OIModelInfo object for given logical URI.
     *
     * @param logicalURI                            the logical URI of the OI-model
     * @return                                      the <code>OIModelInfo</code> object, or <code>null</code> if such object doesn't exist
     * @throws KAONException                        thrown if there is an error
     */
    public OIModelInfo locateByLogicalURI(String logicalURI) throws KAONException {
        Iterator iterator=m_registryOIModelInstance.executeQuery("["+RegistryConstants.OMO_OIModel+"] AND SOME(<"+RegistryConstants.OMO_logicalURI+">='"+logicalURI+"')").iterator();
        if (iterator.hasNext()) {
            Instance oimodelInstance=(Instance)iterator.next();
            return getOIModelInfo(oimodelInstance);
        }
        else
            return null;
    }
    /**
     * Searches the registry.
     *
     * @param oimodelName                           the name of the OI-model (may be <code>null</code>)
     * @param startingDate                          the starting date for the creation of the OI-model (may be <code>null</code> if it should not be included in the search)
     * @param endingDate                            the ending date of the creation of the OI-model (may be <code>null</code> if it should not be included in the search)
     * @param appliedInFields                       the list of application fields (may be <code>null</code>)
     * @param projects                              the list of projects (may be <code>null</code>)
     * @param creators                              the list of creators (may be <code>null</code>)
     * @param terms                                 the list of terms (may be <code>null</code>)
     * @return                                      the collection of OIModelInfo objects matching the criteria (may be empty)
     * @throws KAONException                        thrown if there is an error
     */
    public Collection search(String oimodelName,Date startingDate,Date endingDate,Collection appliedInFields,Collection projects,Collection creators,Collection terms) throws KAONException {
        StringBuffer buffer=new StringBuffer();
        buffer.append('[');
        buffer.append(OMO_OIModel);
        buffer.append(']');
        if (oimodelName!=null) {
            buffer.append(" AND SOME(<");
            buffer.append(OMO_oimodelName);
            buffer.append(">='");
            buffer.append(oimodelName);
            buffer.append("')");
        }
        if (projects!=null && !projects.isEmpty()) {
            buffer.append(" AND SOME(<");
            buffer.append(OMO_createdInProject);
            buffer.append(">=");
            appendInstanceList(projects,buffer);
            buffer.append(')');
        }
        if (appliedInFields!=null && !appliedInFields.isEmpty()) {
            buffer.append(" AND SOME(<");
            buffer.append(OMO_appliedInField);
            buffer.append(">=");
            appendInstanceList(appliedInFields,buffer);
            buffer.append(')');
        }
        if (creators!=null && !creators.isEmpty()) {
            buffer.append(" AND SOME(<");
            buffer.append(OMO_oimodelCreator);
            buffer.append(">=");
            appendInstanceList(creators,buffer);
            buffer.append(')');
        }
        if (terms!=null && !terms.isEmpty()) {
            buffer.append(" AND SOME(<");
            buffer.append(OMO_isAboutTerm);
            buffer.append(">=");
            appendInstanceList(terms,buffer);
            buffer.append(')');
        }
        Collection result=new LinkedList();
        String query=buffer.toString();
        Iterator iterator=m_registryOIModelInstance.executeQuery(query).iterator();
        while (iterator.hasNext()) {
            Instance instance=(Instance)iterator.next();
            OIModelInfo oimodelInfo=getOIModelInfo(instance);
            if (startingDate!=null || endingDate!=null) {
                Date creationDate=oimodelInfo.getCreationDate();
                if (creationDate==null)
                    continue;
                else {
                    if (startingDate!=null && startingDate.after(creationDate))
                        continue;
                    if (endingDate!=null && endingDate.before(creationDate))
                        continue;
                }
            }
            result.add(oimodelInfo);
        }
        return result;
    }
    /**
     * Appends a list of instances from given collection.
     *
     * @param collection                            the collection for which the list of instances is created
     * @param buffer                                the buffer to which the list is appended
     * @throws KAONException                        thrown if there is an error
     */
    protected void appendInstanceList(Collection collection,StringBuffer buffer) throws KAONException {
        buffer.append("{");
        Iterator iterator=collection.iterator();
        while (iterator.hasNext()) {
            String entityURI=((AbstractRegistryEntity)iterator.next()).getEntity().getURI();
            buffer.append('!');
            buffer.append(entityURI);
            buffer.append('!');
            if (iterator.hasNext())
                buffer.append(',');
        }
        buffer.append("}");
    }
    /**
     * Returns all entities of the specified type that match the given filter string in the specified property.
     *
     * @param typeURI                               the uri to the type of the entity being queried
     * @param propertyURI                           the uri to the property being searched
     * @param filter                                the filter used in the search
     * @return                                      iterator with uris to all found entities
     * @throws KAONException                        thrown if there is an error
     */
    protected Iterator filterEntitiesOnSingleProperty(String typeURI,String propertyURI,String filter) throws KAONException {
        Set result=new HashSet();
        Iterator iterator=m_registryOIModelInstance.executeQuery("["+typeURI+"] AND SOME(<"+propertyURI+">='"+filter+"')").iterator();
        while (iterator.hasNext()) {
            Instance instance=(Instance)iterator.next();
            result.add(instance.getURI());
        }
        return result.iterator();
    }
}
