package de.fzi.wim.registry.api;

import java.util.Date;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.text.ParseException;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Encapsulates information about an OI-model in the registry.
 */
public class OIModelInfo extends AbstractRegistryEntity {
    /** The date format. */
    protected static final DateFormat s_dateFormat=new SimpleDateFormat("dd/MM/yyyy");

    /**
     * Creates an instance of this class.
     *
     * @param registry                  the registry
     * @param oimodelInstance           the instance representing the OI-model
     */
    public OIModelInfo(Registry registry,Instance oimodelInstance) {
        super(registry,oimodelInstance);
    }
    /**
     * Returns the URI of the type of this object.
     *
     * @return                                      the URI of the object type
     */
    protected String getTypeURI() {
        return OMO_OIModel;
    }
    /**
     * Returns the name of the OI-model.
     *
     * @return                          the name of the OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public String getOIModelName() throws KAONException {
        return getLiteralPropertyValue(OMO_oimodelName);
    }
    /**
     * Sets the name of the OI-model.
     *
     * @param oimodelName               the name of the OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public void setOIModelName(String oimodelName) throws KAONException {
        setPropertyValue(OMO_oimodelName,oimodelName);
    }
    /**
     * Returns the date of the creation of the OI-model.
     *
     * @return                          the date of the creation of the OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public Date getCreationDate() throws KAONException {
        String creationDate=getLiteralPropertyValue(OMO_creationDate);
        if (creationDate==null)
            return null;
        else
            try {
                return s_dateFormat.parse(creationDate);
            }
            catch (ParseException error) {
                throw new KAONException("Invalid date format '"+creationDate+"'",error);
            }
    }
    /**
     * Sets the date of the creation of the OI-model.
     *
     * @param creationDate              the date of the creation of the OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public void setCreationDate(Date creationDate) throws KAONException {
        if (creationDate==null) {
            Object currentCreationDate=getLiteralPropertyValue(OMO_creationDate);
            if (currentCreationDate!=null)
                removePropertyValue(OMO_creationDate,null);
        }
        else
            setPropertyValue(OMO_creationDate,s_dateFormat.format(creationDate));
    }
    /**
     * Returns the logical URI of the OI-model.
     *
     * @return                          the logical URI of the OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public String getLogicalURI() throws KAONException {
        return getLiteralPropertyValue(OMO_logicalURI);
    }
    /**
     * Sets the logical URI of the OI-model.
     *
     * @param logicalURI                the logical URI
     * @throws KAONException            thrown if there is a problem
     */
    public void setLogicalURI(String logicalURI) throws KAONException {
        setPropertyValue(OMO_logicalURI,logicalURI);
    }
    /**
     * Returns the physical URI of the OI-model.
     *
     * @return                          the physical URI of the OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public String getPhysicalURI() throws KAONException {
        return getLiteralPropertyValue(OMO_physicalURI);
    }
    /**
     * Sets the physical URI of the OI-model.
     *
     * @param physicalURI               the physical URI
     * @throws KAONException            thrown if there is a problem
     */
    public void setPhysicalURI(String physicalURI) throws KAONException {
        setPropertyValue(OMO_physicalURI,physicalURI);
    }
    /**
     * Returns the set of application fields where this OI-model has been applied.
     *
     * @return                          the set of application fields where this OI-model has been applied
     * @throws KAONException            thrown if there is a problem
     */
    public Set getAppliedInField() throws KAONException {
        Set result=new HashSet();
        Property property=m_entity.getOIModel().getProperty(OMO_appliedInField);
        Iterator appliedInFieldInstances=m_entity.getFromPropertyValues(property).iterator();
        while (appliedInFieldInstances.hasNext()) {
            Instance instance=(Instance)appliedInFieldInstances.next();
            result.add(m_registry.getApplicationField(instance.getURI()));
        }
        return result;
    }
    /**
     * Adds the field to this field has been applied.
     *
     * @param applicationField          the application field
     * @throws KAONException            thrown if there is a problem
     */
    public void addAppliedInField(ApplicationField applicationField) throws KAONException {
        addPropertyValue(OMO_appliedInField,applicationField.getEntity());
    }
    /**
     * Removes the field from this field has been applied.
     *
     * @param applicationField          the application field
     * @throws KAONException            thrown if there is a problem
     */
    public void removeAppliedInField(ApplicationField applicationField) throws KAONException {
        removePropertyValue(OMO_appliedInField,applicationField.getEntity());
    }
    /**
     * Returns the set of projects where this OI-model has been created.
     *
     * @return                          the set of projects where this OI-model has been created
     * @throws KAONException            thrown if there is a problem
     */
    public Set getCreatedInProject() throws KAONException {
        Set result=new HashSet();
        Property property=m_entity.getOIModel().getProperty(OMO_createdInProject);
        Iterator createdInProjectInstances=m_entity.getFromPropertyValues(property).iterator();
        while (createdInProjectInstances.hasNext()) {
            Instance instance=(Instance)createdInProjectInstances.next();
            result.add(m_registry.getProject(instance.getURI()));
        }
        return result;
    }
    /**
     * Adds the project to the set of projects where this OI-model has been created.
     *
     * @param project                   the project
     * @throws KAONException            thrown if there is a problem
     */
    public void addProject(Project project) throws KAONException {
        addPropertyValue(OMO_createdInProject,project.getEntity());
    }
    /**
     * Removes the project from the set of projects where this OI-model has been created.
     *
     * @param project                   the project
     * @throws KAONException            thrown if there is a problem
     */
    public void removeProject(Project project) throws KAONException {
        removePropertyValue(OMO_createdInProject,project.getEntity());
    }
    /**
     * Returns the set of creator of this OI-model.
     *
     * @return                          the set of creators of this OI-model
     * @throws KAONException            thrown if there is a problem
     */
    public Set getCreators() throws KAONException {
        Set result=new HashSet();
        Property property=m_entity.getOIModel().getProperty(OMO_oimodelCreator);
        Iterator oimodelCreatorInstances=m_entity.getFromPropertyValues(property).iterator();
        while (oimodelCreatorInstances.hasNext()) {
            Instance instance=(Instance)oimodelCreatorInstances.next();
            result.add(m_registry.getParty(instance.getURI()));
        }
        return result;
    }
    /**
     * Adds the creator to the set of creators of this OI-model.
     *
     * @param creator                   the creator
     * @throws KAONException            thrown if there is a problem
     */
    public void addCreator(Party creator) throws KAONException {
        addPropertyValue(OMO_oimodelCreator,creator.getEntity());
    }
    /**
     * Removes the creator from the set of creators of this OI-model.
     *
     * @param creator                   the creator
     * @throws KAONException            thrown if there is a problem
     */
    public void removeCreator(Party creator) throws KAONException {
        removePropertyValue(OMO_oimodelCreator,creator.getEntity());
    }
    /**
     * Returns the set of terms where this OI-model contains.
     *
     * @return                          the set of terms that this OI-model contains
     * @throws KAONException            thrown if there is a problem
     */
    public Set getAboutTerms() throws KAONException {
        Set result=new HashSet();
        Property property=m_entity.getOIModel().getProperty(OMO_isAboutTerm);
        Iterator isAboutTermInstances=m_entity.getFromPropertyValues(property).iterator();
        while (isAboutTermInstances.hasNext()) {
            Instance instance=(Instance)isAboutTermInstances.next();
            result.add(m_registry.getTerm(instance.getURI()));
        }
        return result;
    }
    /**
     * Adds the term to the set of terms that this OI-model is about.
     *
     * @param term                      the term
     * @throws KAONException            thrown if there is a problem
     */
    public void addAboutTerm(Term term) throws KAONException {
        addPropertyValue(OMO_isAboutTerm,term.getEntity());
    }
    /**
     * Removes the term from the set of terms that this OI-model is about.
     *
     * @param term                      the term
     * @throws KAONException            thrown if there is a problem
     */
    public void removeAboutTerm(Term term) throws KAONException {
        removePropertyValue(OMO_isAboutTerm,term.getEntity());
    }
}
