package de.fzi.wim.registry.api;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * An abstract element of the registry.
 */
public abstract class AbstractRegistryEntity implements RegistryConstants {
    /** The registry. */
    protected Registry m_registry;
    /** The instance entity. */
    protected Instance m_entity;

    /**
     * Creates an instance of this class.
     *
     * @param registry                  the registry
     * @param entity                    the entity
     */
    public AbstractRegistryEntity(Registry registry,Instance entity) {
        m_registry=registry;
        m_entity=entity;
    }
    /**
     * Returns the instance of this entity.
     *
     * @return                          the instance of the entity
     */
    public Instance getEntity() {
        return m_entity;
    }
    /**
     * Adds a pending change.
     *
     * @param changeEvent                           the event
     */
    public void applyChange(ChangeEvent changeEvent) {
        m_registry.applyChange(changeEvent);
    }
    /**
     * Commits all pending changes.
     *
     * @throws KAONException                        thrown if there is an error commiting the changes
     */
    public void commit() throws KAONException {
        m_registry.commit();
    }
    /**
     * Returns <code>true</code> if this entity is part of the registry.
     *
     * @return                                      <code>true</code> if the entity is part of the registry
     * @throws KAONException                        thrown if there is an error
     */
    public boolean isInRegistry() throws KAONException {
        return m_entity.isInOIModel();
    }
    /**
     * Creates this object.
     *
     * @throws KAONException
     */
    public void create() throws KAONException {
        applyChange(new AddEntity(m_entity));
        Concept parentConcept=m_registry.getRegistryOIModelInstance().getConcept(getTypeURI());
        applyChange(new AddInstanceOf(parentConcept,m_entity));
    }
    /**
     * Deletes this object.
     */
    public void delete() {
        applyChange(new RemoveEntity(m_entity));
    }
    /**
     * Returns the literal value of the given property.
     *
     * @param propertyURI                           the property name
     * @param instance                              the instance
     * @return                                      the value of the property (or <code>null</code> if the value is not defined or not literal)
     * @throws KAONException                        thrown if there is an error
     */
    protected String getLiteralPropertyValue(String propertyURI,Instance instance) throws KAONException {
        Property property=instance.getOIModel().getProperty(propertyURI);
        Object value=instance.getFromPropertyValue(property);
        if (value instanceof String)
            return (String)value;
        else
            return null;
    }
    /**
     * Returns the literal value of the given property.
     *
     * @param propertyURI                           the property name
     * @return                                      the value of the property (or <code>null</code> if the value is not defined or not literal)
     * @throws KAONException                        thrown if there is an error
     */
    protected String getLiteralPropertyValue(String propertyURI) throws KAONException {
        return getLiteralPropertyValue(propertyURI,m_entity);
    }
    /**
     * Sets the value of the property.
     *
     * @param propertyURI                           the property name
     * @param instance                              the instance
     * @param value                                 the value of the property (or <code>null</code> if the value is be removed)
     * @throws KAONException                        thrown if there is an error
     */
    protected void setPropertyValue(String propertyURI,Instance instance,Object value) throws KAONException {
        Property property=instance.getOIModel().getProperty(propertyURI);
        PropertyInstance propertyInstance=instance.getOIModel().getPropertyInstance(property,instance,value);
        applyChange(new SetPropertyInstanceValue(propertyInstance));
    }
    /**
     * Sets the value of the property.
     *
     * @param propertyURI                           the property name
     * @param value                                 the value of the property (or <code>null</code> if the value is be removed)
     * @throws KAONException                        thrown if there is an error
     */
    protected void setPropertyValue(String propertyURI,Object value) throws KAONException {
        setPropertyValue(propertyURI,m_entity,value);
    }
    /**
     * Adds the value of the property.
     *
     * @param propertyURI                           the property name
     * @param instance                              the instance
     * @param value                                 the value of the property (or <code>null</code> if the value is be removed)
     * @throws KAONException                        thrown if there is an error
     */
    protected void addPropertyValue(String propertyURI,Instance instance,Object value) throws KAONException {
        Property property=instance.getOIModel().getProperty(propertyURI);
        PropertyInstance propertyInstance=instance.getOIModel().getPropertyInstance(property,instance,value);
        applyChange(new AddPropertyInstance(propertyInstance));
    }
    /**
     * Adds the value of the property.
     *
     * @param propertyURI                           the property name
     * @param value                                 the value of the property (or <code>null</code> if the value is be removed)
     * @throws KAONException                        thrown if there is an error
     */
    protected void addPropertyValue(String propertyURI,Object value) throws KAONException {
        addPropertyValue(propertyURI,m_entity,value);
    }
    /**
     * Removes the value of the property.
     *
     * @param propertyURI                           the property name
     * @param instance                              the instance
     * @param value                                 the value of the property (or <code>null</code> if the value is be removed)
     * @throws KAONException                        thrown if there is an error
     */
    protected void removePropertyValue(String propertyURI,Instance instance,Object value) throws KAONException {
        Property property=instance.getOIModel().getProperty(propertyURI);
        PropertyInstance propertyInstance=instance.getOIModel().getPropertyInstance(property,instance,value);
        applyChange(new RemovePropertyInstance(propertyInstance));
    }
    /**
     * Removes the value of the property.
     *
     * @param propertyURI                           the property name
     * @param value                                 the value of the property (or <code>null</code> if the value is be removed)
     * @throws KAONException                        thrown if there is an error
     */
    protected void removePropertyValue(String propertyURI,Object value) throws KAONException {
        removePropertyValue(propertyURI,m_entity,value);
    }
    /**
     * Returns the URI of the type of this object.
     *
     * @return                                      the URI of the object type
     */
    protected abstract String getTypeURI();
    /**
     * Checks whether two registry entities are equal.
     *
     * @param that                                  the other entity
     * @return                                      <code>true</code> if the registry entities are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof AbstractRegistryEntity))
            return false;
        AbstractRegistryEntity entity=(AbstractRegistryEntity)that;
        return m_entity.equals(entity.m_entity);
    }
    /**
     * Returns the hashcode.
     *
     * @return                                      the hashcode
     */
    public int hashCode() {
        return m_entity.hashCode();
    }
}
