package edu.unika.aifb.rdf.rdfserver.tools.loader;

import java.net.URL;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import org.xml.sax.SAXException;
import org.xml.sax.InputSource;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.syntax.*;
import edu.unika.aifb.rdf.api.util.*;
import edu.unika.aifb.rdf.mainmemory.*;

import edu.unika.aifb.rdf.rdfserver.client.RemoteModel;
import edu.unika.aifb.rdf.rdfserver.interfaces.RemoteConnection;

/**
 * Loads an RDF file to the KAON RDF Server.
 */
public class Main implements RDFConsumer {
    /** Name of the JBoss host (or <code>null</code> if JNDI is to be used. */
    protected String m_hostName;
    /** JBoss port. */
    protected int m_hostPort;
    /** Model alias. */
    protected String m_alias;
    /** RDF Model. */
    protected Model m_model;
    /** Node factory of the model. */
    protected NodeFactory m_nodeFactory;
    /** Number of statements. */
    protected int m_uncommitedStatements;
    /** Total number of statements imported. */
    protected int m_importedStatements;

    /**
     * Creates this importer.
     */
    public Main() {
    }
    /**
     * Called when model parsing is started.
     *
     * @param physicalURI           physical URI of the model
     */
    public void startModel(String physicalURI) throws SAXException {
        try {
            m_uncommitedStatements=0;
            m_importedStatements=0;
            RemoteConnection connection;
            if (m_hostName==null)
                connection=new RemoteConnection();
            else
                connection=new RemoteConnection(m_hostName,m_hostPort);
            String modelPhysicalURI="jboss://"+m_hostName+":"+m_hostPort+"/"+m_alias;
            try {
                m_model=new RemoteModel(connection.loadModel(m_alias),new NodeFactoryImpl(),modelPhysicalURI);
            }
            catch (Exception e) {
                m_model=new RemoteModel(connection.createModel(m_alias),new NodeFactoryImpl(),modelPhysicalURI);
            }
            m_model.setAutocommit(false);
            m_nodeFactory=m_model.getNodeFactory();
        }
        catch (Exception e) {
            throw new SAXException(e.getMessage());
        }
    }
    /**
     * Called when model parsing is finished.
     */
    public void endModel() throws SAXException {
        try {
            m_model.commit();
        }
        catch (ModelException e) {
            throw new SAXException(e.getMessage());
        }
        finally {
            m_model=null;
            m_nodeFactory=null;
        }
    }
    /**
     * Called when a statement with resource value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     */
    public void statementWithResourceValue(String subject,String predicate,String object) throws SAXException {
        try {
            m_model.add(m_nodeFactory.createStatement(m_nodeFactory.createResource(subject),m_nodeFactory.createResource(predicate),m_nodeFactory.createResource(object)));
            statementAdded();
        }
        catch (ModelException e) {
            throw new SAXException(e.getMessage());
        }
    }
    /**
     * Called when a statement with literal value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                literal object value
     * @param language              the language
     * @param datatype              the datatype
     */
    public void statementWithLiteralValue(String subject,String predicate,String object,String language,String datatype) throws SAXException {
        try {
            m_model.add(m_nodeFactory.createStatement(m_nodeFactory.createResource(subject),m_nodeFactory.createResource(predicate),m_nodeFactory.createLiteral(object)));
            statementAdded();
        }
        catch (ModelException e) {
            throw new SAXException(e.getMessage());
        }
    }
    /**
     * Receives the logical URI of the model.
     *
     * @param logicalURI            logical URI of the model
     */
    public void logicalURI(String logicalURI) {
    }
    /**
     * Receives the notification that the model being parsed includes another model with supplied logical URIs.
     *
     * @param logicalURI            logical URI of the model
     * @param physicalURI           physical URI of the model
     */
    public void includeModel(String logicalURI,String physicalURI) {
    }
    /**
     * Receives the notification that the attribute and its value has been parsed.
     *
     * @param key                   the key of the attribute
     * @param value                 the value of the attribute
     */
    public void addModelAttribte(String key,String value) {
    }
    /**
     * Called each time when a statement has been added.
     */
    protected void statementAdded() throws ModelException {
        m_importedStatements++;
        m_uncommitedStatements++;
        if (m_uncommitedStatements==100) {
            m_model.commit();
            m_uncommitedStatements=0;
            System.out.println("[statements imported:] "+m_importedStatements);
        }
    }
    /**
     * Performs import.
     *
     * @param rdfURL                URL of the RDF model
     * @param alias                 name of the model
     * @param hostName              name of the host
     * @param hostPort              port of the host
     */
    public void doImport(String rdfURL,String alias,String hostName,int hostPort) throws Exception {
        System.out.println("Importing RDF URL "+rdfURL+" to ontology named "+alias);
        if (hostName==null)
            System.out.println("Host will be connected from JNDI environment.");
        else
            System.out.println("Host for JBoss is "+hostName+" at port "+hostPort);
        m_alias=alias;
        m_hostName=hostName;
        m_hostPort=hostPort;
        URL url=new URL(rdfURL);
        InputSource inputSource=new InputSource(url.openStream());
        inputSource.setSystemId(url.toString());
        RDFParser parser=RDFManager.createParser();
        parser.parse(inputSource,this);
        System.out.println("Import done, "+m_importedStatements+" imported");
    }
    /**
     * Starts the application.
     *
     * @param args              command line arguments
     */
    public static void main(String[] args) throws Exception {
        BufferedReader in=new BufferedReader(new InputStreamReader(System.in));
        String rdfURL=null;
        if (args.length>=1)
            rdfURL=args[0];
        else {
            System.out.print("URL of the RDF file: ");
            rdfURL=in.readLine();
        }
        String alias=null;
        if (args.length>=2)
            alias=args[1];
        else {
            System.out.print("Model alias: ");
            alias=in.readLine();
        }
        String hostName=null;
        if (args.length>=3)
            hostName=args[2];
        else {
            System.out.print("Name of JBoss host (leave empty to get host from JNDI properties): ");
            hostName=in.readLine();
        }
        int hostPort=0;
        if (hostName!=null) {
            String portString;
            if (args.length>=3)
                portString=args[3];
            else {
                System.out.print("Port of JBoss at host (leave empty for 1099): ");
                portString=in.readLine();
            }
            if (portString==null || portString.length()==0)
                hostPort=1099;
            else
                hostPort=Integer.parseInt(portString);
        }
        if (rdfURL==null || alias==null) {
            System.out.println("Usage: edu.unika.aifb.kaon.rdfserver.tools.loader.DBLoader [<rdf-URL> [<alias> [<JBoss-host-name> [<JBoss-host-port>]]]]");
            System.exit(1);
        }
        Main loader=new Main();
        loader.doImport(rdfURL,alias,hostName,hostPort);
    }
}
