package edu.unika.aifb.rdf.rdfserver.tools.dbloader;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.sql.DriverManager;
import java.sql.Connection;
import java.sql.PreparedStatement;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.util.*;

/**
 * Loads a RDF file into a database.
 */
public class Main {
    /** Map of known resources. */
    protected Map m_resources;
    /** Map of known literals. */
    protected Map m_literals;
    /** Connection to the database. */
    protected Connection m_connection;
    /** Statement for inserting resources. */
    protected PreparedStatement m_insertResource;
    /** Statement for inserting literals. */
    protected PreparedStatement m_insertLiteral;
    /** Statement for inserting statements. */
    protected PreparedStatement m_insertStatement;

    /**
     * Creates this importer.
     */
    public Main() {
        m_resources=new HashMap();
        m_literals=new HashMap();
    }
    /**
     * Performs import.
     *
     * @param rdfName               URI of the RDF model
     * @param modelAlias            name of the model
     * @param defaultNamespace      default namespace
     * @param connectionString      JDBC connection string
     * @param driverClass           class of the driver
     * @exception Exception         thrown if there is an error
     */
    public void doImport(String rdfName,String modelAlias,String defaultNamespace,String connectionString,String driverClass) throws Exception {
        System.out.println("Importing RDF URL "+rdfName+" to ontology named "+modelAlias);
        System.out.println("JDBC URL: "+connectionString);
        System.out.println("JDBC driver class: "+driverClass);
        Model model=RDFManager.openModel(rdfName,null);
        int size=model.size();
        System.out.println("Parsing done, model size: "+size);
        Class.forName(driverClass);
        m_connection=DriverManager.getConnection(connectionString);
        m_connection.setAutoCommit(false);
        PreparedStatement insertModel=m_connection.prepareStatement("INSERT INTO model (alias,logicalURI) VALUES (?,?)");
        insertModel.setString(1,modelAlias);
        insertModel.setString(2,defaultNamespace);
        insertModel.addBatch();
        insertModel.executeBatch();
        insertModel.close();
        Iterator statements=model.iterator();
        int processed=0;
        m_insertStatement=m_connection.prepareStatement("INSERT INTO statement (id,modelKey,subjKey,predKey,objKey,literal) VALUES (?,?,?,?,?,?)");
        m_insertResource=m_connection.prepareStatement("INSERT INTO resource (id,label) VALUES (?,?)");
        m_insertLiteral=m_connection.prepareStatement("INSERT INTO literal (id,label) VALUES (?,?)");
        while (statements.hasNext()) {
            Statement statement=(Statement)statements.next();
            Resource subject=statement.subject();
            Resource predicate=statement.predicate();
            RDFNode object=statement.object();
            Integer subjectKey=getResource(subject);
            Integer predicateKey=getResource(predicate);
            Integer objectKey;
            if (object instanceof Resource)
                objectKey=getResource((Resource)object);
            else
                objectKey=getLiteral((Literal)object);
            m_insertStatement.setInt(1,processed);
            m_insertStatement.setString(2,modelAlias);
            m_insertStatement.setInt(3,subjectKey.intValue());
            m_insertStatement.setInt(4,predicateKey.intValue());
            m_insertStatement.setInt(5,objectKey.intValue());
            m_insertStatement.setBoolean(6,object instanceof Literal);
            m_insertStatement.addBatch();
            processed++;
            if ((processed % 100)==0) {
                System.out.println("Processed "+processed+" statements, "+((processed*100.0)/size)+" %");
                commit();
            }
        }
        commit();
        m_insertStatement.close();
        m_insertLiteral.close();
        m_insertResource.close();
        PreparedStatement insertCounter=m_connection.prepareStatement("INSERT pkcounter (counter,type) VALUES (?,?)");
        insertCounter.setInt(1,m_resources.size());
        insertCounter.setString(2,"resource");
        insertCounter.addBatch();
        insertCounter.setInt(1,m_literals.size());
        insertCounter.setString(2,"literal");
        insertCounter.addBatch();
        insertCounter.setInt(1,size);
        insertCounter.setString(2,"statement");
        insertCounter.addBatch();
        insertCounter.executeBatch();
        m_connection.commit();
        insertCounter.close();
        m_connection.close();
    }
    /**
     * Commits all updates.
     */
    protected void commit() throws Exception {
        m_insertLiteral.executeBatch();
        m_insertResource.executeBatch();
        m_insertStatement.executeBatch();
        m_connection.commit();
    }
    /**
     * Returns a resource.
     */
    protected Integer getResource(Resource resource) throws Exception {
        Integer result=(Integer)m_resources.get(resource);
        if (result==null) {
            result=new Integer(m_resources.size());
            m_resources.put(resource,result);
            m_insertResource.setInt(1,result.intValue());
            m_insertResource.setString(2,resource.getURI());
            m_insertResource.addBatch();
        }
        return result;
    }
    /**
     * Returns a literal.
     */
    protected Integer getLiteral(Literal literal) throws Exception {
        Integer result=(Integer)m_literals.get(literal);
        if (result==null) {
            result=new Integer(m_literals.size());
            m_literals.put(literal,result);
            m_insertLiteral.setInt(1,result.intValue());
            m_insertLiteral.setString(2,literal.getLabel());
            m_insertLiteral.addBatch();
        }
        return result;
    }
    /**
     * Starts the application.
     *
     * @param args              command line arguments
     */
    public static void main(String[] args) throws Exception {
        BufferedReader in=new BufferedReader(new InputStreamReader(System.in));
        String rdfName=null;
        if (args.length>=1)
            rdfName=args[0];
        else {
            System.out.print("URI of the RDF file: ");
            rdfName=in.readLine();
        }
        String modelAlias=null;
        if (args.length>=2)
            modelAlias=args[1];
        else {
            System.out.print("Model alias: ");
            modelAlias=in.readLine();
        }
        String defaultNamespace=null;
        if (args.length>=3)
            defaultNamespace=args[2];
        else {
            System.out.print("Default namespace for the model: ");
            defaultNamespace=in.readLine();
        }
        String connectionString=null;
        if (args.length>=4)
            connectionString=args[3];
        else {
            System.out.print("Database JDBC connection string: ");
            connectionString=in.readLine();
        }
        String driverClass=null;
        if (args.length>=5)
            driverClass=args[4];
        else {
            System.out.print("JDBC driver class: ");
            driverClass=in.readLine();
        }
        if (rdfName==null || modelAlias==null || defaultNamespace==null || connectionString==null || driverClass==null) {
            System.out.println("Usage: edu.unika.aifb.kaon.rdfserver.tools.dbloader.Main [<rdf-file> [<alias> [<default-namespace> [<jdbc-connection-string> [<driver-class>]]]]]");
            System.exit(1);
        }
        Main loader=new Main();
        loader.doImport(rdfName,modelAlias,defaultNamespace,connectionString,driverClass);
    }
}
