package edu.unika.aifb.rdf.rdfserver.ejb;

import java.util.Collections;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import javax.ejb.EntityBean;
import javax.ejb.EntityContext;
import javax.ejb.EJBException;
import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import javax.ejb.FinderException;
import javax.naming.NamingException;
import java.rmi.RemoteException;

import edu.unika.aifb.rdf.rdfserver.interfaces.Node;
import edu.unika.aifb.rdf.rdfserver.interfaces.Resource;
import edu.unika.aifb.rdf.rdfserver.interfaces.ResourceHome;
import edu.unika.aifb.rdf.rdfserver.interfaces.ResourceUtil;
import edu.unika.aifb.rdf.rdfserver.interfaces.Statement;
import edu.unika.aifb.rdf.rdfserver.interfaces.StatementHome;
import edu.unika.aifb.rdf.rdfserver.interfaces.StatementInfo;
import edu.unika.aifb.rdf.rdfserver.interfaces.StatementUtil;
import edu.unika.aifb.rdf.rdfserver.interfaces.Literal;
import edu.unika.aifb.rdf.rdfserver.interfaces.LiteralHome;
import edu.unika.aifb.rdf.rdfserver.interfaces.LiteralUtil;
import edu.unika.aifb.rdf.rdfserver.interfaces.Counter;
import edu.unika.aifb.rdf.rdfserver.interfaces.CounterHome;
import edu.unika.aifb.rdf.rdfserver.interfaces.CounterUtil;

/**
 * @ejb.bean name="Model" cmp-version="2.x" type="CMP" jndi-name="ejb/kaon/Model" primkey-field="alias"
 * @ejb.interface remote-class="edu.unika.aifb.rdf.rdfserver.interfaces.Model"
 * @ejb.pk class="java.lang.String"
 * @ejb.transaction type="Required"
 *
 * @ejb.ejb-ref ejb-name="Resource"
 * @ejb.ejb-ref ejb-name="Literal"
 * @ejb.ejb-ref ejb-name="Statement"
 * @ejb.ejb-ref ejb-name="Counter"
 *
 * @jboss.create-table "${jboss.create.table}"
 * @jboss.remove-table "${jboss.remove.table}"
 * @jboss.tuned-updates "${jboss.tuned.updates}"
 * @jboss.read-only "${jboss.read.only}"
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public abstract class ModelBean implements EntityBean {
    protected LiteralHome m_literalHome;
    protected ResourceHome m_resourceHome;
    protected StatementHome m_statementHome;

    public void setEntityContext(EntityContext entityContext) throws EJBException {
        try {
            m_literalHome=LiteralUtil.getHome();
            m_resourceHome=ResourceUtil.getHome();
            m_statementHome=StatementUtil.getHome();
        }
        catch (Exception e) {
            throw new EJBException(e.getMessage());
        }
    }
    public void unsetEntityContext() throws EJBException {
        m_literalHome=null;
        m_resourceHome=null;
    }
    /**
     * @ejb.interface-method
     * @ejb.persistent-field
     * @ejb.pk-field
     */
    public abstract String getAlias();
    public abstract void setAlias(String alias);

    /**
     * @ejb.interface-method
     * @ejb.persistent-field
     */
    public abstract String getLogicalURI();
    /**
     * @ejb.interface-method
     */
    public abstract void setLogicalURI(String new_uri);

    /**
     * @ejb.create-method
     * @ejb.transaction type="Required"
     */
    public String ejbCreate(String alias) {
        setAlias(alias);
        return null;
    }
    public void ejbPostCreate(String alias) {
    }
    public void ejbLoad() {
    }
    public void ejbStore() {
    }
    public void ejbActivate() {
    }
    public void ejbPassivate() {
    }
    public void ejbRemove() {
    }

    /**
     * @ejb.interface-method
     */
    public boolean contains(StatementInfo statementInfo) throws RemoteException,FinderException {
        if (!getAlias().equals(statementInfo.getModelID()))
            return false;
        return !findStatements(statementInfo).isEmpty();
    }

    /**
     * @ejb.interface-method
     */
    public boolean contains(Statement statement) throws RemoteException,FinderException {
        return equals(statement.getModel());
    }

    /**
     * Locates single resource by label. If resource is not found, <code>null</code> is returned.
     */
    protected Resource locateResourceByLabel(String label) throws RemoteException,FinderException {
        Collection result=m_resourceHome.findByLabel(label);
        if (result.isEmpty())
            return null;
        return (Resource)result.iterator().next();
    }

    /**
     * Locates single literal by label. If literal is not found, <code>null</code> is returned.
     */
    protected Literal locateLiteralByLabel(String label) throws RemoteException,FinderException {
        Collection result=m_literalHome.findByLabel(label);
        if (result.isEmpty())
            return null;
        return (Literal)result.iterator().next();
    }

    /**
     * @ejb.interface-method
     */
    public Collection findStatements(StatementInfo statementInfo) throws RemoteException,FinderException {
        Resource subj=null;
        if (statementInfo.getSubject()!=null) {
            subj=locateResourceByLabel(statementInfo.getSubject());
            if (subj==null)
                return Collections.EMPTY_LIST;
        }
        Resource pred=null;
        if (statementInfo.getPredicate()!=null) {
            pred=locateResourceByLabel(statementInfo.getPredicate());
            if (pred==null)
                return Collections.EMPTY_LIST;
        }
        Node obj=null;
        if (statementInfo.getObject()!=null) {
            if (statementInfo.getObjectIsLiteral())
                obj=locateLiteralByLabel(statementInfo.getObject());
            else
                obj=locateResourceByLabel(statementInfo.getObject());
            if (obj==null)
                return Collections.EMPTY_LIST;
        }
        return find(subj,pred,obj);
    }

    /**
     * @return Collection[StatementInfo] all statements infos matching the criteria
     * @ejb.interface-method
     */
    public Collection findStatmentInfos(StatementInfo statementInfo) throws RemoteException, FinderException {
        return getStatementInfos(findStatements(statementInfo));
    }

    /**
     * Converts a collection of statements into a collection of StatementInfo objects.
     */
    protected Collection getStatementInfos(Collection statements) throws RemoteException,FinderException {
        Collection result=new LinkedList();
        Iterator objects=statements.iterator();
        while (objects.hasNext()) {
            Statement statement=(Statement)objects.next();
            result.add(statement.getStatementInfo());
        }
        return result;
    }

    /**
     * Adds a statement to a model.
     * @ejb.interface-method
     * @ejb.transaction type="Required"
     */
    public void add(StatementInfo statementInfo) throws RemoteException,FinderException,CreateException {
        if (findStatmentInfos(statementInfo).size()!=0)
            throw new CreateException("Statement already exists.");
        Resource subject=locateResourceByLabel(statementInfo.getSubject());
        if (subject==null)
            subject=m_resourceHome.create(statementInfo.getSubject());
        Resource predicate=locateResourceByLabel(statementInfo.getPredicate());
        if (predicate==null)
            predicate=m_resourceHome.create(statementInfo.getPredicate());
        Node object;
        if (statementInfo.getObjectIsLiteral()) {
            object=locateLiteralByLabel(statementInfo.getObject());
            if (object==null)
                object=m_literalHome.create(statementInfo.getObject());
        }
        else {
            object=locateResourceByLabel(statementInfo.getObject());
            if (object==null)
                object=m_resourceHome.create(statementInfo.getObject());
        }
        m_statementHome.create(getAlias(),((Integer)subject.getPrimaryKey()).intValue(),((Integer)predicate.getPrimaryKey()).intValue(),((Integer)object.getPrimaryKey()).intValue(),statementInfo.getObjectIsLiteral());
    }

    /**
     * @ejb.interface-method
     */
    public Collection find(Resource subj,Resource pred,Node obj) throws RemoteException,FinderException {
        if (subj==null && pred==null && obj==null)
            return
        getStatements();
        if (pred==null && obj==null)
            return findSubj(subj);
        if (subj==null && obj==null)
            return findPred(pred);
        if (subj==null && pred==null)
            return findObj(obj);
        if (subj==null)
            return findPredObj(pred,obj);
        if (pred==null)
            return findSubjObj(subj,obj);
        if (obj==null)
            return findSubjPred(subj,pred);
        // Nothing is null
        Integer subjKey=(Integer)subj.getPrimaryKey();
        Integer predKey=(Integer)pred.getPrimaryKey();
        Integer objKey=(Integer)obj.getPrimaryKey();
        return m_statementHome.findByAllAttributes(getAlias(),subjKey,predKey,objKey,new Boolean(obj instanceof Literal));
    }

    protected Collection findSubj(Resource subj) throws RemoteException,FinderException {
        Integer subjKey=(Integer)subj.getPrimaryKey();
        return m_statementHome.findByModelSubject(getAlias(),subjKey);
    }

    protected Collection findPred(Resource pred) throws RemoteException,FinderException {
        Integer predKey=(Integer) pred.getPrimaryKey();
        return m_statementHome.findByModelPredicate(getAlias(),predKey);
    }

    protected Collection findObj(Node obj) throws RemoteException,FinderException {
        Integer objKey=(Integer)obj.getPrimaryKey();
        return m_statementHome.findByModelObject(getAlias(),objKey,new Boolean(obj instanceof Literal));
    }

    protected Collection findPredObj(Resource pred, Node obj) throws RemoteException,FinderException {
        Integer predKey=(Integer)pred.getPrimaryKey();
        Integer objKey=(Integer)obj.getPrimaryKey();
        return m_statementHome.findByModelPredObj(getAlias(),predKey,objKey,new Boolean(obj instanceof Literal));
    }

    protected Collection findSubjObj(Resource subj,Node obj) throws RemoteException,FinderException {
        Integer subjKey=(Integer)subj.getPrimaryKey();
        Integer objKey=(Integer)obj.getPrimaryKey();
        return m_statementHome.findByModelSubjObj(getAlias(),subjKey,objKey,new Boolean(obj instanceof Literal));
    }

    protected Collection findSubjPred(Resource subj,Resource pred) throws RemoteException,FinderException {
        Integer subjKey=(Integer)subj.getPrimaryKey();
        Integer predKey=(Integer)pred.getPrimaryKey();
        return m_statementHome.findByModelSubjPred(getAlias(),subjKey,predKey);
    }

    /**
     * @ejb.interface-method
     * @ejb.transaction type="Required"
     */
    public void remove(StatementInfo statementInfo) throws RemoteException,RemoveException,FinderException {
        Collection collection=findStatements(statementInfo);
        Iterator iterator=collection.iterator();
        while (iterator.hasNext()) {
            Statement statement=(Statement)iterator.next();
            statement.remove();
        }
    }

    /**
     * Applies the list of changes to the model.
     * @ejb.interface-method
     * @ejb.transaction type="Required"
     */
    public void applyChanges(Collection statementInfoList) throws RemoteException,CreateException,RemoveException,FinderException {
        Iterator iterator=statementInfoList.iterator();
        while (iterator.hasNext()) {
            StatementInfo statementInfo=(StatementInfo)iterator.next();
            switch (statementInfo.getStatementAction()) {
            case 1:
                add(statementInfo);
                break;
            case 2:
                remove(statementInfo);
                break;
            }
        }
    }

    /**
     * @ejb.interface-method
     */
    public Collection getStatements() throws RemoteException,FinderException {
        return m_statementHome.findByModelKey(getAlias());
    }

    /**
     * @ejb.interface-method
     */
    public Collection getStatementInfos() throws RemoteException,FinderException {
        return getStatementInfos(getStatements());
    }

    /**
     * @ejb.interface-method
     */
    public int size() throws RemoteException {
        // TO DO:
        // Optimize by SQL ABFRAGE SELECT COUNT(*) FROM TRIPLE WHERE MODEL = ...
        try {
            return getStatements().size();
        } catch (FinderException fe) {
            return 0;
        }
    }

    /**
     * @ejb.interface-method
     * @ejb.transaction type="Required"
     */
    public String getUniqueResourceURI(String prefix) throws RemoteException,CreateException {
        try {
            CounterHome home=CounterUtil.getHome();
            String mangledPrefix="unique-"+prefix;
            Counter counter;
            try {
                counter=home.findByPrimaryKey(mangledPrefix);
            }
            catch (FinderException e) {
                counter=home.create(mangledPrefix);
            }
            return prefix+counter.getNextCounterValue();
        }
        catch (NamingException e) {
            throw new CreateException(e.getMessage());
        }
    }
}
