package edu.unika.aifb.rdf.rdfserver.client;

import java.util.Map;
import java.util.ListIterator;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Collection;
import javax.ejb.FinderException;
import javax.ejb.CreateException;
import javax.ejb.RemoveException;
import java.rmi.RemoteException;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.mainmemory.*;
import edu.unika.aifb.rdf.rdfserver.interfaces.*;

/**
 * Persistent RDF model implementation that is backed up by an EJB server.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class RemoteModel extends AbstractModel {
    protected String m_physicalURI;
    protected edu.unika.aifb.rdf.rdfserver.interfaces.Model m_model;
    protected List m_statementActions;
    protected String m_alias;

    /**
     * Creates a remote model from an open connection to the server.
     *
     * @param model                 model to which this model attaches
     * @param nodeFactory           the node factory
     * @param physicalURI           physical URI of the model
     */
    public RemoteModel(edu.unika.aifb.rdf.rdfserver.interfaces.Model model,NodeFactory nodeFactory,String physicalURI) throws ModelException {
        super(nodeFactory);
        m_model=model;
        m_physicalURI=physicalURI;
        try {
            m_alias=m_model.getAlias();
            if (m_model.getLogicalURI()==null)
                m_model.setLogicalURI(m_physicalURI);
        }
        catch (RemoteException e) {
            throw new ModelException("Remote call error",e);
        }
    }
    /**
     * Returns <code>true</code> since this model is persistent.
     *
     * @return              always <code>true</code>
     */
    public boolean isPersistent() {
        return true;
    }
    /**
     * Returns the logical URI of the model.
     *
     * @return              logical URI of the model
     */
    public String getLogicalURI() throws ModelException {
        try {
            return m_model.getLogicalURI();
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
    }
    /**
     * Sets the logical URI of the model.
     *
     * @param logicalURI    logical URI of the model
     */
    public void setLogicalURI(String logicalURI) throws ModelException {
        try {
            m_model.setLogicalURI(logicalURI);
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
    }
    /**
     * Returns the physical URI of the model.
     *
     * @return              physical URI of the model
     */
    public String getPhysicalURI() {
        return m_physicalURI;
    }
    /**
     * Sets the physical URI of the model.
     *
     * @param physicalURI   physical URI of the model
     */
    public void setPhysicalURI(String physicalURI) {
        m_physicalURI=physicalURI;
    }
    /**
     * Number of triples in the model.
     * @return  number of triples
     */
    public int thisSize() throws ModelException {
        try {
            return m_model.size();
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
    }

    public boolean thisIsEmpty() throws ModelException {
        return size()==0;
    }
    /**
     * Generates a collection of <code>Statement</code> objects from a collection of <code>StatementInfo</code> objects.
     */
    protected Collection getStatements(Collection input) throws ModelException {
        List result=new LinkedList();
        Iterator i=input.iterator();
        while (i.hasNext()) {
            StatementInfo statementInfo=(StatementInfo)i.next();
            edu.unika.aifb.rdf.api.model.Resource subject=m_nodeFactory.createResource(statementInfo.getSubject());
            edu.unika.aifb.rdf.api.model.Resource predicate=m_nodeFactory.createResource(statementInfo.getPredicate());
            edu.unika.aifb.rdf.api.model.RDFNode object;
            if (statementInfo.getObjectIsLiteral())
                object=m_nodeFactory.createLiteral(statementInfo.getObject());
            else
                object=m_nodeFactory.createResource(statementInfo.getObject());
            result.add(m_nodeFactory.createStatement(subject,predicate,object));
        }
        return result;
    }
    /**
     * Enumerate triples int the model.
     */
    public Iterator thisIterator() throws ModelException {
        try {
            return getStatements(m_model.getStatementInfos()).iterator();
        } catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        } catch (FinderException e) {
            throw new ModelException("Server exception",e);
        }
    }
    /**
     * Returns a StatementInfo for a Statement.
     */
    public StatementInfo getStatementInfo(edu.unika.aifb.rdf.api.model.Statement t,int statementAction) throws ModelException {
        return new StatementInfo(null,m_alias,t.subject().getLabel(),t.predicate().getLabel(),t.object().getLabel(),t.object() instanceof edu.unika.aifb.rdf.api.model.Literal,statementAction);
    }
    /**
     * Returns a Statement for a StatementInfo.
     */
    public edu.unika.aifb.rdf.api.model.Statement getStatement(StatementInfo t) throws ModelException {
        edu.unika.aifb.rdf.api.model.Resource subject=m_nodeFactory.createResource(t.getSubject());
        edu.unika.aifb.rdf.api.model.Resource predicate=m_nodeFactory.createResource(t.getPredicate());
        edu.unika.aifb.rdf.api.model.RDFNode object;
        if (t.getObjectIsLiteral())
            object=m_nodeFactory.createLiteral(t.getObject());
        else
            object=m_nodeFactory.createResource(t.getObject());
        return m_nodeFactory.createStatement(subject,predicate,object);
    }
    /**
     * Tests if the model contains the given triple.
     * @return  <code>true</code> if the triple belongs to the model; <code>false</code> otherwise.
     */
    public synchronized boolean thisContains(edu.unika.aifb.rdf.api.model.Statement t) throws ModelException {
        try {
            if (!isAutocommit() && m_statementActions.size()>0) {
                String subject=t.subject().getURI();
                String predicate=t.predicate().getURI();
                boolean objectIsLiteral=t.object() instanceof edu.unika.aifb.rdf.api.model.Literal;
                String object=t.object().getLabel();
                ListIterator iterator=m_statementActions.listIterator(m_statementActions.size());
                while (iterator.hasPrevious()) {
                    StatementInfo statementInfo=(StatementInfo)iterator.previous();
                    if (subject.equals(statementInfo.getSubject()) && predicate.equals(statementInfo.getPredicate()) && object.equals(statementInfo.getObject()) && objectIsLiteral==statementInfo.getObjectIsLiteral())
                        return statementInfo.getStatementAction()==1;
                }
            }
            return m_model.contains(getStatementInfo(t,0));
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
        catch (FinderException e) {
            throw new ModelException("Server exception",e);
        }
    }
    /**
     * Tests if the model contains the given triple, where each component may be <code>null</code>.
     * @return  <code>true</code> if the triple belongs to the model; <code>false</code> otherwise.
     */
    public synchronized boolean thisContains(edu.unika.aifb.rdf.api.model.Resource subject,edu.unika.aifb.rdf.api.model.Resource predicate,edu.unika.aifb.rdf.api.model.RDFNode object) throws ModelException {
        try {
            String subjectLabel=null;
            if (subject!=null)
                subjectLabel=subject.getLabel();
            String predicateLabel=null;
            if (predicate!=null)
                predicateLabel=predicate.getLabel();
            String objectLabel=null;
            if (object!=null)
                objectLabel=object.getLabel();
            StatementInfo statementInfo=new StatementInfo(null,m_alias,subjectLabel,predicateLabel,objectLabel,object instanceof edu.unika.aifb.rdf.api.model.Literal);
            return m_model.contains(statementInfo);
        } catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        } catch (FinderException e) {
            throw new ModelException("Server exception",e);
        }
    }
    /**
     * Adds a new triple to the model. A literal is created out of the string parameter <code>object</code>.
     * This method is just a shortcut.
     */
    public synchronized void add(edu.unika.aifb.rdf.api.model.Resource subject,edu.unika.aifb.rdf.api.model.Resource predicate,String object) throws ModelException {
        add(subject,predicate,m_nodeFactory.createLiteral(object));
    }
    /**
     * Adds a new triple to the model. The object of the triple is an RDFNode.
     */
    public synchronized void add(edu.unika.aifb.rdf.api.model.Resource subject,edu.unika.aifb.rdf.api.model.Resource predicate,edu.unika.aifb.rdf.api.model.RDFNode object) throws ModelException {
        try {
            StatementInfo statementInfo=new StatementInfo(null,m_alias,subject.getLabel(),predicate.getLabel(),object.getLabel(),object instanceof edu.unika.aifb.rdf.api.model.Literal,1);
            if (isAutocommit())
                m_model.add(statementInfo);
            else
                m_statementActions.add(statementInfo);
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
        catch (CreateException e) {
            if ("Statement already exists.".equals(e.getMessage()))
                throw new StatementAlreadyExists(m_nodeFactory.createStatement(subject,predicate,object));
            else
                throw new ModelException("Server exception",e);
        }
        catch (FinderException e) {
            throw new ModelException("Server exception",e);
        }
    }
    /**
     * Adds a new triple to the model.
     */
    public synchronized void add(edu.unika.aifb.rdf.api.model.Statement t) throws ModelException {
        add(t.subject(),t.predicate(),t.object());
    }
    /**
     * Removes the triple from the model.
     */
    public synchronized void remove(edu.unika.aifb.rdf.api.model.Statement t) throws ModelException {
        try {
            StatementInfo statementInfo=getStatementInfo(t,2);
            if (isAutocommit())
                m_model.remove(statementInfo);
            else
                m_statementActions.add(statementInfo);
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
        catch (RemoveException e) {
            throw new ModelException("Server exception",e);
        }
        catch (FinderException e) {
            throw new StatementDoesntExist(t);
        }
    }
    /**
     * General method to search for triples. <code>null</code> input for any parameter will match anything.
     * <p>Example: <code>Model result = m.find( null, RDF.type, new Resource("http://...#MyClass") )</code>
     * <p>finds all instances of the class <code>MyClass</code>
     */
    public synchronized void thisFind(edu.unika.aifb.rdf.api.model.Resource subject, edu.unika.aifb.rdf.api.model.Resource predicate, edu.unika.aifb.rdf.api.model.RDFNode object,edu.unika.aifb.rdf.api.model.Model result) throws ModelException {
        try {
            String subjectLabel=null;
            if (subject!=null)
                subjectLabel=subject.getLabel();
            String predicateLabel=null;
            if (predicate!=null)
                predicateLabel=predicate.getLabel();
            String objectLabel=null;
            if (object!=null)
                objectLabel=object.getLabel();
            StatementInfo statementInfo=new StatementInfo(null,m_alias,subjectLabel,predicateLabel,objectLabel,object instanceof edu.unika.aifb.rdf.api.model.Literal);
            Iterator statementInfos=m_model.findStatmentInfos(statementInfo).iterator();
            while (statementInfos.hasNext()) {
                statementInfo=(StatementInfo)statementInfos.next();
                result.add(getStatement(statementInfo));
            }
        } catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        } catch (FinderException e) {
            throw new ModelException("Server exception",e);
        }
    }
    public String getUniqueResourceURI(String prefix) throws ModelException {
        try {
            return m_model.getUniqueResourceURI(prefix);
        }
        catch (RemoteException e) {
            throw new ModelException("Server exception",e);
        }
        catch (CreateException e) {
            throw new ModelException("Server exception",e);
        }
    }
    public String getLabel() throws ModelException {
        return getURI();
    }
    public String getLocalName() throws ModelException {
        return getURI();
    }
    public String getNamespace() throws ModelException {
        return getURI();
    }
    public String getURI() throws ModelException {
        return getLogicalURI();
    }
    public boolean supportsTransactions() {
        return true;
    }
    public void setAutocommit(boolean autocommit) throws ModelException {
        if (autocommit) {
            if (m_statementActions!=null && m_statementActions.size()!=0)
                throw new ModelException("Autocommit mode cannot be changed during a transaction.");
            m_statementActions=null;
        }
        else
            m_statementActions=new LinkedList();
    }
    public boolean isAutocommit() {
        return m_statementActions==null;
    }
    public void commit() throws ModelException {
        if (m_statementActions!=null)
            try {
                m_model.applyChanges(m_statementActions);
                m_statementActions.clear();
            }
            catch (RemoteException e) {
                throw new ModelException("Server exception",e);
            }
            catch (CreateException e) {
                throw new ModelException("Server exception",e);
            }
            catch (RemoveException e) {
                throw new ModelException("Server exception",e);
            }
            catch (FinderException e) {
                throw new ModelException("Server exception",e);
            }
    }
    public void rollback() {
        if (m_statementActions!=null)
            m_statementActions.clear();
    }
    public String getAttribute(String key) {
        return null;
    }
    public void setAttribute(String key,String value) throws ModelException {
        throw new ModelException("RDF Server doesn't support setting model attributes.");
    }
    public Map getAttributes() {
        return null;
    }
}
