package edu.unika.aifb.rdf.rdfcrawler;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.util.*;

/**
 * Configuration of the crawler.
 */
public class Configuration {
    protected static final String SYNDICATOR_SCHEMA="http://kaon.semanticweb.org/syndicator/syndicator.rdf#";

    protected final int m_timeOut=60;
    protected int m_defaultDepth;
    protected List m_domainFilter;
    protected List m_providerList;
    protected String m_logURL;
    protected String m_outputURL;
    protected String m_cacheURL;
    protected String m_proxyHost;
    protected int m_proxyPort;

    public Configuration(String configurationURL) throws ConfigurationException {
        int hashIndex=configurationURL.indexOf("#");
        if (hashIndex==-1)
            throw new ConfigurationException("Please specify the configuration you want to use within the configuration URL "+configurationURL);
        String sourceURL=configurationURL.substring(0,hashIndex);
        try {
            Model model=RDFManager.openModel(sourceURL,null);
            parseModel(model,configurationURL);
        }
        catch (ModelException e) {
            throw new ConfigurationException("Error while parsing, no configuration loaded.",e);
        }
    }
    private void parseModel(Model model,String configurationURL) throws ModelException {
        Resource configuration=model.getNodeFactory().createResource(configurationURL);
        m_cacheURL=getValue(configuration,"cache_url",model);
        m_outputURL=getValue(configuration,"output_url",model);
        m_logURL=getValue(configuration,"log_url",model);
        try {
            String defaultDepth=getValue(configuration,"default_depth",model);
            if (defaultDepth!=null)
                m_defaultDepth=Integer.parseInt(defaultDepth);
            if (m_defaultDepth<0)
                m_defaultDepth=0;
        }
        catch (NumberFormatException ignored) {
            m_defaultDepth=0;
        }
        m_domainFilter=getVectorFromBag(configuration,"domain_filter",model);
        m_providerList=getVectorFromBag(configuration,"provider_list",model);
        m_proxyHost=getValue(configuration,"proxy_host",model);
        try {
            String proxyPort=getValue(configuration,"proxy_port",model);
            if (proxyPort!=null)
                m_proxyPort=Integer.parseInt(proxyPort);
        }
        catch (NumberFormatException ignored) {
            m_proxyPort=8080;
        }
    }
    private String getValue(Resource configurationURL,String property,Model model) throws ModelException {
        Resource propertyResource=model.getNodeFactory().createResource(SYNDICATOR_SCHEMA,property);
        Iterator iterator=model.find(configurationURL,propertyResource,null).iterator();
        if (iterator.hasNext()) {
            Statement statement=(Statement)iterator.next();
            return statement.object().getLabel();
        }
        return null;
    }
    private List getVectorFromBag(Resource configurationURL,String property,Model model) throws ModelException {
        List result=new ArrayList();
        String bagURI=getValue(configurationURL,property,model);
        if (bagURI!=null) {
            Resource bagURIResource=model.getNodeFactory().createResource(bagURI);
            Iterator iterator=model.find(bagURIResource,null,null).iterator();
            while (iterator.hasNext()) {
                Statement statement=(Statement)iterator.next();
                if (!statement.predicate().getLabel().equals("http://www.w3.org/1999/02/22-rdf-syntax-ns#type"))
                    result.add(statement.object().getLabel());
            }
        }
        return result;
    }
    public int getTimeOut() {
        return m_timeOut;
    }
    public int getDefaultDepth() {
        return m_defaultDepth;
    }
    public List getDomainFilter() {
        return m_domainFilter;
    }
    public List getProviderList() {
        return m_providerList;
    }
    public String getLogURL() {
        return m_logURL;
    }
    public String getOutputURL() {
        return m_outputURL;
    }
    public String getCacheURL() {
        return m_cacheURL;
    }
    public String getProxyHost() {
        return m_proxyHost;
    }
    public int getProxyPort() {
        return m_proxyPort;
    }
}
