package edu.unika.aifb.rdf.rdfcrawler;

import edu.unika.aifb.rdf.api.model.*;

/**
 * This class gets URIs one by one and decides when
 * to start new threads. It maintains a static reference
 * to URLList and Model, so that the threads
 * (also known as Channels) can update
 * the status of this list and the accumulating RDF model
 */
public class ChannelPool {
    protected Model m_model;
    protected int m_capacity;
    protected static URLList m_urlList;
    protected Cache m_cache;
    protected boolean m_endSignal;
    protected int m_activeChannels;
    protected int m_deadline;

    /**
     * <p>This constructor initializes ChannelPool - normally there
     * is just one instance of it per program.
     * It stores a static reference to the URLList, the list
     * may be updated by requests from ChannelPool or individual
     * channels therein.</p>
     *
     * <p>Also static reference of the cache.</p>
     *
     * <p>We pass also capacity constraint - no more channels than "capacity"
     * should be created for optimum performance.
     * "deadline" - for how many seconds we are performing operations
     * in the network (networking is assumed to be the bottleneck).
     * After deadline no more new threads are created, but the
     * existing ones may be allowed to finish their document processing.</p>
     */

    public ChannelPool(Model model,URLList urlList, Cache cache, int capacity, int deadline) {
        m_model=model;
        m_urlList=urlList;
        m_cache=cache;
        m_capacity=capacity;
        m_deadline=deadline;
    }
    public synchronized void crawl() {
        Channel[] channels=new Channel[m_capacity];
        for (int i=0;i<m_capacity;i++) {
            m_activeChannels++;
            channels[i]=new Channel(this);
            Thread thread=new Thread(channels[i]);
            thread.start();
        }
        long startTime=System.currentTimeMillis();
        while (m_activeChannels!=0) {
            try {
                wait(1000);
            }
            catch (InterruptedException e) {
                break;
            }
            if (System.currentTimeMillis()-startTime>m_deadline*1000)
                break;
        }
        m_endSignal=true;
        notifyAll();
        while (m_activeChannels!=0) {
            try {
                wait();
            }
            catch (InterruptedException e) {
                break;
            }
        }
    }
    public Cache getCache() {
        return m_cache;
    }
    public Model getModel() {
        return m_model;
    }
    synchronized void channelTerminating() {
        m_activeChannels--;
        notifyAll();
    }
    synchronized URLStruct getUnprocessedURL() {
        if (m_endSignal)
            return null;
        URLStruct urlStruct=m_urlList.getUnprocessedURL();
        while (!m_endSignal && urlStruct==null && !m_urlList.listProcessed()) {
            try {
                wait();
            }
            catch (InterruptedException e) {
                break;
            }
            urlStruct=m_urlList.getUnprocessedURL();
        }
        return urlStruct;
    }
    synchronized void markURLProcessed(URLStruct struct) {
        m_urlList.markURLProcessed(struct);
        notifyAll();
    }
    synchronized void markURLInError(URLStruct struct,Exception e) {
        m_urlList.markURLInError(struct,e);
        notifyAll();
    }
    synchronized void addUnprocessedURL(String documentURL,String parentURL,boolean decrement) {
        m_urlList.addURL(documentURL,parentURL,decrement);
    }
}
