package edu.unika.aifb.rdf.mainmemory.test;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import junit.framework.TestCase;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.syntax.*;
import edu.unika.aifb.rdf.api.util.*;

/**
 * Test cases for RDF serializer.
 */
public class TestRDFSerializer extends TestCase {
    protected static final String CR=System.getProperty("line.separator");

    protected static final String MODEL1_PHYSICAL=TestModelContext.class.getResource("res/model1.xml").toString();
    protected static final String MODEL2_PHYSICAL=TestModelContext.class.getResource("res/model2.xml").toString();
    protected static final String MODEL3_PHYSICAL=TestModelContext.class.getResource("res/model3.xml").toString();

    public static final String RDFNS="http://www.w3.org/1999/02/22-rdf-syntax-ns#";
    public static final String RDFSCHEMA="http://www.w3.org/TR/1999/PR-rdf-schema-19990303#";
    public static final String XML_START="<?xml version='1.0'?>"+CR;
    public static final String RDF_START_INTERNAL=
        "<rdf:RDF xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'"+CR+
        "         xmlns:rdfs='http://www.w3.org/2000/01/rdf-schema#'"+CR+
        "         xmlns:test='test:'"+CR+
        ">"+CR;
    public static final String RDF_START=XML_START+RDF_START_INTERNAL;
    public static final String RDF_END="</rdf:RDF>"+CR;

    protected RDFSerializer m_serializer;
    protected ModelContext m_modelContext;

    public TestRDFSerializer(String name) {
        super(name);
    }
    protected void setUp() throws Exception {
        m_serializer=RDFManager.createSerializer();
        m_modelContext=new ModelContext();
        m_modelContext.mapLogicalToPhysicalURI("http://test.org/model1.xml",MODEL1_PHYSICAL);
    }
    protected Model parseModel(String systemID,String rdf) throws Exception {
        InputSource inputSource=new InputSource(new StringReader(rdf));
        inputSource.setSystemId(systemID);
        Model model=RDFManager.createModel(systemID,null);
        RDFParser parser=RDFManager.createParser();
        parser.parse(inputSource,new ModelConsumer(model));
        return model;
    }
    protected void assertModelsSame(Model model1,Model model2) throws Exception {
        assertEquals(model1.thisSize(),model2.thisSize());
        assertEquals(model1.getAttributes(),model2.getAttributes());
        Iterator iterator=model1.thisIterator();
        while (iterator.hasNext()) {
            Statement statement=(Statement)iterator.next();
            if (!model2.contains(statement))
                fail("Model 2 doesn't contain statement ["+statement.subject()+","+statement.predicate()+","+statement.object()+"]");
        }
    }
    protected void assertModelsSame(Model model,String rdf) throws Exception {
        assertModelsSame(model,parseModel("file:/c:/something",rdf));
    }
    protected String serializeModel(Model model) throws Exception {
        StringWriter writer=new StringWriter();
        m_serializer.serialize(model,writer,RDFSerializer.DEFAULT_ENCODING);
        return writer.toString();
    }
    public void testSerialization() throws Exception {
        Model model=parseModel("file:/c:/temp/a.rdf",
            RDF_START+
            "<rdfs:Class rdf:ID='xx'>"+CR+
            "    <test:value1 rdf:resource='test:resource'/>"+CR+
            "    <test:value2 rdf:resource='test:resource'/>"+CR+
            "</rdfs:Class>"+CR+
            "<rdf:Description rdf:about='#yy'>"+CR+
            "    <test:literal>Literal</test:literal>"+CR+
            "</rdf:Description>"+CR+
            "<rdf:Description rdf:ID='xx'>"+CR+
            "    <test:value3>Literal3</test:value3>"+CR+
            "</rdf:Description>"+CR+
            RDF_END);
        String serialized=serializeModel(model);
        assertSerializationsEquals(
            "<?xml version='1.0' encoding='"+RDFSerializer.DEFAULT_ENCODING+"'?>"+CR+
            "<!DOCTYPE rdf:RDF ["+CR+
            "    <!ENTITY a 'test:'>"+CR+
            "    <!ENTITY rdf 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'>"+CR+
            "    <!ENTITY rdfs 'http://www.w3.org/2000/01/rdf-schema#'>"+CR+
            "]>"+CR+
            CR+
            "<rdf:RDF xml:base=\"file:/c:/temp/a.rdf\""+CR+
            "    xmlns:a=\"&a;\""+CR+
            "    xmlns:rdf=\"&rdf;\""+CR+
            "    xmlns:rdfs=\"&rdfs;\">"+CR+
            CR+
            "<rdfs:Class rdf:ID=\"xx\""+CR+
            "    a:value3=\"Literal3\">"+CR+
            "    <a:value1 rdf:resource=\"&a;resource\"/>"+CR+
            "    <a:value2 rdf:resource=\"&a;resource\"/>"+CR+
            "</rdfs:Class>"+CR+
            "<rdf:Description rdf:ID=\"yy\""+CR+
            "    a:literal=\"Literal\"/>"+CR+
            CR+
            "</rdf:RDF>"+CR,serialized);
        assertModelsSame(model,serialized);
    }
    public void testSerializationOfIncludes() throws Exception {
        Model model2=m_modelContext.getModelPhysical(MODEL2_PHYSICAL);
        String serialized=serializeModel(model2);
        assertSerializationsEquals(
            "<?xml version='1.0' encoding='"+RDFSerializer.DEFAULT_ENCODING+"'?>"+CR+
            "<!DOCTYPE rdf:RDF ["+CR+
            "    <!ENTITY a 'test:'>"+CR+
            "    <!ENTITY rdf 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'>"+CR+
            "]>"+CR+
            CR+
            "<?include-rdf logicalURI=\"http://test.org/model1.xml\" physicalURI=\"model1.xml\"?>"+CR+
            CR+
            "<rdf:RDF xml:base=\"http://test.org/model2.xml\""+CR+
            "    xmlns:a=\"&a;\""+CR+
            "    xmlns:rdf=\"&rdf;\">"+CR+
            CR+
            "<rdf:Description rdf:about=\"&a;obj2\""+CR+
            "    a:value=\"Value Obj2\"/>"+CR+
            CR+
            "</rdf:RDF>"+CR,serialized);
        assertModelsSame(model2,serialized);
    }
    public void testSerializationOfSamePredicates() throws Exception {
        Model model=RDFManager.createModel("file:/c:/temp/bla.kaon",null);
        NodeFactory nodeFactory=model.getNodeFactory();
        Resource subject=nodeFactory.createResource("test:subject");
        Resource predicate=nodeFactory.createResource("test:predicate");
        Resource predicate2=nodeFactory.createResource("test:predicate2");
        Literal value1=nodeFactory.createLiteral("value1");
        Literal value2=nodeFactory.createLiteral("value2");
        model.add(nodeFactory.createStatement(subject,predicate,value1));
        model.add(nodeFactory.createStatement(subject,predicate,value2));
        model.add(nodeFactory.createStatement(subject,predicate2,value1));
        String serialized=serializeModel(model);
        assertSerializationsEquals(
            "<?xml version='1.0' encoding='"+RDFSerializer.DEFAULT_ENCODING+"'?>"+CR+
            "<!DOCTYPE rdf:RDF ["+CR+
            "    <!ENTITY a 'test:'>"+CR+
            "    <!ENTITY rdf 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'>"+CR+
            "]>"+CR+
            CR+
            "<rdf:RDF xml:base=\"file:/c:/temp/bla.kaon\""+CR+
            "    xmlns:a=\"&a;\""+CR+
            "    xmlns:rdf=\"&rdf;\">"+CR+
            CR+
            "<rdf:Description rdf:about=\"&a;subject\""+CR+
            "    a:predicate=\"value1\""+CR+
            "    a:predicate2=\"value1\">"+CR+
            "    <a:predicate>value2</a:predicate>"+CR+
            "</rdf:Description>"+CR+
            CR+
            "</rdf:RDF>"+CR,serialized);
        assertModelsSame(model,serialized);
    }
    public void testNonAbbreviableElements() throws Exception {
        Model model=RDFManager.createModel("file:/c:/temp/bla.kaon",null);
        model.setLogicalURI("http://localhost/test");
        NodeFactory nodeFactory=model.getNodeFactory();
        Resource subject1=nodeFactory.createResource("http://localhost/test#subject1");
        Resource subject2=nodeFactory.createResource("http://localhost/test#subject2");
        Resource type=nodeFactory.createResource(RDFNS+"type");
        Resource nonAbbreviable=nodeFactory.createResource("http://localhost/test#non+abbreviable+type");
        Resource abbreviable=nodeFactory.createResource("http://localhost/test#abbreviable-type");
        model.add(nodeFactory.createStatement(subject1,type,nonAbbreviable));
        model.add(nodeFactory.createStatement(subject2,type,abbreviable));
        String serialized=serializeModel(model);
        assertSerializationsEquals(
            "<?xml version='1.0' encoding='"+RDFSerializer.DEFAULT_ENCODING+"'?>"+CR+
            "<!DOCTYPE rdf:RDF ["+CR+
            "    <!ENTITY a 'http://localhost/test#'>"+CR+
            "    <!ENTITY rdf 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'>"+CR+
            "]>"+CR+
            CR+
            "<rdf:RDF xml:base=\"http://localhost/test\""+CR+
            "    xmlns:a=\"&a;\""+CR+
            "    xmlns:rdf=\"&rdf;\">"+CR+
            CR+
            "<rdf:Description rdf:ID=\"subject1\">"+CR+
            "    <rdf:type rdf:resource=\"#non+abbreviable+type\"/>"+CR+
            "</rdf:Description>"+CR+
            "<a:abbreviable-type rdf:ID=\"subject2\"/>"+CR+
            CR+
            "</rdf:RDF>"+CR,serialized);
        assertModelsSame(model,serialized);
    }
    public void testAttributes() throws Exception {
        Model model=RDFManager.createModel("file:/c:/temp/bla.kaon",null);
        model.setLogicalURI("http://localhost/test");
        NodeFactory nodeFactory=model.getNodeFactory();
        Resource subject=nodeFactory.createResource("http://localhost/test#subject");
        Resource type=nodeFactory.createResource(RDFNS+"type");
        Resource typeID=nodeFactory.createResource("http://localhost/test#typeID");
        model.add(nodeFactory.createStatement(subject,type,typeID));
        model.setAttribute("attributeKey","attributeValue");
        String serialized=serializeModel(model);
        assertSerializationsEquals(
            "<?xml version='1.0' encoding='"+RDFSerializer.DEFAULT_ENCODING+"'?>"+CR+
            "<!DOCTYPE rdf:RDF ["+CR+
            "    <!ENTITY a 'http://localhost/test#'>"+CR+
            "    <!ENTITY rdf 'http://www.w3.org/1999/02/22-rdf-syntax-ns#'>"+CR+
            "]>"+CR+
            CR+
            "<?model-attribute key=\"attributeKey\" value=\"attributeValue\"?>"+CR+
            CR+
            "<rdf:RDF xml:base=\"http://localhost/test\""+CR+
            "    xmlns:a=\"&a;\""+CR+
            "    xmlns:rdf=\"&rdf;\">"+CR+
            CR+
            "<a:typeID rdf:ID=\"subject\"/>"+CR+
            CR+
            "</rdf:RDF>"+CR,serialized);
        assertModelsSame(model,serialized);
    }
    protected void assertSerializationsEquals(String expected,String actual) {
        if (!expected.equals(actual)) {
            int index=0;
            int length=Math.max(expected.length(),actual.length());
            while (index<length) {
                if (expected.charAt(index)!=actual.charAt(index))
                    break;
                index++;
            }
            assertTrue("Serializations not equal."+CR+"Expected: "+expected+CR+"Actual: "+actual+CR+"Difference at index: "+index+CR+"Rest is: "+expected.substring(index),false);
        }
    }
}
