package edu.unika.aifb.rdf.mainmemory.test;

import java.io.StringReader;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import junit.framework.TestCase;

import edu.unika.aifb.rdf.api.syntax.*;
import edu.unika.aifb.rdf.api.util.*;

/**
 * Test cases for RDF parser.
 */
public class TestRDFParser extends TestCase {
    protected static final String CR=System.getProperty("line.separator");

    public static final String RDFNS="http://www.w3.org/1999/02/22-rdf-syntax-ns#";
    public static final String RDFSCHEMA="http://www.w3.org/TR/1999/PR-rdf-schema-19990303#";
    public static final String XML_START="<?xml version='1.0'?>"+CR;
    public static final String RDF_START_INTERNAL=
        "<rdf:RDF xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'"+CR+
        "         xmlns:rdfs='http://www.w3.org/2000/01/rdf-schema#'"+CR+
        "         xmlns:test='test:'"+CR+
        ">"+CR;
    public static final String RDF_START=XML_START+RDF_START_INTERNAL;
    public static final String RDF_END="</rdf:RDF>"+CR;

    public TestRDFParser(String name) {
        super(name);
    }
    protected void doTest(String systemID,String rdf,Statement[] result,String inputPhysycalURI,String inputLogicalURI) throws Exception {
        doTest(systemID,rdf,result,inputPhysycalURI,inputLogicalURI,new String[0][0]);
    }
    protected void doTest(String systemID,String rdf,Statement[] result,final String inputPhysycalURI,String inputLogicalURI,String[][] includedModels) throws Exception {
        final String[] parsedLogicalURI=new String[1];
        final Set set=new HashSet();
        final List includedModelsList=new ArrayList();
        final RDFConsumer consumer=new RDFConsumer() {
            public void startModel(String physicalURI) {
                assertEquals(inputPhysycalURI,physicalURI);
            }
            public void endModel() {
            }
            public void statementWithResourceValue(String subject,String predicate,String object) throws SAXException {
                set.add(new Statement(subject,predicate,object,false));
            }
            public void statementWithLiteralValue(String subject,String predicate,String object,String language,String datatype) throws SAXException {
                set.add(new Statement(subject,predicate,object,true));
            }
            public void logicalURI(String logicalURI) {
                parsedLogicalURI[0]=logicalURI;
            }
            public void includeModel(String logicalURI,String physicalURI) throws SAXException {
                includedModelsList.add(new String[] { logicalURI,physicalURI });
            }
            public void addModelAttribte(String key,String value) throws SAXException {
            }
        };
        InputSource inputSource=new InputSource(new StringReader(rdf));
        inputSource.setSystemId(systemID);
        RDFParser parser=RDFManager.createParser();
        parser.parse(inputSource,consumer);
        boolean ok=set.size()==result.length;
        for (int i=0;ok && i<result.length;i++)
            ok=set.contains(result[i]);
        if (!ok) {
            System.out.println("Expecting:");
            printStatements(result);
            System.out.println("Got:");
            Statement[] got=new Statement[set.size()];
            set.toArray(got);
            printStatements(got);
            fail();
        }
        assertEquals(inputLogicalURI,parsedLogicalURI[0]);
        assertEquals(includedModels.length,includedModelsList.size());
        for (int i=0;i<includedModels.length;i++) {
            String[] element=(String[])includedModelsList.get(i);
            assertEquals(includedModels[i][0],element[0]);
            assertEquals(includedModels[i][1],element[1]);
        }
    }
    protected void printStatements(Statement[] statements) {
        System.out.println("-----------------------------------");
        for (int i=0;i<statements.length;i++)
            System.out.println("( "+statements[i].m_subject+", "+statements[i].m_predicate+", "+(statements[i].m_objectIsLiteral ? "[" : "")+statements[i].m_object+(statements[i].m_objectIsLiteral ? "]" : "")+" )");
        System.out.println("===================================");
    }
    public void testDescription1() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1>value</test:prop1>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","value",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription2() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1' test:prop2='value2'>"+CR+
            "    <test:prop1>value1</test:prop1>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","value1",true),
                new Statement("file:/a.rdf#res1","test:prop2","value2",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription3() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:ID='reif'>value1</test:prop1>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","value1",true),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res1",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop1",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","value1",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription4() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:resource='test:refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription5() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:resource='#refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","file:/a.rdf#refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription6() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:resource='refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","file:/refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription7() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:resource='special:refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","special:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription8() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:about='dir/res1'>"+CR+
            "    <test:prop1 rdf:resource='special:refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/dir/res1","test:prop1","special:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription9() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' rdf:bagID='bag1' test:prop1='value1'>"+CR+
            "    <test:prop2 rdf:ID='reif' rdf:resource='special:refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","special:refres",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_1","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_2","file:/a.rdf#reif",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"predicate","test:prop1",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"object","value1",true),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop2",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","special:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription10() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:resource='test:refres'/>"+CR+
            "    <test:prop2 rdf:resource='test:refres'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","test:refres",false),
                new Statement("file:/a.rdf#res1","test:prop2","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testDescription11() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res'>"+CR+
            "    <test:prop1 rdf:ID='reif' rdf:bagID='bag1' rdf:resource='special:refres' test:nested1='nested1' test:nested2='nested2'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","special:refres",false),
                new Statement("special:refres","test:nested1","nested1",true),
                new Statement("special:refres","test:nested2","nested2",true),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop1",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","special:refres",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_1","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_2","file:/a.rdf#genid2",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"subject","special:refres",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"predicate","test:nested1",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"object","nested1",true),
                new Statement("file:/a.rdf#genid2",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"subject","special:refres",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"predicate","test:nested2",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"object","nested2",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testTwoDescriptions() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res1'>"+CR+
            "    <test:prop1 rdf:resource='special:refres1'/>"+CR+
            "</rdf:Description>"+
            "<rdf:Description rdf:ID='res2'>"+CR+
            "    <test:prop2 rdf:resource='special:refres2'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res1","test:prop1","special:refres1",false),
                new Statement("file:/a.rdf#res2","test:prop2","special:refres2",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testNoDescriptions() throws Exception {
        doTest("file:/a.rdf",RDF_START+RDF_END,new Statement[0],"file:/a.rdf","file:/a.rdf");
    }
    public void testTypedNode() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<test:myClass test:prop1='value1'>"+CR+
            "    <test:prop2 rdf:resource='special:refres'/>"+CR+
            "</test:myClass>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#genid1",RDFNS+"type","test:myClass",false),
                new Statement("file:/a.rdf#genid1","test:prop1","value1",true),
                new Statement("file:/a.rdf#genid1","test:prop2","special:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testTypedNodeBagID() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<test:myClass rdf:ID='res' test:prop1='value1' rdf:bagID='bag1'>"+CR+
            "    <test:prop2 rdf:ID='reif' rdf:resource='special:refres'/>"+CR+
            "</test:myClass>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res",RDFNS+"type","test:myClass",false),
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","special:refres",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_1","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_2","file:/a.rdf#genid2",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_3","file:/a.rdf#reif",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"predicate",RDFNS+"type",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"object","test:myClass",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"predicate","test:prop1",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"object","value1",true),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop2",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","special:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testTwoTypedNodes() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<test:myClass rdf:about='res1'>"+CR+
            "    <test:prop1 rdf:resource='special:refres1'/>"+CR+
            "</test:myClass>"+
            "<test:myClass rdf:ID='res2'>"+CR+
            "    <test:prop2 rdf:resource='special:refres2'/>"+CR+
            "</test:myClass>"+
            RDF_END,new Statement[] {
                new Statement("file:/res1","test:prop1","special:refres1",false),
                new Statement("file:/res1",RDFNS+"type","test:myClass",false),
                new Statement("file:/a.rdf#res2","test:prop2","special:refres2",false),
                new Statement("file:/a.rdf#res2",RDFNS+"type","test:myClass",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testNestedDescription() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' test:prop1='value1'>"+CR+
            "    <test:prop2>"+CR+
            "        <rdf:Description rdf:ID='nested' test:nestedprop1='nested1'>"+CR+
            "            <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "        </rdf:Description>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","file:/a.rdf#nested",false),
                new Statement("file:/a.rdf#nested","test:nestedprop1","nested1",true),
                new Statement("file:/a.rdf#nested","test:nestedprop2","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testNestedDescriptionID() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' test:prop1='value1'>"+CR+
            "    <test:prop2 rdf:ID='reif'>"+CR+
            "        <rdf:Description rdf:ID='nested' test:nestedprop1='nested1'>"+CR+
            "            <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "        </rdf:Description>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","file:/a.rdf#nested",false),
                new Statement("file:/a.rdf#nested","test:nestedprop1","nested1",true),
                new Statement("file:/a.rdf#nested","test:nestedprop2","test:refres",false),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop2",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","file:/a.rdf#nested",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testNestedTypedNode() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' test:prop1='value1'>"+CR+
            "    <test:prop2>"+CR+
            "        <test:myClass rdf:ID='nested' test:nestedprop1='nested1'>"+CR+
            "            <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "        </test:myClass>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","file:/a.rdf#nested",false),
                new Statement("file:/a.rdf#nested",RDFNS+"type","test:myClass",false),
                new Statement("file:/a.rdf#nested","test:nestedprop1","nested1",true),
                new Statement("file:/a.rdf#nested","test:nestedprop2","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testNestedTypedNodeID() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' test:prop1='value1'>"+CR+
            "    <test:prop2 rdf:ID='reif'>"+CR+
            "        <test:myClass rdf:ID='nested' test:nestedprop1='nested1'>"+CR+
            "            <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "        </test:myClass>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","file:/a.rdf#nested",false),
                new Statement("file:/a.rdf#nested",RDFNS+"type","test:myClass",false),
                new Statement("file:/a.rdf#nested","test:nestedprop1","nested1",true),
                new Statement("file:/a.rdf#nested","test:nestedprop2","test:refres",false),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop2",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","file:/a.rdf#nested",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testParseLiteral() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' test:prop1='value1'>"+CR+
            "    <test:prop2 rdf:parseType='Literal'>"+CR+
            "        <test:myClass rdf:ID='nested' test:nestedprop1='nested1'>"+CR+
            "            <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "        </test:myClass>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","\n"+
                    "        <test:myClass rdf:ID=\"nested\" test:nestedprop1=\"nested1\">\n"+
                    "            <test:nestedprop2 rdf:resource=\"test:refres\"></test:nestedprop2>\n"+
                    "        </test:myClass>\n"+
                    "    ",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testParseTypeResource() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res' test:prop1='value1'>"+CR+
            "    <test:prop2 rdf:parseType='Resource'>"+CR+
            "        <test:nestedprop1>nested1</test:nestedprop1>"+CR+
            "        <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop1","value1",true),
                new Statement("file:/a.rdf#res","test:prop2","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#genid1","test:nestedprop1","nested1",true),
                new Statement("file:/a.rdf#genid1","test:nestedprop2","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testParseTypeResourceBagID() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:ID='res'>"+CR+
            "    <test:prop2 rdf:ID='reif' rdf:parseType='Resource' rdf:bagID='bag1'>"+CR+
            "        <test:nestedprop2 rdf:resource='test:refres'/>"+CR+
            "    </test:prop2>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#res","test:prop2","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#genid1","test:nestedprop2","test:refres",false),
                new Statement("file:/a.rdf#reif",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#reif",RDFNS+"subject","file:/a.rdf#res",false),
                new Statement("file:/a.rdf#reif",RDFNS+"predicate","test:prop2",false),
                new Statement("file:/a.rdf#reif",RDFNS+"object","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#bag1",RDFNS+"_1","file:/a.rdf#genid2",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"type",RDFNS+"statement",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"subject","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"predicate","test:nestedprop2",false),
                new Statement("file:/a.rdf#genid2",RDFNS+"object","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testParseBag() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Bag rdf:ID='bag'>"+CR+
            "    <rdf:li>value1</rdf:li>"+
            "    <rdf:li rdf:resource='test:refres'/>"+CR+
            "    <rdf:li>"+CR+
            "        <rdf:Description rdf:about='res' test:prop1='value1'>"+CR+
            "            <test:prop2>value2</test:prop2>"+CR+
            "        </rdf:Description>"+CR+
            "    </rdf:li>"+CR+
            "    <rdf:li rdf:parseType='Resource'>"+CR+
            "        <test:nestedprop1>nested1</test:nestedprop1>"+CR+
            "    </rdf:li>"+CR+
            "    <rdf:li rdf:parseType='Literal'><test:nestedprop1>nested1</test:nestedprop1></rdf:li>"+CR+
            "</rdf:Bag>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#bag",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#bag",RDFNS+"_1","value1",true),
                new Statement("file:/a.rdf#bag",RDFNS+"_2","test:refres",false),
                new Statement("file:/a.rdf#bag",RDFNS+"_3","file:/res",false),
                new Statement("file:/a.rdf#bag",RDFNS+"_4","file:/a.rdf#genid1",false),
                new Statement("file:/a.rdf#bag",RDFNS+"_5","<test:nestedprop1>nested1</test:nestedprop1>",true),
                new Statement("file:/res","test:prop1","value1",true),
                new Statement("file:/res","test:prop2","value2",true),
                new Statement("file:/a.rdf#genid1","test:nestedprop1","nested1",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testParseContainers() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Bag rdf:ID='bag'>"+CR+
            "    <rdf:li>value1</rdf:li>"+
            "    <rdf:li rdf:resource='test:refres'/>"+CR+
            "</rdf:Bag>"+
            "<rdf:Seq rdf:ID='seq'>"+CR+
            "    <rdf:li>value1</rdf:li>"+
            "    <rdf:li rdf:resource='test:refres'/>"+CR+
            "</rdf:Seq>"+
            "<rdf:Alt rdf:ID='alt'>"+CR+
            "    <rdf:li>value1</rdf:li>"+
            "    <rdf:li rdf:resource='test:refres'/>"+CR+
            "</rdf:Alt>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#bag",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#bag",RDFNS+"_1","value1",true),
                new Statement("file:/a.rdf#bag",RDFNS+"_2","test:refres",false),
                new Statement("file:/a.rdf#seq",RDFNS+"type",RDFNS+"Seq",false),
                new Statement("file:/a.rdf#seq",RDFNS+"_1","value1",true),
                new Statement("file:/a.rdf#seq",RDFNS+"_2","test:refres",false),
                new Statement("file:/a.rdf#alt",RDFNS+"type",RDFNS+"Alt",false),
                new Statement("file:/a.rdf#alt",RDFNS+"_1","value1",true),
                new Statement("file:/a.rdf#alt",RDFNS+"_2","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testParseAnonymousContainer() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Bag>"+CR+
            "    <rdf:li>value1</rdf:li>"+
            "    <rdf:li rdf:resource='test:refres'/>"+CR+
            "</rdf:Bag>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#genid1",RDFNS+"type",RDFNS+"Bag",false),
                new Statement("file:/a.rdf#genid1",RDFNS+"_1","value1",true),
                new Statement("file:/a.rdf#genid1",RDFNS+"_2","test:refres",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testNoRDF() throws Exception {
        try {
            doTest("file:/a.rdf","<?xml version='1.0'?>"+CR,new Statement[0],"file:/a.rdf","file:/a.rdf");
            fail("Shouldn't be able to parse an empty document.");
        }
        catch (SAXParseException mustHappen) {
        }
    }
    public void testAboutEach() throws Exception {
        try {
            doTest("file:/a.rdf",RDF_START+
                "<rdf:Descrpition rdf:aboutEach='#id'>"+CR+
                "    <rdf:li>value1</rdf:li>"+
                "    <rdf:li rdf:resource='test:refres'/>"+CR+
                "</rdf:Descrpition>"+
                RDF_END,new Statement[0],"file:/a.rdf","file:/a.rdf");
            fail("Shouldn't be able to parse rdf:aboutEach.");
        }
        catch (RDFParseException mustHappen) {
            assertEquals("[line=6:column=-1] rdf:aboutEach attribute is not supported.",mustHappen.getMessage());
            assertEquals("file:/a.rdf",mustHappen.getSystemId());
            assertEquals(6,mustHappen.getLineNumber());
        }
    }
    public void testAboutEachPrefix() throws Exception {
        try {
            doTest("file:/a.rdf",RDF_START+
                "<rdf:Descrpition rdf:aboutEachPrefix='#id'>"+CR+
                "    <rdf:li>value1</rdf:li>"+
                "    <rdf:li rdf:resource='test:refres'/>"+CR+
                "</rdf:Descrpition>"+
                RDF_END,new Statement[0],"file:/a.rdf","file:/a.rdf");
            fail("Shouldn't be able to parse rdf:aboutEachPrefix.");
        }
        catch (RDFParseException mustHappen) {
            assertEquals("[line=6:column=-1] rdf:aboutEachPrefix attribute is not supported.",mustHappen.getMessage());
            assertEquals("file:/a.rdf",mustHappen.getSystemId());
            assertEquals(6,mustHappen.getLineNumber());
        }
    }
    public void testAboutAndID() throws Exception {
        try {
            doTest("file:/a.rdf",RDF_START+
                "<rdf:Descrpition rdf:ID='ID' rdf:about='#id'>"+CR+
                "    <rdf:li>value1</rdf:li>"+
                "    <rdf:li rdf:resource='test:refres'/>"+CR+
                "</rdf:Descrption>"+
                RDF_END,new Statement[0],"file:/a.rdf","file:/a.rdf");
            fail("Shouldn't be able to parse rdf:about and rdf:ID on the same node.");
        }
        catch (RDFParseException mustHappen) {
            assertEquals("[line=6:column=-1] Element cannot specify both rdf:ID and rdf:about attributes.",mustHappen.getMessage());
            assertEquals("file:/a.rdf",mustHappen.getSystemId());
            assertEquals(6,mustHappen.getLineNumber());
        }
    }
    public void testMisplacedText1() throws Exception {
        try {
            doTest("file:/a.rdf",RDF_START+
                "<rdf:Descrpition rdf:ID='ID'>"+CR+
                "    misplaced<rdf:li>value1</rdf:li>"+
                "    <rdf:li rdf:resource='test:refres'/>"+CR+
                "</rdf:Descrption>"+
                RDF_END,new Statement[0],"file:/a.rdf","file:/a.rdf");
            fail("Shouldn't be able to parse misplaced text.");
        }
        catch (RDFParseException mustHappen) {
            assertEquals("[line=7:column=-1] Cannot process characters when object properties are excepted.",mustHappen.getMessage());
            assertEquals("file:/a.rdf",mustHappen.getSystemId());
            assertEquals(7,mustHappen.getLineNumber());
        }
    }
    public void testMisplacedText2() throws Exception {
        try {
            doTest("file:/a.rdf",RDF_START+
                "misplaced<rdf:Descrpition rdf:ID='ID'>"+CR+
                "    <rdf:li>value1</rdf:li>"+
                "    <rdf:li rdf:resource='test:refres'/>"+CR+
                "</rdf:Descrption>"+
                RDF_END,new Statement[0],"file:/a.rdf","file:/a.rdf");
            fail("Shouldn't be able to parse misplaced text.");
        }
        catch (RDFParseException mustHappen) {
            assertEquals("[line=6:column=-1] Expecting an object element instead of character content.",mustHappen.getMessage());
            assertEquals("file:/a.rdf",mustHappen.getSystemId());
            assertEquals(6,mustHappen.getLineNumber());
        }
    }
    public void testLogicalURI() throws Exception {
        doTest("file:/c:/temp/test.kaon",XML_START+
            "<rdf:RDF xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'"+CR+
            "         xmlns:rdfs='http://www.w3.org/2000/01/rdf-schema#'"+CR+
            "         xmlns:test='test:'"+CR+
            "         xml:base='file:/q/w/e'"+
            ">"+CR+
            "<rdf:Description rdf:about='#id'>"+CR+
            "    <test:prop>value</test:prop>"+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/q/w/e#id","test:prop","value",true),
            },"file:/c:/temp/test.kaon","file:/q/w/e");
    }
    public void testURIResolutionFromOpaqueSource() throws Exception {
        doTest("urn:1234",RDF_START+
            "<rdf:Description rdf:about='#id'>"+CR+
            "    <test:prop>value</test:prop>"+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("#id","test:prop","value",true),
            },"urn:1234","urn:1234");
    }
    public void testOpaqueURIResolution() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:about='test:id'>"+CR+
            "    <test:prop>value</test:prop>"+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("test:id","test:prop","value",true),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testHierarchicalURIResolution() throws Exception {
        doTest("file:/b/a.rdf",RDF_START+
            "<rdf:Description rdf:about='../hh'>"+CR+
            "    <test:prop>value</test:prop>"+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/hh","test:prop","value",true),
            },"file:/b/a.rdf","file:/b/a.rdf");
    }
    public void testXMLBase() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description xml:base='a.rdf/c.rdf' rdf:about='hh'>"+CR+
            "    <test:prop xml:base='../d.rdf' rdf:resource='#res'/>"+
            "</rdf:Description>"+
            "<rdf:Description rdf:about='hh'>"+CR+
            "    <test:prop rdf:resource='#res'/>"+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf/hh","test:prop","file:/d.rdf#res",false),
                new Statement("file:/hh","test:prop","file:/a.rdf#res",false),
            },"file:/a.rdf","file:/a.rdf");
    }
    public void testIncludeRDFModels() throws Exception {
        doTest("file:/a.rdf",XML_START+
            "<?include-rdf physicalURI='physical:/uri1'?>"+CR+
            "<?include-rdf physicalURI='physical:/uri2'?>"+CR+
            "<?include-rdf logicalURI='logical:/uri2'?>"+CR+
            "<?include-rdf logicalURI='logical:/uri2' physicalURI='physical:/uri23'?>"+CR+
            RDF_START_INTERNAL+
            "<rdf:Description xml:base='a.rdf/c.rdf' rdf:about='hh'>"+CR+
            "    <test:prop xml:base='../d.rdf' rdf:resource='#res'/>"+
            "</rdf:Description>"+
            "<rdf:Description rdf:about='hh'>"+CR+
            "    <test:prop rdf:resource='#res'/>"+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf/hh","test:prop","file:/d.rdf#res",false),
                new Statement("file:/hh","test:prop","file:/a.rdf#res",false),
            },"file:/a.rdf","file:/a.rdf",new String[][] {
                { null,"physical:/uri1" },
                { null,"physical:/uri2" },
                { "logical:/uri2",null },
                { "logical:/uri2","physical:/uri23" },
            });
    }
    public void testParseTypeCollection() throws Exception {
        doTest("file:/c:/temp/collections.txt",RDF_START+
            "<rdf:Description rdf:about='http://example.edu/courses/6.001'>"+CR+
            "   <test:students rdf:parseType='Collection'>"+CR+
            "       <test:Student rdf:about='http://example.edu/students/Amy'/>"+CR+
            "       <rdf:Description rdf:about='http://example.edu/students/Tim'/>"+CR+
            "   </test:students>"+CR+
            "   <test:value>value</test:value>"+CR+
            "</rdf:Description>"+CR+
            "<rdf:Description rdf:about='http://example.edu/courses/6.002'>"+CR+
            "   <test:students rdf:parseType='Collection'/>"+CR+
            "</rdf:Description>"+CR+
            "<rdf:Description rdf:about='http://example.edu/courses/6.003'>"+CR+
            "   <test:students rdf:parseType='Collection' rdf:ID='reifid_1'>"+CR+
            "       <rdf:Description rdf:about='http://example.edu/students/John'/>"+CR+
            "   </test:students>"+CR+
            "</rdf:Description>"+CR+
            "<rdf:Description rdf:about='http://example.edu/courses/6.004'>"+CR+
            "   <test:students rdf:parseType='Collection' rdf:ID='reifid_2'/>"+CR+
            "</rdf:Description>"+CR+
            RDF_END,new Statement[] {
                new Statement("http://example.edu/courses/6.001","test:students","file:/c:/temp/collections.txt#genid1",false),
                new Statement("file:/c:/temp/collections.txt#genid1","http://www.w3.org/1999/02/22-rdf-syntax-ns#type","http://www.w3.org/1999/02/22-rdf-syntax-ns#List",false),
                new Statement("file:/c:/temp/collections.txt#genid1","http://www.w3.org/1999/02/22-rdf-syntax-ns#first","http://example.edu/students/Amy",false),
                new Statement("http://example.edu/students/Amy","http://www.w3.org/1999/02/22-rdf-syntax-ns#type","test:Student",false),
                new Statement("file:/c:/temp/collections.txt#genid1","http://www.w3.org/1999/02/22-rdf-syntax-ns#rest","file:/c:/temp/collections.txt#genid2",false),
                new Statement("file:/c:/temp/collections.txt#genid2","http://www.w3.org/1999/02/22-rdf-syntax-ns#type","http://www.w3.org/1999/02/22-rdf-syntax-ns#List",false),
                new Statement("file:/c:/temp/collections.txt#genid2","http://www.w3.org/1999/02/22-rdf-syntax-ns#first","http://example.edu/students/Tim",false),
                new Statement("file:/c:/temp/collections.txt#genid2","http://www.w3.org/1999/02/22-rdf-syntax-ns#rest","http://www.w3.org/1999/02/22-rdf-syntax-ns#nil",false),
                new Statement("http://example.edu/courses/6.001","test:value","value",true),
                new Statement("http://example.edu/courses/6.002","test:students","http://www.w3.org/1999/02/22-rdf-syntax-ns#nil",false),
                new Statement("http://example.edu/courses/6.003","test:students","file:/c:/temp/collections.txt#genid3",false),
                new Statement("file:/c:/temp/collections.txt#reifid_1","http://www.w3.org/1999/02/22-rdf-syntax-ns#subject","http://example.edu/courses/6.003",false),
                new Statement("file:/c:/temp/collections.txt#reifid_1","http://www.w3.org/1999/02/22-rdf-syntax-ns#type","http://www.w3.org/1999/02/22-rdf-syntax-ns#statement",false),
                new Statement("file:/c:/temp/collections.txt#reifid_1","http://www.w3.org/1999/02/22-rdf-syntax-ns#object","file:/c:/temp/collections.txt#genid3",false),
                new Statement("file:/c:/temp/collections.txt#reifid_1","http://www.w3.org/1999/02/22-rdf-syntax-ns#predicate","test:students",false),
                new Statement("file:/c:/temp/collections.txt#genid3","http://www.w3.org/1999/02/22-rdf-syntax-ns#type","http://www.w3.org/1999/02/22-rdf-syntax-ns#List",false),
                new Statement("file:/c:/temp/collections.txt#genid3","http://www.w3.org/1999/02/22-rdf-syntax-ns#first","http://example.edu/students/John",false),
                new Statement("file:/c:/temp/collections.txt#genid3","http://www.w3.org/1999/02/22-rdf-syntax-ns#rest","http://www.w3.org/1999/02/22-rdf-syntax-ns#nil",false),
                new Statement("http://example.edu/courses/6.004","test:students","http://www.w3.org/1999/02/22-rdf-syntax-ns#nil",false),
                new Statement("file:/c:/temp/collections.txt#reifid_2","http://www.w3.org/1999/02/22-rdf-syntax-ns#subject","http://example.edu/courses/6.004",false),
                new Statement("file:/c:/temp/collections.txt#reifid_2","http://www.w3.org/1999/02/22-rdf-syntax-ns#type","http://www.w3.org/1999/02/22-rdf-syntax-ns#statement",false),
                new Statement("file:/c:/temp/collections.txt#reifid_2","http://www.w3.org/1999/02/22-rdf-syntax-ns#object","http://www.w3.org/1999/02/22-rdf-syntax-ns#nil",false),
                new Statement("file:/c:/temp/collections.txt#reifid_2","http://www.w3.org/1999/02/22-rdf-syntax-ns#predicate","test:students",false),
            },"file:/c:/temp/collections.txt","file:/c:/temp/collections.txt");
    }
    public void testNodeID() throws Exception {
        doTest("file:/a.rdf",RDF_START+
            "<rdf:Description rdf:nodeID='someid'>"+CR+
            "    <test:prop1 rdf:nodeID='someid'/>"+CR+
            "</rdf:Description>"+
            RDF_END,new Statement[] {
                new Statement("file:/a.rdf#someid","test:prop1","file:/a.rdf#someid",false),
            },"file:/a.rdf","file:/a.rdf");
    }

    protected static class Statement {
        public String m_subject;
        public String m_predicate;
        public String m_object;
        public boolean m_objectIsLiteral;
        public int m_hashCode;

        public Statement(String subject,String predicate,String object,boolean objectIsLiteral) {
            m_subject=subject;
            m_predicate=predicate;
            m_object=object;
            m_objectIsLiteral=objectIsLiteral;
            m_hashCode=(m_subject+"%"+m_predicate+"%"+m_object+"%"+m_objectIsLiteral).hashCode();
        }
        public int hashCode() {
            return m_hashCode;
        }
        public boolean equals(Object that) {
            if (that instanceof Statement) {
                Statement statement=(Statement)that;
                return m_subject.equals(statement.m_subject) && m_predicate.equals(statement.m_predicate) && m_object.equals(statement.m_object) && m_objectIsLiteral==statement.m_objectIsLiteral;
            }
            return false;
        }
    }
}
