package edu.unika.aifb.rdf.mainmemory.test;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import junit.framework.TestCase;
import java.io.StringReader;
import org.xml.sax.InputSource;

import edu.unika.aifb.rdf.api.util.*;
import edu.unika.aifb.rdf.api.syntax.*;
import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.mainmemory.*;

/**
 * Test cases for RDF modularization.
 */
public class TestModularization extends TestCase {
    protected static final String CR=System.getProperty("line.separator");

    protected static final String RDF="http://www.w3.org/1999/02/22-rdf-syntax-ns#";
    protected static final String RDFS="http://www.w3.org/2000/01/rdf-schema#";

    protected static final String RDF_START=
        "<?xml version='1.0' encoding='ISO-8859-1'?>"+CR+
        "<rdf:RDF xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'"+CR+
        "         xmlns:rdfs='http://www.w3.org/2000/01/rdf-schema#'"+CR+
        "         xmlns:kaon='http://kaon.semanticweb.org/2001/11/kaon-lexical#'"+CR+
        "         xmlns:test='test:'"+CR+
        ">"+CR;

    protected static final String MODEL=
        RDF_START+
        "<rdfs:Class rdf:about='test:Root'/>"+CR+
        "<rdfs:Class rdf:about='test:A'>"+CR+
        "    <rdfs:subClassOf rdf:resource='test:Root'/>"+CR+
        "</rdfs:Class>"+
        "<rdf:Description rdf:about='test:subject1'>"+CR+
        "    <test:predicate1>value1</test:predicate1>"+CR+
        "    <test:predicate2 rdf:resource='test:value2'/>"+CR+
        "</rdf:Description>"+CR+
        "</rdf:RDF>";

    protected static final String INCLUDED1=
        RDF_START+
        "<test:Something rdf:about='test:value2'>"+CR+
        "    <test:value2_value>value</test:value2_value>"+CR+
        "</test:Something>"+CR+
        "</rdf:RDF>";

    protected static final String INCLUDED2=
        RDF_START+
        "<rdf:Description rdf:about='test:subject55'>"+CR+
        "    <test:predicate55>value55</test:predicate55>"+CR+
        "</rdf:Description>"+CR+
        "</rdf:RDF>";

    protected static final String INCLUDED3=
        RDF_START+
        "<rdf:Description rdf:about='test:subject1'>"+CR+
        "    <test:predicate1>value1</test:predicate1>"+CR+
        "</rdf:Description>"+CR+
        "</rdf:RDF>";

    protected NodeFactory m_nodeFactory;
    protected Model m_model;
    protected Model m_included1;
    protected Model m_included2;
    protected Model m_included3;

    public TestModularization(String name) {
        super(name);
    }
    protected void assertContains(Model model,String subject,String predicate,String object,boolean isResource) throws Exception {
        NodeFactory nodeFactory=model.getNodeFactory();
        Statement statement;
        if (isResource)
            statement=nodeFactory.createStatement(nodeFactory.createResource(subject),nodeFactory.createResource(predicate),nodeFactory.createResource(object));
        else
            statement=nodeFactory.createStatement(nodeFactory.createResource(subject),nodeFactory.createResource(predicate),nodeFactory.createLiteral(object));
        assertTrue(model.contains(statement));
    }
    protected void assertNotContains(Model model,String subject,String predicate,String object,boolean isResource) throws Exception {
        NodeFactory nodeFactory=model.getNodeFactory();
        Statement statement;
        if (isResource)
            statement=nodeFactory.createStatement(nodeFactory.createResource(subject),nodeFactory.createResource(predicate),nodeFactory.createResource(object));
        else
            statement=nodeFactory.createStatement(nodeFactory.createResource(subject),nodeFactory.createResource(predicate),nodeFactory.createLiteral(object));
        assertTrue(!model.contains(statement));
    }
    protected Model parseModel(String rdf) throws Exception {
        InputSource inputSource=new InputSource(new StringReader(rdf));
        inputSource.setSystemId("file:/a.rdf");
        Model model=new TransactionableModelImpl(m_nodeFactory);
        RDFParser parser=new RDFParser();
        parser.parse(inputSource,new ModelConsumer(model));
        return model;
    }
    protected void setUp() throws Exception {
        m_nodeFactory=new NodeFactoryImpl();
        m_model=parseModel(MODEL);
        m_included1=parseModel(INCLUDED1);
        m_included2=new TransactionableModelImpl(m_nodeFactory);
        m_included3=parseModel(INCLUDED3);
        m_model.addIncludedModel(m_included1);
        m_model.addIncludedModel(m_included2);
        m_included2.addIncludedModel(m_included3);
    }
    public void testModelContents() throws Exception {
        ModelTester tester=new ModelTester(m_model);
        tester.statement("test:Root",RDF+"type",RDFS+"Class",true);
        tester.statement("test:A",RDF+"type",RDFS+"Class",true);
        tester.statement("test:A",RDFS+"subClassOf","test:Root",true);
        tester.statement("test:subject1","test:predicate1","value1",false);
        tester.statement("test:subject1","test:predicate2","test:value2",true);
        tester.statement("test:value2",RDF+"type","test:Something",true);
        tester.statement("test:value2","test:value2_value","value",false);
        tester.statement("test:subject1","test:predicate1","value1",false);
        tester.test();
    }
    public void testIncluded1Contents() throws Exception {
        ModelTester tester=new ModelTester(m_included1);
        tester.statement("test:value2",RDF+"type","test:Something",true);
        tester.statement("test:value2","test:value2_value","value",false);
        tester.test();
    }
    public void testIncluded2Contents() throws Exception {
        ModelTester tester=new ModelTester(m_included2);
        tester.statement("test:subject1","test:predicate1","value1",false);
        tester.test();
    }
    public void testIncluded3Contents() throws Exception {
        ModelTester tester=new ModelTester(m_included3);
        tester.statement("test:subject1","test:predicate1","value1",false);
        tester.test();
    }
    public void testAddExistingStatement() throws Exception {
        NodeFactory nodeFactory=m_model.getNodeFactory();
        Statement statement=nodeFactory.createStatement(nodeFactory.createResource("test:subject1"),nodeFactory.createResource("test:predicate1"),nodeFactory.createLiteral("value1"));
        try {
            m_model.add(statement);
            fail();
        }
        catch (ModelException mustHappen) {
        }
    }
    public void testAddNonExistingStatement() throws Exception {
        NodeFactory nodeFactory=m_included1.getNodeFactory();
        Statement statement=nodeFactory.createStatement(nodeFactory.createResource("test:X"),nodeFactory.createResource("test:Y"),nodeFactory.createLiteral("Z"));
        m_included1.add(statement);
        assertContains(m_model,"test:X","test:Y","Z",false);
        assertEquals(9,m_model.size());
    }
    public void testRemoveExistingStatement() throws Exception {
        NodeFactory nodeFactory=m_included1.getNodeFactory();
        Statement statement=nodeFactory.createStatement(nodeFactory.createResource("test:value2"),nodeFactory.createResource("test:value2_value"),nodeFactory.createLiteral("value"));
        m_included1.remove(statement);
        assertNotContains(m_model,"test:value2","test:value2_value","value",false);
        assertEquals(7,m_model.size());
    }
    public void testRemoveNonExistingStatement() throws Exception {
        NodeFactory nodeFactory=m_included1.getNodeFactory();
        Statement statement=nodeFactory.createStatement(nodeFactory.createResource("test:value2"),nodeFactory.createResource("test:value2_value"),nodeFactory.createLiteral("Z"));
        try {
            m_model.remove(statement);
            fail();
        }
        catch (ModelException mustHappen) {
        }
    }
    public void testAddingIncludeIndirect() throws Exception {
        Model model=parseModel(MODEL);
        Model included1=parseModel(INCLUDED1);
        Model included2=parseModel(INCLUDED2);
        model.addIncludedModel(included1);
        ModelTester tester=new ModelTester(model);
        tester.statement("test:Root",RDF+"type",RDFS+"Class",true);
        tester.statement("test:A",RDF+"type",RDFS+"Class",true);
        tester.statement("test:A",RDFS+"subClassOf","test:Root",true);
        tester.statement("test:subject1","test:predicate1","value1",false);
        tester.statement("test:subject1","test:predicate2","test:value2",true);
        tester.statement("test:value2",RDF+"type","test:Something",true);
        tester.statement("test:value2","test:value2_value","value",false);
        tester.testContents();
        tester.testIterator();
        included1.addIncludedModel(included2);
        tester.statement("test:subject55","test:predicate55","value55",false);
        tester.testContents();
        tester.testIterator();
    }

    protected static class ModelTester {
        protected Model m_model;
        protected List m_list;

        public ModelTester(Model model) {
            m_model=model;
            m_list=new ArrayList();
        }
        public void statement(String subject,String predicate,String object,boolean isResource) throws Exception {
            NodeFactory nodeFactory=m_model.getNodeFactory();
            Statement statement;
            if (isResource)
                statement=nodeFactory.createStatement(nodeFactory.createResource(subject),nodeFactory.createResource(predicate),nodeFactory.createResource(object));
            else
                statement=nodeFactory.createStatement(nodeFactory.createResource(subject),nodeFactory.createResource(predicate),nodeFactory.createLiteral(object));
            m_list.add(statement);
        }
        public void test() throws Exception {
            testContents();
        }
        public void testContents() throws Exception {
            assertEquals(m_list.size(),m_model.size());
            Iterator iterator=m_list.iterator();
            while (iterator.hasNext()) {
                Statement statement=(Statement)iterator.next();
                assertTrue("Model should contain statement "+statement,m_model.contains(statement));
            }
        }
        public void testIterator() throws Exception {
            List list=new ArrayList(m_list);
            Iterator iterator=m_model.iterator();
            while (iterator.hasNext()) {
                Statement statement=(Statement)iterator.next();
                int index=list.indexOf(statement);
                assertTrue("Statement "+statement+" not found in control set",index!=-1);
                list.remove(index);
            }
            assertTrue("Control set has more elements",list.isEmpty());
        }
    }
}
