package edu.unika.aifb.rdf.mainmemory;

import java.util.Iterator;
import java.util.Map;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Set;
import java.util.AbstractSet;
import java.util.NoSuchElementException;
import java.lang.ref.WeakReference;
import java.lang.ref.ReferenceQueue;

/**
 * Implementation of the <code>Map</code> interface with <em>weak values<em>. An entry is kept in this map
 * until there is a strong reference to the value in the map.
 */
public class WeakValueHashMap extends AbstractMap implements Map {
    /* Hash table mapping WeakKeys to values */
    private Map hash;
    /* Reference queue for cleared WeakKeys */
    private ReferenceQueue queue=new ReferenceQueue();
    /** Entry set view of this map. */
    private Set entrySet=null;

    /**
     * Constructs a new, empty <code>WeakHashMap</code> with the given
     * initial capacity and the given load factor.
     *
     * @param initialCapacity           initial capacity of the <code>WeakHashMap</code>
     * @param loadFactor                load factor of the <code>WeakHashMap</code>
     * @throws IllegalArgumentException if the initial capacity is less than zero, or if the load factor is nonpositive
     */
    public WeakValueHashMap(int initialCapacity,float loadFactor) {
    	hash=new HashMap(initialCapacity,loadFactor);
    }
    /**
     * Constructs a new, empty <code>WeakHashMap</code> with the given initial capacity and the default
     * load factor, which is <code>0.75</code>.
     *
     * @param initialCapacity           initial capacity of the <code>WeakHashMap</code>
     * @throws IllegalArgumentException if the initial capacity is less than zero
     */
    public WeakValueHashMap(int initialCapacity) {
    	hash=new HashMap(initialCapacity);
    }
    /**
     * Constructs a new, empty <code>WeakHashMap</code> with the default initial capacity and the default
     * load factor, which is <code>0.75</code>.
     */
    public WeakValueHashMap() {
    	hash=new HashMap();
    }
    /**
     * Constructs a new <code>WeakHashMap</code> with the same mappings as the
     * specified <tt>Map</tt>.  The <code>WeakHashMap</code> is created with an
     * initial capacity of twice the number of mappings in the specified map
     * or 11 (whichever is greater), and a default load factor, which is
     * <tt>0.75</tt>.
     *
     * @param map                       map whose mappings are to be placed in this map.
     */
    public WeakValueHashMap(Map map) {
	    this(Math.max(2*map.size(),11),0.75f);
    	putAll(map);
    }
    /**
     * Returns <code>true</code> if this map contains a mapping for the
     * specified key.
     *
     * @param key                       key whose presence in this map is to be tested
     */
    public boolean containsKey(Object key) {
    	return hash.containsKey(key);
    }
    /**
     * Returns the value to which this map maps the specified <code>key</code>.
     * If this map does not contain a value for this key, then return
     * <code>null</code>.
     *
     * @param key                       key whose associated value, if any, is to be returned
     */
    public Object get(Object key) {
    	WeakValue weakValue=(WeakValue)hash.get(key);
        if (weakValue==null)
            return null;
        return weakValue.get();
    }
    /**
     * Updates this map so that the given <code>key</code> maps to the given
     * <code>value</code>.  If the map previously contained a mapping for
     * <code>key</code> then that mapping is replaced and the previous value is
     * returned.
     *
     * @param key                       key that is to be mapped to the given <code>value</code>
     * @param value                     value to which the given <code>key</code> is to be mapped
     * @return                          previous value to which this key was mapped, or <code>null</code> if if there was no mapping for the key
     */
    public Object put(Object key,Object value) {
        processQueue();
        WeakValue weakValue=(WeakValue)hash.put(key,value==null ? null : new WeakValue(value,queue,key));
        if (weakValue==null)
            return null;
        return weakValue.get();
    }
    /**
     * Removes the mapping for the given <code>key</code> from this map, if present.
     *
     * @param key                       key whose mapping is to be removed
     * @return                          value to which this key was mapped, or <code>null</code> if there was no mapping for the key
     */
    public Object remove(Object key) {
        processQueue();
        WeakValue weakValue=(WeakValue)hash.remove(key);
        if (weakValue==null)
            return null;
        return weakValue.get();
    }
    /**
     * Removes all mappings from this map.
     */
    public void clear() {
        processQueue();
        hash.clear();
    }
    /**
     * Returns a <code>Set</code> view of the mappings in this map.
     */
    public Set entrySet() {
        if (entrySet==null)
            entrySet=new WeakEntrySet();
        return entrySet;
    }
    /**
     * Removes all invalidated entries from the map, that is, removes all entries
     * whose keys have been discarded. This method should be invoked once by
     * each public mutator in this class. We don't invoke this method in
     * public accessors because that can lead to surprising
     * ConcurrentModificationExceptions.
     */
    private void processQueue() {
        WeakValue value;
        while ((value=(WeakValue)queue.poll())!=null)
            hash.remove(value.key);
    }


    /**
     * Value elements stored in the actual hash map.
     */
    private static class WeakValue extends WeakReference {
        /** Key of the weak value. */
        public Object key;

        /**
         * Creates a weak value.
         *
         * @param value             value to be stored in the hash-table
         * @param queue             reference queue where the reference is enqueued before it is garbage-collected
         * @param key               key of the value
         */
        public WeakValue(Object value,ReferenceQueue queue,Object key) {
            super(value,queue);
            this.key=key;
        }
    }

    /**
     * Entry in this map.
     */
    private static class WeakEntry implements Map.Entry {
        /** Actual entry in the parallel map. */
    	protected Map.Entry entry;
        /* Strong reference to value, so that the GC will leave it alone as long as this WeakEntry exists. */
	    protected Object value;

        /**
         * Creates a weak entry and attaches it to the parallel entry in the <code>Map</code>.
         */
        protected WeakEntry(Map.Entry entry,Object value) {
            this.entry=entry;
            this.value=value;
        }
        /**
         * Returns the key of this entry.
         */
        public Object getKey() {
            return entry.getKey();
        }
        /**
         * Returns the value of this entry.
         */
        public Object getValue() {
            return value;
        }
        /**
         * Sets the value of this entry.
         */
        public Object setValue(Object value) {
            this.value=value;
            return entry.setValue(value);
        }
        /**
         * Checks if objects are equal, by taking into account <code>null</code> reference semantics.
         */
        private static boolean objectEquals(Object o1,Object o2) {
            return (o1==null) ? (o2==null) : o1.equals(o2);
        }
        /**
         * Checks if two entries are equal.
         */
        public boolean equals(Object other) {
            if (!(other instanceof Map.Entry))
                return false;
            Map.Entry otherEntry=(Map.Entry)other;
            return objectEquals(getKey(),otherEntry.getKey()) && objectEquals(getValue(),otherEntry.getValue());
        }
        /**
         * Entry hash code.
         */
        public int hashCode() {
            Object key=getKey();
            return (key==null ? 0 : key.hashCode()) ^ (value==null ? 0 : value.hashCode());
        }
    }

    /**
     * Implements a set of weak entries for this map.
     */
    private class WeakEntrySet extends AbstractSet {
        /** Entry set of the parallel map. */
        protected Set hashEntrySet=hash.entrySet();

        /**
         * Creates an iterator over this set.
         */
        public Iterator iterator() {
            return new Iterator() {
                Iterator hashIterator=hashEntrySet.iterator();
                Map.Entry next=advance();

                public boolean hasNext() {
                    return next!=null;
                }
                public Object next() {
                    if (next==null)
                        throw new NoSuchElementException();
                    Map.Entry entry=next;
                    next=advance();
                    return entry;
                }
                public void remove() {
                    hashIterator.remove();
                }
                protected Map.Entry advance() {
                    while (hashIterator.hasNext()) {
                        Map.Entry entry=(Map.Entry)hashIterator.next();
                        WeakValue weakValue=(WeakValue)entry.getValue();
                        Object value=null;
                        // skip entries containing values eaten by the garbage collector
                        if (weakValue==null || (value=weakValue.get())!=null)
                            return new WeakEntry(entry,value);
                    }
                    return null;
                }
            };
        }
        /**
         * Tests whether set is empty.
         */
        public boolean isEmpty() {
            // it is not sufficient to test the size only, since entries mighe have lost their values in the meanwhile
            return !iterator().hasNext();
        }
        /**
         * Returns the size of this set.
         */
        public int size() {
            int count=0;
            Iterator iterator=hashEntrySet.iterator();
            while (iterator.hasNext()) {
                Map.Entry entry=(Map.Entry)iterator.next();
                WeakValue weakValue=(WeakValue)entry.getValue();
                if (weakValue==null || weakValue.get()!=null)
                    count++;
            }
            return count;
        }
        /**
         * Removes an object from the set and returns <code>true</code> if the set actually contained the object.
         */
        public boolean remove(Object object) {
            processQueue();
            if (!(object instanceof Map.Entry))
                return false;
            Map.Entry entry=(Map.Entry)object;
            Object key=entry.getKey();
            WeakValue weakValue=(WeakValue)entry.getValue();
            Object value=(weakValue==null ? null : weakValue.get());
            Object hashValue=hash.get(key);
            if (hashValue==null ? (value==null && hash.containsKey(key)) : hashValue.equals(value)) {
                hash.remove(key);
                return true;
            }
            return false;
        }
        /**
         * Returns the hash code of the set.
         */
        public int hashCode() {
            int hashCode=0;
            for (Iterator iterator=hashEntrySet.iterator();iterator.hasNext();) {
                Map.Entry entry=(Map.Entry)iterator.next();
                WeakValue weakValue=(WeakValue)entry.getValue();
                Object value=null;
                if (weakValue!=null && (value=weakValue.get())==null)
                    continue;
                hashCode+=(entry.getKey().hashCode() ^ (value==null ? 0 : value.hashCode()));
            }
            return hashCode;
        }
    }
}
