package edu.unika.aifb.rdf.mainmemory;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

import edu.unika.aifb.rdf.api.model.*;

public class TransactionableModelImpl extends ModelImpl {
    protected List m_rollbackLog;

    public TransactionableModelImpl(NodeFactory nodeFactory) {
        super(nodeFactory);
    }
    public void add(Statement statement) throws ModelException {
        nonTransactionalAdd(statement);
        if (m_rollbackLog!=null)
            m_rollbackLog.add(0,new RemoveStatementRollbackLogEntry(statement));
    }
    public void remove(Statement statement) throws ModelException {
        nonTransactionalRemove(statement);
        if (m_rollbackLog!=null)
            m_rollbackLog.add(0,new AddStatementRollbackLogEntry(statement));
    }
    public boolean supportsTransactions() {
        return true;
    }
    public void setAutocommit(boolean autocommit) throws ModelException {
        if (autocommit) {
            if (m_rollbackLog!=null && m_rollbackLog.size()!=0)
                throw new ModelException("Autocommit mode cannot be changed during a transaction.");
            m_rollbackLog=null;
        }
        else
            m_rollbackLog=new LinkedList();
    }
    public boolean isAutocommit() {
        return m_rollbackLog==null;
    }
    public void commit() {
        if (m_rollbackLog!=null)
            m_rollbackLog.clear();
    }
    public void rollback() throws ModelException {
        if (m_rollbackLog!=null)
            try {
                Iterator logElements=m_rollbackLog.iterator();
                while (logElements.hasNext()) {
                    RollbackLogEntry entry=(RollbackLogEntry)logElements.next();
                    entry.apply(this);
                }
            }
            finally {
                m_rollbackLog.clear();
            }
    }
    protected void nonTransactionalAdd(Statement statement) throws ModelException {
        super.add(statement);
    }
    protected void nonTransactionalRemove(Statement statement) throws ModelException {
        super.remove(statement);
    }

    protected static abstract class RollbackLogEntry {
        protected Statement m_statement;

        public RollbackLogEntry(Statement statement) {
            m_statement=statement;
        }
        public abstract void apply(TransactionableModelImpl model) throws ModelException;
    }

    protected static class RemoveStatementRollbackLogEntry extends RollbackLogEntry {

        public RemoveStatementRollbackLogEntry(Statement statement) {
            super(statement);
        }
        public void apply(TransactionableModelImpl model) throws ModelException {
            model.nonTransactionalRemove(m_statement);
        }
    }

    protected static class AddStatementRollbackLogEntry extends RollbackLogEntry {

        public AddStatementRollbackLogEntry(Statement statement) {
            super(statement);
        }
        public void apply(TransactionableModelImpl model) throws ModelException {
            model.nonTransactionalAdd(m_statement);
        }
    }

}
