package edu.unika.aifb.rdf.mainmemory;

import java.net.URI;
import java.net.URL;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.InputSource;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.util.*;

/**
 * RDF factory for in-memory RDF models.
 */
public class RDFFactoryImpl implements RDFFactory {
    public static final RDFFactoryImpl INSTANCE=new RDFFactoryImpl();
    static {
        RDFManager.registerFactory(INSTANCE);
    }

    /**
     * Creates an instance of this class.
     */
    protected RDFFactoryImpl() {
    }
    /**
     * Create an empty model. If this factory cannot service this URI, <code>null</code> should be returned.
     *
     * @param uri           URI of the model
     * @param nodeFactory   the node factory for the new model (may be <code>null</code>)
     * @return              new model or <code>null</code> if model with given URI cannot be created by this implementation
     */
    public Model createModel(String uri,NodeFactory nodeFactory) throws ModelException {
        try {
            new URL(uri);
        }
        catch (MalformedURLException e) {
            return null;
        }
        if (nodeFactory==null)
            nodeFactory=new NodeFactoryImpl();
        Model model=new TransactionableModelImpl(nodeFactory);
        model.setPhysicalURI(uri);
        model.setLogicalURI(uri);
        return model;
    }
    /**
     * Opens an existing model. If this factory cannot service this URI, <code>null</code> should be returned.
     *
     * @param uri           URI of the model
     * @param nodeFactory   the node factory for the new model (may be <code>null</code>)
     * @return              model information or <code>null</code> if model with given URI cannot be created by this implementation
     */
    public RDFManager.ModelInfo openModelEx(String uri,NodeFactory nodeFactory) throws ModelException {
        URL url;
        try {
            url=new URL(uri);
        }
        catch (MalformedURLException e) {
            return null;
        }
        try {
            InputStream inputStream=url.openStream();
            try {
                return openModelEx(uri,nodeFactory,inputStream,null);
            }
            finally {
                inputStream.close();
            }
        }
        catch (IOException e) {
            throw new ModelException("I/O error reading RDF XML data.",e);
        }
    }
    /**
     * Opens an existing model. If this factory cannot service this URI, <code>null</code> should be returned.
     *
     * @param uri           URI of the model
     * @param nodeFactory   the node factory for the new model (may be <code>null</code>)
     * @param inputStream   the input stream
     * @param modelConsumer the model consumer (may be <code>null</code>, but if not, then it can be used to transform the ontology while reading)
     * @return              model information or <code>null</code> if model with given URI cannot be created by this implementation
     */
    public RDFManager.ModelInfo openModelEx(String uri,NodeFactory nodeFactory,InputStream inputStream,ModelConsumer modelConsumer) throws ModelException {
        if (nodeFactory==null)
            nodeFactory=new NodeFactoryImpl();
        if (modelConsumer==null)
            modelConsumer=new ModelConsumer();
        RDFManager.ModelInfo modelInfo=new RDFManager.ModelInfo();
        modelInfo.m_model=new TransactionableModelImpl(nodeFactory);
        modelConsumer.initialize(modelInfo.m_model);
        try {
            InputSource inputSource=new InputSource(inputStream);
            inputSource.setSystemId(uri);
            RDFManager.createParser().parse(inputSource,modelConsumer);
        }
        catch (SAXParseException e) {
            throw new ModelException("[line="+e.getLineNumber()+", column="+e.getColumnNumber()+"] Error Parsing XML data.",e);
        }
        catch (SAXException e) {
            throw new ModelException("Error parsing RDF XML data.",e);
        }
        catch (IOException e) {
            throw new ModelException("I/O error reading RDF XML data.",e);
        }
        List includedModels=modelConsumer.getIncludedModels();
        modelInfo.m_includedModels=new String[includedModels.size()][2];
        includedModels.toArray(modelInfo.m_includedModels);
        return modelInfo;
    }
    /**
     * Deletes an existing model. If this factory cannot service this URI, <code>false</code> should be returned.
     *
     * @param uri           URI of the model
     * @return              <code>true</code> if this factory could delete the model
     */
    public boolean deleteModel(String uri) throws ModelException {
        try {
            File file=new File(new URI(uri));
            if (file.exists())
                if (!file.delete())
                    throw new ModelException("Cannot RDF model denoted by "+uri.toString());
            return true;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
        catch (URISyntaxException e) {
            return false;
        }
    }
}
