package edu.unika.aifb.rdf.mainmemory;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Collection;
import java.util.Arrays;
import java.util.NoSuchElementException;

import edu.unika.aifb.rdf.api.model.*;

public abstract class AbstractModel implements Model {
    protected NodeFactory m_nodeFactory;
    protected List m_includedModels;
    protected Model[] m_allIncludedModels;
    protected int[] m_inclusionIndexes;
    protected int m_inclusionIndex;

    public AbstractModel(NodeFactory nodeFactory) {
        m_nodeFactory=nodeFactory;
        m_includedModels=new ArrayList();
        m_inclusionIndex=0;
        m_allIncludedModels=new Model[0];
        m_inclusionIndexes=new int[0];
    }
    public NodeFactory getNodeFactory() {
        return m_nodeFactory;
    }
    public int getInclusionIndex() {
        return m_inclusionIndex;
    }
    public Collection getIncludedModels() {
        return new ArrayList(m_includedModels);
    }
    public void addIncludedModel(Model model) throws ModelException {
        if (!m_includedModels.contains(model)) {
            m_includedModels.add(model);
            updateAllIncludedModels();
            m_inclusionIndex++;
        }
    }
    public void removeIncludedModel(Model model) throws ModelException {
        if (m_includedModels.contains(model)) {
            m_includedModels.remove(model);
            updateAllIncludedModels();
            m_inclusionIndex++;
        }
    }
    protected void updateAllIncludedModels() throws ModelException {
        // this is the fix-point algorithm that allows us to have recusive inclusions
        Set result=new HashSet();
        result.add(this);
        Set newModels=new HashSet(result);
        int oldSize=-1;
        while (oldSize!=result.size()) {
            oldSize=result.size();
            Iterator iterator=new HashSet(newModels).iterator();
            newModels.clear();
            while (iterator.hasNext()) {
                Model includedModel=(Model)iterator.next();
                Collection includedModels=includedModel.getIncludedModels();
                newModels.addAll(includedModels);
                result.addAll(includedModels);
            }
        }
        result.remove(this);
        m_allIncludedModels=new Model[result.size()];
        result.toArray(m_allIncludedModels);
        m_inclusionIndexes=new int[m_allIncludedModels.length];
        for (int i=0;i<m_allIncludedModels.length;i++)
            m_inclusionIndexes[i]=m_allIncludedModels[i].getInclusionIndex();
    }
    protected void checkIncludedModelVersions() throws ModelException {
        for (int i=0;i<m_allIncludedModels.length;i++)
            if (m_inclusionIndexes[i]!=m_allIncludedModels[i].getInclusionIndex()) {
                updateAllIncludedModels();
                return;
            }
    }
    public Collection getAllIncludedModels() throws ModelException {
        checkIncludedModelVersions();
        return Arrays.asList(m_allIncludedModels);
    }
    public int size() throws ModelException {
        checkIncludedModelVersions();
        int size=thisSize();
        for (int i=m_allIncludedModels.length-1;i>=0;--i)
            size+=m_allIncludedModels[i].thisSize();
        return size;
    }
    public boolean isEmpty() throws ModelException {
        if (!thisIsEmpty())
            return false;
        checkIncludedModelVersions();
        for (int i=m_allIncludedModels.length-1;i>=0;--i)
            if (!m_allIncludedModels[i].isEmpty())
                return false;
        return true;
    }
    public Iterator iterator() throws ModelException {
        checkIncludedModelVersions();
        return new ModelIterator();
    }
    public boolean contains(Statement statement) throws ModelException {
        if (thisContains(statement))
            return true;
        checkIncludedModelVersions();
        for (int i=m_allIncludedModels.length-1;i>=0;--i)
            if (m_allIncludedModels[i].thisContains(statement))
                return true;
        return false;
    }
    public boolean contains(Resource subject,Resource predicate,RDFNode object) throws ModelException {
        if (thisContains(subject,predicate,object))
            return true;
        checkIncludedModelVersions();
        for (int i=m_allIncludedModels.length-1;i>=0;--i)
            if (m_allIncludedModels[i].thisContains(subject,predicate,object))
                return true;
        return false;
    }
    public Model findModel(Statement statement) throws ModelException {
        if (thisContains(statement))
            return this;
        checkIncludedModelVersions();
        for (int i=m_allIncludedModels.length-1;i>=0;--i)
            if (m_allIncludedModels[i].thisContains(statement))
                return m_allIncludedModels[i];
        return null;
    }
    public Model find(Resource subject,Resource predicate,RDFNode object) throws ModelException {
        Model result=new SearchResultsModelImpl(getNodeFactory());
        find(subject,predicate,object,result);
        return result;
    }
    public Model thisFind(Resource subject,Resource predicate,RDFNode object) throws ModelException {
        Model result=new SearchResultsModelImpl(getNodeFactory());
        thisFind(subject,predicate,object,result);
        return result;
    }
    public void find(Resource subject,Resource predicate,RDFNode object,Model result) throws ModelException {
        checkIncludedModelVersions();
        thisFind(subject,predicate,object,result);
        for (int i=m_allIncludedModels.length-1;i>=0;--i)
            m_allIncludedModels[i].thisFind(subject,predicate,object,result);
    }

    protected class ModelIterator implements Iterator {
        protected Iterator[] m_iterators;
        protected int m_index;

        public ModelIterator() throws ModelException {
            m_iterators=new Iterator[1+m_allIncludedModels.length];
            m_iterators[0]=thisIterator();
            for (int i=0;i<m_allIncludedModels.length;i++)
                m_iterators[i+1]=m_allIncludedModels[i].thisIterator();
        }
        public boolean hasNext() {
            if (m_index>=m_iterators.length)
                return false;
            return m_iterators[m_index].hasNext();
        }
        public Object next() {
            if (m_index>=m_iterators.length)
                throw new NoSuchElementException();
            Object object=m_iterators[m_index].next();
            while (m_index<m_iterators.length && !m_iterators[m_index].hasNext())
                m_index++;
            return object;
        }
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
