package edu.unika.aifb.rdf.api.util;

import java.util.*;
import java.io.*;
import java.net.*;

import edu.unika.aifb.rdf.api.model.*;


/**
 * Useful utility methods.
 */
public class RDFUtil {
    /** Default encoding. */
    public static final String DEFAULT_ENCODING="UTF-8";

    /**
     * Extracts the namespace prefix out of a URI.
     */
    public static String guessNamespace(String uri) {
        int l = getNamespaceEnd(uri);
        return l > 1 ? uri.substring(0, l) : "";
    }
    /**
     * Delivers the name out of the URI (without the namespace prefix).
     */
    public static String guessName(String uri) {
        return uri.substring(getNamespaceEnd(uri));
    }
    /**
     * Extracts the namespace prefix out of a URI.
     */
    public static String getNamespace(Resource r) throws ModelException {
        return guessNamespace(r.getURI());
    }
    /**
     * Delivers the name out of the URI (without the namespace prefix).
     */
    public static String getLocalName(Resource r) throws ModelException {
        return guessName(r.getURI());
    }
    /**
     * Position of the namespace end
     */
    static int getNamespaceEnd(String uri) {
        int l = uri.length()-1;
        do {
            char c = uri.charAt(l);
            if(c == '#' || c == ':' || c == '/')
                break;
            l--;
        } while (l >= 0);
        l++;
        return l;
    }

    public static void writeModel(Model m,Writer writer,String encoding) throws ModelException,IOException {
        RDFManager.createSerializer().serialize(m,writer,encoding);
    }

    public static void writeModel(Model m,OutputStream outputStream,String encoding) throws ModelException,IOException {
        RDFManager.createSerializer().serialize(m,outputStream,encoding);
    }

    public static String writeModel(Model m,String encoding) throws ModelException,IOException {
        StringWriter w=new StringWriter();
        writeModel(m,w,encoding);
        return w.toString();
    }

    public static void writeModel(Model m,URL url,String encoding) throws IOException,ModelException,URISyntaxException {
        File file=new File(new URI(url.toString()));
        writeModel(m,file,encoding);
    }

    public static void writeModel(Model m,File file,String encoding) throws IOException,ModelException {
        OutputStream stream=new BufferedOutputStream(new FileOutputStream(file));
        try {
            writeModel(m,stream,encoding);
        }
        finally {
            stream.close();
        }
     }

    public static Model immutableModel(Model model) {
        return new ImmutableModel(model);
    }

    public static String relativize(URI baseURI,URI relativeURI) {
        if (!baseURI.getScheme().equals(relativeURI.getScheme()))
            return relativeURI.toString();
        String result=new String();
        String currentToken1="";
        String currentToken2="";
        String path1=baseURI.getPath();
        String path2=relativeURI.getPath();
        StringTokenizer st1=new StringTokenizer(path1,"/");
        StringTokenizer st2=new StringTokenizer(path2,"/");
        int length1=st1.countTokens();
        int length2=st2.countTokens();
        int ind=0;
        while (st1.hasMoreTokens() && st2.hasMoreTokens()) {
            currentToken1=st1.nextToken();
            currentToken2=st2.nextToken();
            if (!currentToken1.equals(currentToken2))
                break;
            ind++;
        }
        if(ind==0)
            return relativeURI.toString();
        if (length1==length2 && length1==ind)
            if (currentToken1.equals(currentToken2))
                return ".";
        if (ind<length1) {
            result=result.concat("..");
            for(int j=1;j<length1-ind;j++)
                result=result.concat("/..");
        }
        if (ind<length1 && ind<length2)
            result=result.concat("/");
        if (ind<length2) {
            if(!currentToken1.equals(currentToken2))
                result= result.concat(currentToken2);
            else
                if (st2.hasMoreTokens())
                    result=result.concat(st2.nextToken());
            for (int j=ind+1;j<length2;j++) {
                result=result.concat("/");
                result=result.concat(st2.nextToken());
            }
        }
        return result;
    }

    protected static class ImmutableModel implements Model {
        protected Model m_model;

        public ImmutableModel(Model model) {
            m_model=model;
        }
        public String getLabel() throws ModelException {
            return m_model.getLabel();
        }
        public String getURI() throws ModelException {
            return m_model.getURI();
        }
        public boolean isPersistent() throws ModelException {
            return m_model.isPersistent();
        }
        public String getLogicalURI() throws ModelException {
            return m_model.getLogicalURI();
        }
        public void setLogicalURI(String logicalURI) throws ModelException {
            throw new ModelException("Cannot set the logical URI of an immutable model.");
        }
        public String getPhysicalURI() throws ModelException {
            return m_model.getPhysicalURI();
        }
        public void setPhysicalURI(String physicalURI) throws ModelException {
            throw new ModelException("Cannot set the physical URI of an immutable model.");
        }
        public int getInclusionIndex() {
            return m_model.getInclusionIndex();
        }
        public Collection getIncludedModels() throws ModelException {
            return m_model.getIncludedModels();
        }
        public Collection getAllIncludedModels() throws ModelException {
            return m_model.getAllIncludedModels();
        }
        public void addIncludedModel(Model model) throws ModelException {
            m_model.addIncludedModel(model);
        }
        public void removeIncludedModel(Model model) throws ModelException {
            m_model.removeIncludedModel(model);
        }
        public int size() throws ModelException {
            return m_model.size();
        }
        public int thisSize() throws ModelException {
            return m_model.thisSize();
        }
        public boolean isEmpty() throws ModelException {
            return m_model.isEmpty();
        }
        public boolean thisIsEmpty() throws ModelException {
            return m_model.thisIsEmpty();
        }
        public Iterator iterator() throws ModelException {
            return m_model.iterator();
        }
        public Iterator thisIterator() throws ModelException {
            return m_model.thisIterator();
        }
        public boolean contains(Statement statement) throws ModelException {
            return m_model.contains(statement);
        }
        public boolean contains(Resource subject,Resource predicate,RDFNode object) throws ModelException {
            return m_model.contains(subject,predicate,object);
        }
        public boolean thisContains(Statement statement) throws ModelException {
            return m_model.contains(statement);
        }
        public boolean thisContains(Resource subject,Resource predicate,RDFNode object) throws ModelException {
            return m_model.thisContains(subject,predicate,object);
        }
        public Model findModel(Statement statement) throws ModelException {
            return m_model.findModel(statement);
        }
        public void add(Statement statement) throws ModelException {
            throw new ModelException("Cannot change an immutable model.");
        }
        public void remove(Statement statement) throws ModelException {
            throw new ModelException("Cannot change an immutable model.");
        }
        public Model find(Resource subject,Resource predicate,RDFNode object) throws ModelException {
            return m_model.find(subject,predicate,object);
        }
        public Model thisFind(Resource subject,Resource predicate,RDFNode object) throws ModelException {
            return m_model.thisFind(subject,predicate,object);
        }
        public void find(Resource subject,Resource predicate,RDFNode object,Model result) throws ModelException {
            m_model.find(subject,predicate,object,result);
        }
        public void thisFind(Resource subject,Resource predicate,RDFNode object,Model result) throws ModelException {
            m_model.thisFind(subject,predicate,object,result);
        }
        public NodeFactory getNodeFactory() throws ModelException {
            return m_model.getNodeFactory();
        }
        public String getUniqueResourceURI(String prefix) throws ModelException {
            return m_model.getUniqueResourceURI(prefix);
        }
        public boolean supportsTransactions() {
            return false;
        }
        public void setAutocommit(boolean autocommit) throws ModelException {
            throw new ModelException("Cannot change autocommit mode on an immutable model.");
        }
        public boolean isAutocommit() {
            return false;
        }
        public void commit() throws ModelException {
            throw new ModelException("Cannot call commit on an immutable model.");
        }
        public void rollback() throws ModelException {
            throw new ModelException("Cannot call rollback on an immutable model.");
        }
        public String getAttribute(String key) throws ModelException {
            return m_model.getAttribute(key);
        }
        public void setAttribute(String key,String value) throws ModelException {
            throw new ModelException("Cannot set attributes on an immutable model.");
        }
        public Map getAttributes() throws ModelException {
            return m_model.getAttributes();
        }
    }
}
