package edu.unika.aifb.rdf.api.util;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.syntax.*;

/**
 * A mamager for RDF factories. An RDF factory must be registered with the manager before it can be used.
 */
public class RDFManager {
    /** Map of all RDF factories. */
    protected static final List s_factories=new ArrayList();
    static {
        try {
            registerFactory("edu.unika.aifb.rdf.mainmemory.RDFFactoryImpl");
        }
        catch (NoRDFFactoryException cannotHappen) {
        }
    }

    /**
     * Disallow creating this class.
     */
    private RDFManager() {
    }
    /**
     * Registers a RDF factory with this manager.
     *
     * @param factory                   a RDF factory
     */
    public static void registerFactory(RDFFactory factory) {
        if (!s_factories.contains(factory))
            s_factories.add(0,factory);
    }
    /**
     * Registers a RDF factory with given class with this manager.
     *
     * @param factoryClass              class of a RDF factory
     */
    public static void registerFactory(String factoryClass) throws NoRDFFactoryException {
        try {
            Class.forName(factoryClass);
        }
        catch (ClassNotFoundException e) {
            throw new NoRDFFactoryException("Cannot load RDF factory with class name '"+factoryClass+"'",e);
        }
    }
    /**
     * Creates a new RDF parser.
     */
    public static RDFParser createParser() {
        return new RDFParser();
    }
    /**
     * Creates a new RDF serializer
     */
    public static RDFSerializer createSerializer() {
        return new RDFSerializer();
    }
    /**
     * Create an empty model.
     */
    public static Model createModel(String physicalURI,NodeFactory nodeFactory) throws ModelException,NoRDFFactoryException {
        physicalURI=fixURISpaces(physicalURI);
        Iterator factories=s_factories.iterator();
        while (factories.hasNext()) {
            RDFFactory factory=(RDFFactory)factories.next();
            Model model=factory.createModel(physicalURI,nodeFactory);
            if (model!=null)
                return model;
        }
        throw new NoRDFFactoryException();
    }
    /**
     * Opens a model with given URI.
     */
    public static Model openModel(String physicalURI,NodeFactory nodeFactory) throws ModelException,NoRDFFactoryException {
        return openModelEx(physicalURI,nodeFactory).m_model;
    }
    /**
     * Opens a model with given URI and additionally returns the list of included physical and logical model URIs.
     */
    public static ModelInfo openModelEx(String physicalURI,NodeFactory nodeFactory) throws ModelException,NoRDFFactoryException {
        physicalURI=fixURISpaces(physicalURI);
        Iterator factories=s_factories.iterator();
        while (factories.hasNext()) {
            RDFFactory factory=(RDFFactory)factories.next();
            ModelInfo modelInfo=factory.openModelEx(physicalURI,nodeFactory);
            if (modelInfo!=null)
                return modelInfo;
        }
        throw new NoRDFFactoryException();
    }
    /**
     * Deletes a model.
     */
    public static void deleteModel(String physicalURI) throws ModelException,NoRDFFactoryException {
        physicalURI=fixURISpaces(physicalURI);
        Iterator factories=s_factories.iterator();
        while (factories.hasNext()) {
            RDFFactory factory=(RDFFactory)factories.next();
            if (factory.deleteModel(physicalURI))
                return;
        }
        throw new NoRDFFactoryException();
    }
    /**
     * Fixes the spaces of an URI.
     */
    protected static String fixURISpaces(String uri) {
        if (uri.indexOf(' ')==-1)
            return uri;
        StringBuffer buffer=new StringBuffer(uri);
        int start=0;
        int i=buffer.indexOf(" ",start);
        while (i!=-1) {
            buffer.delete(i,i+1);
            buffer.insert(i,"%20");
            start=i;
            i=buffer.indexOf(" ",start);
        }
        return buffer.toString();
    }

    public static class ModelInfo {
        public Model m_model;
        public String[][] m_includedModels;
    }
}
