package edu.unika.aifb.rdf.api.util;

import java.util.List;
import java.util.ArrayList;
import org.xml.sax.SAXException;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.syntax.*;

/**
 * A standard class used for parsing into an RDF model
 */
public class ModelConsumer implements RDFConsumer {
    protected Model m_model;
    protected List m_includedModels;

    /**
     * Creates an instance of this class.
     */
    public ModelConsumer() {
        this(null);
    }
    /**
     * Creates an instance of this class.
     *
     * @param model                 an RDF model to fill with triples. This frequently will be an empty model.
     */
    public ModelConsumer(Model model) {
        m_includedModels=new ArrayList();
        initialize(model);
    }
    /**
     * Initializes this object.
     *
     * @param model                 an RDF model to fill with triples. This frequently will be an empty model.
     */
    public void initialize(Model model) {
        m_model=model;
    }
    /**
     * Called when model parsing is started.
     *
     * @param physicalURI           physical URI of the model
     */
    public void startModel(String physicalURI) throws SAXException {
        try {
            if (m_model.supportsTransactions())
                m_model.setAutocommit(true);
            m_model.setPhysicalURI(physicalURI);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Called when model parsing is finished.
     */
    public void endModel() throws SAXException {
    }
    /**
     * Called when a statement with resource value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                URI of the object resource
     */
    public void statementWithResourceValue(String subject,String predicate,String object) throws SAXException {
        try {
            NodeFactory nodeFactory=m_model.getNodeFactory();
            Resource subjectResource=nodeFactory.createResource(subject);
            Resource predicateResource=nodeFactory.createResource(predicate);
            Resource objectResource=nodeFactory.createResource(object);
            Statement statement=nodeFactory.createStatement(subjectResource,predicateResource,objectResource);
            m_model.add(statement);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Called when a statement with literal value is added to the model.
     *
     * @param subject               URI of the subject resource
     * @param predicate             URI of the predicate resource
     * @param object                literal object value
     * @param language              the language
     * @param datatype              the URI of the literal's datatype (may be <code>null</code>)
     */
    public void statementWithLiteralValue(String subject,String predicate,String object,String language,String datatype) throws SAXException {
        try {
            NodeFactory nodeFactory=m_model.getNodeFactory();
            Resource subjectResource=nodeFactory.createResource(subject);
            Resource predicateResource=nodeFactory.createResource(predicate);
            Literal objectLiteral=nodeFactory.createLiteral(object,datatype,language);
            Statement statement=nodeFactory.createStatement(subjectResource,predicateResource,objectLiteral);
            m_model.add(statement);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Receives the logical URI of the model.
     *
     * @param logicalURI            logcal URI of the model
     */
    public void logicalURI(String logicalURI) throws SAXException {
        try {
            m_model.setLogicalURI(logicalURI);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Receives the notification that the model being parsed includes another model with supplied logical URI.
     *
     * @param logicalURI            logical URI of the model
     * @param physicalURI           physical URI of the model
     */
    public void includeModel(String logicalURI,String physicalURI) {
        m_includedModels.add(new String[] { logicalURI,physicalURI });
    }
    /**
     * Receives the notification that the attribute and its value has been parsed.
     *
     * @param key                   the key of the attribute
     * @param value                 the value of the attribute
     */
    public void addModelAttribte(String key,String value) throws SAXException {
        try {
            m_model.setAttribute(key,value);
        }
        catch (ModelException e) {
            throw new SAXException("RDF exception",e);
        }
    }
    /**
     * Returns the list containing information about the included RDF models.
     *
     * @return                      the list of the form [ logicalURI,physicalURI ] containing information about included models
     */
    public List getIncludedModels() {
        return m_includedModels;
    }
}
