package edu.unika.aifb.rdf.api.syntax;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;
import java.io.IOException;
import java.io.Writer;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URISyntaxException;

import edu.unika.aifb.rdf.api.util.*;

/**
 * The writer for RDF models. This class is independent of the RDF API and can be used for programmatic serialization of any
 * type of RDF data.
 */
public class RDFWriter implements RDFConstants {
    public static final String DEFAULT_ENCODING="UTF-8";

    protected static final char USE_ANY_QUOTE=(char)0;
    protected static final char USE_CDATA=(char)1;
    protected static final int MAX_ALLOWED_ABBREVIATED_LENGTH=60;
    protected static final String INDENTATION="    ";
    protected static final String OPAQUE_URI="opaque:uri";

    protected Map m_defaultNamespaces;
    protected Map m_namespaces;
    protected int m_nextAutomaticPrefixIndex;
    protected PrintWriter m_out;
    protected String m_rdfIDElementText;
    protected String m_rdfAboutElementText;
    protected String m_rdfResourceElementText;
    protected String m_rdfParseTypeText;
    protected String m_rdfDatatypeText;
    protected String m_rdfElementText;
    protected URI m_baseURI;
    protected URI m_physicalURI;
    protected String m_currentSubject;
    protected RDFStatement m_groupTypeStatement;
    protected List m_attributeStatements;
    protected Set m_attributePredicates;
    protected List m_contentStatements;

    /**
     * Creates an instance of this class.
     */
    public RDFWriter() {
        m_defaultNamespaces=new HashMap();
        addNamespacePrefix("rdf",RDFNS);
        addNamespacePrefix("rdfs",RDFSNS);
        addNamespacePrefix("kaon",KAONNS);
    }
    /**
     * Returns the writer.
     *
     * @return                                  the writer
     */
    public PrintWriter getOut() {
        return m_out;
    }
    /**
     * Registers a namespace prefix for this serializer.
     *
     * @param prefix                            the prefix for the namespace
     * @param namespace                         the namespace
     */
    public void addNamespacePrefix(String prefix,String namespace) {
        m_defaultNamespaces.put(namespace,prefix);
    }
    /**
     * Prepares the collection of namespaces.
     */
    public void prepareNamespaceCollection() {
        m_namespaces=new HashMap();
        m_namespaces.put(RDFNS,m_defaultNamespaces.get(RDFNS));
    }
    /**
     * Starts the serialization process. It should be called after all namespaces have been collected. It will write out
     * the head of the XML file, but will not write out the head of the rdf:RDF node.
     *
     * @param writer                            the writer receiving the information
     * @param physicalURI                       the physical URI of the model
     * @param logicalURI                        the logical URI of the model
     * @param encoding                          the encoding of the model
     * @throws IOException                      thrown if there is an error
     */
    public void startSerialization(Writer writer,String physicalURI,String logicalURI,String encoding) throws IOException {
        m_rdfElementText=getElementText(RDF_RDF);
        m_rdfIDElementText=getElementText(RDF_ID);
        m_rdfAboutElementText=getElementText(RDF_ABOUT);
        m_rdfResourceElementText=getElementText(RDF_RESOURCE);
        m_rdfParseTypeText=getElementText(RDF_PARSE_TYPE);
        m_rdfDatatypeText=getElementText(RDF_DATATYPE);
        m_out=new PrintWriter(writer);
        try {
            m_physicalURI=new URI(physicalURI);
            if (!m_physicalURI.isOpaque()) {
                String path=m_physicalURI.getPath();
                int slashPosition=path.lastIndexOf('/');
                if (slashPosition!=-1) {
                    String pathNoFile=path.substring(0,slashPosition+1);
                    m_physicalURI=new URI(m_physicalURI.getScheme(),m_physicalURI.getUserInfo(),m_physicalURI.getHost(),m_physicalURI.getPort(),pathNoFile,m_physicalURI.getQuery(),m_physicalURI.getFragment());
                }
            }
        }
        catch (URISyntaxException e) {
            IOException error=new IOException("Invalid physical URI '"+physicalURI+"'");
            error.initCause(e);
            throw error;
        }
        try {
            if (logicalURI==null)
                m_baseURI=new URI(OPAQUE_URI);      // turns off URI relativization
            else
                m_baseURI=new URI(logicalURI);
        }
        catch (URISyntaxException e) {
            IOException error=new IOException("Invalid physical URI '"+logicalURI+"'");
            error.initCause(e);
            throw error;
        }
        // start writing the contents
        m_out.println("<?xml version='1.0' encoding='"+encoding+"'?>");
        m_out.println("<!DOCTYPE "+m_rdfElementText+" [");
        writeEntityDeclarations();
        m_out.println("]>");
    }
    /**
     * Starts the RDF contents.
     *
     * @throws IOException                      thrown if there is an error
     */
    public void startRDFContents() {
        m_out.println();
        m_out.print("<"+m_rdfElementText);
        // write the xml:base
        String baseURI=m_baseURI.toString();
        if (!OPAQUE_URI.equals(baseURI))
            m_out.print(" xml:base=\""+baseURI+"\"");
        writeNamespaceDeclarations();
        m_out.println(">");
        m_out.println();
        m_groupTypeStatement=null;
        m_attributeStatements=new ArrayList();
        m_attributePredicates=new HashSet();
        m_contentStatements=new ArrayList();
        m_currentSubject=null;
    }
    /**
     * Finishes the RDF model.
     *
     * @throws IOException                      thrown if there is an error
     */
    public void finishRDFContents() throws IOException {
        writeGroup();
        m_out.println();
        m_out.println("</"+m_rdfElementText+">");
    }
    /**
     * Performs the cleanup after serialization.
     *
     * @throws IOException                      thrown if there is an error
     */
    public void cleanUp() {
        m_namespaces=null;
        m_currentSubject=null;
        m_groupTypeStatement=null;
        m_attributeStatements=null;
        m_attributePredicates=null;
        m_contentStatements=null;
        m_rdfResourceElementText=null;
        if (m_out!=null) {
            m_out.flush();
            m_out=null;
        }
    }
    /**
     * Writes a single model inclusion statement.
     *
     * @param logicalURI                        the logical URI of the included model
     * @param physicalURI                       the physical URI of the model
     * @throws IOException                      thrown if there is an error
     */
    public void writeInclusion(String logicalURI,String physicalURI) throws IOException {
        try {
            if (logicalURI!=null || physicalURI!=null) {
                m_out.print("<?include-rdf");
                if (logicalURI!=null) {
                    m_out.print(" logicalURI=\"");
                    m_out.print(logicalURI);
                    m_out.print("\"");
                }
                if (physicalURI!=null) {
                    m_out.print(" physicalURI=\"");
                    String relativizedURI=RDFUtil.relativize(m_physicalURI,new URI(physicalURI));
                    m_out.print(relativizedURI);
                    m_out.print("\"");
                }
                m_out.println("?>");
            }
        }
        catch (URISyntaxException e) {
            IOException error=new IOException("Invalid physical URI '"+physicalURI+"'");
            error.initCause(e);
            throw error;
        }
    }
    /**
     * Writes a single model attribute.
     *
     * @param attributeName                     the name of the attribute
     * @param attributeValue                    the value of the attribute
     */
    public void writeModelAttribute(String attributeName,String attributeValue) {
        m_out.print("<?model-attribute key=\"");
        m_out.print(attributeName);
        m_out.print("\" value=\"");
        m_out.print(attributeValue);
        m_out.println("\"?>");
    }
    /**
     * Writes the attributes of the model from the map.
     *
     * @param attributes                        the map of attributes
     */
    public void writeModelAttributes(Map attributes) {
        if (!attributes.isEmpty()) {
            m_out.println();
            Iterator keys=attributes.keySet().iterator();
            while (keys.hasNext()) {
                String key=(String)keys.next();
                String value=(String)attributes.get(key);
                writeModelAttribute(key,value);
            }
        }
    }
    /**
     * Writes the entity declarations.
     */
    protected void writeEntityDeclarations() {
        for (Iterator iterator=m_namespaces.keySet().iterator();iterator.hasNext();) {
            String namespace=(String)iterator.next();
            String prefix=(String)m_namespaces.get(namespace);
            m_out.println(INDENTATION+"<!ENTITY "+prefix+" '"+namespace+"'>");
        }
    }
    /**
     * Writes the namespace declarations.
     */
    protected void writeNamespaceDeclarations() {
        for (Iterator iterator=m_namespaces.keySet().iterator();iterator.hasNext();) {
            String namespace=(String)iterator.next();
            String prefix=(String)m_namespaces.get(namespace);
            m_out.println();
            m_out.print(INDENTATION+"xmlns:"+prefix+"=\"&"+prefix+";\"");
        }
    }
    /**
     * Writes a statement to the model. To get nicer serialization, statements with the same subject should be added one after another.
     * The serializer will group the statements with the same subject and then write them out in the abbreviated form.
     *
     * @param subject                           the subject URI
     * @param predicate                         the predicate URI
     * @param object                            the object
     * @param language                          the language
     * @param datatype                          the datatype
     * @param isLiteral                         <code>true</code> if this statement has literal value
     * @throws IOException                      thrown if there is an error
     */
    public void writeStatement(String subject,String predicate,String object,String language,String datatype,boolean isLiteral) throws IOException {
        RDFStatement statement=new RDFStatement(subject,predicate,object,language,datatype,isLiteral);
        // if this statement has different subject, then write out the current group
        if (m_currentSubject==null || !m_currentSubject.equals(statement.m_subject)) {
            writeGroup();
            m_currentSubject=subject;
        }
        // classify the statement
        if (m_groupTypeStatement==null && canAbbreviateStatementAsTypeDeclaration(statement))
            m_groupTypeStatement=statement;
        else if (canAbbreviateLiteralValue(statement) && !m_attributePredicates.contains(statement.m_predicate)) {
            m_attributeStatements.add(statement);
            m_attributePredicates.add(statement.m_predicate);
        }
        else
            m_contentStatements.add(statement);
    }
    /**
     * Writes a group of statements with the same subject that has collected in the internal variables of the serializer.
     *
     * @throws IOException                      thrown if there is an error
     */
    protected void writeGroup() throws IOException {
        if (m_currentSubject==null || (m_groupTypeStatement==null && m_attributeStatements.isEmpty() && m_contentStatements.isEmpty()))
            return;
        String outerElementName;
        if (m_groupTypeStatement!=null)
            outerElementName=getElementText(m_groupTypeStatement.m_object);
        else
            outerElementName=getElementText(RDF_DESCRIPTION);
        m_out.print('<');
        m_out.print(outerElementName);
        m_out.print(' ');
        writeIDOrAbout(m_currentSubject);
        writeAttributeStatements();
        if (m_contentStatements.isEmpty())
            m_out.println("/>");
        else {
            m_out.println(">");
            writeContentStatements();
            m_out.print("</");
            m_out.print(outerElementName);
            m_out.println('>');
        }
        m_groupTypeStatement=null;
        m_attributeStatements.clear();
        m_attributePredicates.clear();
        m_contentStatements.clear();
    }
    /**
     * Writes the URI in the format for rdf:ID or rdf:about, dependeing whether the URI can be relativized.
     *
     * @param resourceURI                       the URI to be written
     * @throws IOException                      thrown if there is an error
     */
    protected void writeIDOrAbout(String resourceURI) throws IOException {
        try {
            URI relativizedURI=m_baseURI.relativize(new URI(resourceURI));
            String relativizedURIString=relativizedURI.toString();
            if (!relativizedURI.isAbsolute() && relativizedURIString.startsWith("#")) {
                m_out.print(m_rdfIDElementText);
                m_out.print("=\"");
                m_out.print(relativizedURIString.substring(1));
            }
            else {
                m_out.print(m_rdfAboutElementText);
                m_out.print("=\"");
                if (!relativizedURI.isAbsolute())
                    m_out.print(relativizedURIString);
                else
                    writeAbsoluteResourceReference(relativizedURIString);
            }
            m_out.print("\"");
        }
        catch (URISyntaxException e) {
            IOException error=new IOException("Invalid URI syntax '"+resourceURI+"'");
            error.initCause(e);
            throw error;
        }
    }
    /**
     * Writes out the statements that have been collected as attributes.
     *
     * @throws IOException                      thrown if there is an error
     */
    protected void writeAttributeStatements() throws IOException {
        for (int i=0;i<m_attributeStatements.size();i++) {
            RDFStatement statement=(RDFStatement)m_attributeStatements.get(i);
            m_out.println();
            m_out.print(INDENTATION);
            m_out.print(getElementText(statement.m_predicate));
            m_out.print('=');
            String value=statement.m_object;
            char quote=getValueQuoteType(value);
            m_out.print(quote);
            m_out.print(escapeValueForXML(value));
            m_out.print(quote);
        }
    }
    /**
     * Writes out the statements that have been collected as content statements.
     *
     * @throws IOException                      thrown if there is an error
     */
    protected void writeContentStatements() throws IOException {
        for (int i=0;i<m_contentStatements.size();i++) {
            RDFStatement statement=(RDFStatement)m_contentStatements.get(i);
            m_out.print(INDENTATION);
            m_out.print('<');
            String predicateElementText=getElementText(statement.m_predicate);
            m_out.print(predicateElementText);
            if (!statement.m_isLiteral) {
                writeResourceReference(statement.m_object);
                m_out.println("/>");
            }
            else {
                if (statement.m_language!=null) {
                    m_out.print(" ");
                    m_out.print(XMLLANG);
                    m_out.print("=\"");
                    m_out.print(statement.m_language);
                    m_out.print("\"");
                }
                if (statement.m_datatype!=null)
                    if (RDF_XMLLITERAL.equals(statement.m_datatype)) {
                        m_out.print(" ");
                        m_out.print(m_rdfParseTypeText);
                        m_out.print("=\"");
                        m_out.print(PARSE_TYPE_LITERAL);
                        m_out.print("\"");
                    }
                    else {
                        m_out.print(" ");
                        m_out.print(m_rdfDatatypeText);
                        m_out.print("=\"");
                        m_out.print(statement.m_datatype);
                        m_out.print("\"");
                    }
                m_out.print('>');
                writeTextValue(statement.m_object);
                m_out.print("</");
                m_out.print(predicateElementText);
                m_out.println('>');
            }
        }
    }
    /**
     * Writes a reference to the resource.
     *
     * @param resourceURI                       the URI to be written
     * @throws IOException                      thrown if there is an error
     */
    protected void writeResourceReference(String resourceURI) throws IOException {
        try {
            m_out.print(' ');
            m_out.print(m_rdfResourceElementText);
            m_out.print("=\"");
            URI relativizedURI=m_baseURI.relativize(new URI(resourceURI));
            if (!relativizedURI.isAbsolute())
                m_out.print(relativizedURI.toString());
            else
                writeAbsoluteResourceReference(resourceURI);
            m_out.print("\"");
        }
        catch (URISyntaxException e) {
            IOException error=new IOException("Invalid URI syntax '"+resourceURI+"'");
            error.initCause(e);
            throw error;
        }
    }
    /**
     * Writes an absolute reference to the resource.
     *
     * @param resourceURI                       the URI to be written
     */
    protected void writeAbsoluteResourceReference(String resourceURI) {
        String namespace=RDFUtil.guessNamespace(resourceURI);
        String localName=RDFUtil.guessName(resourceURI);
        String text=resourceURI;
        if (namespace!=null && namespace.length()!=0) {
            String prefix=(String)m_namespaces.get(namespace);
            if (prefix!=null)
                text="&"+prefix+";"+localName;
        }
        m_out.print(text);
    }
    /**
     * Writes out the text value.
     *
     * @param textValue                         the text value to write
     */
    protected void writeTextValue(String textValue) {
        if (getValueQuoteType(textValue)==USE_CDATA)
            writeEscapedCDATA(textValue);
        else
            m_out.print(escapeValueForXML(textValue));
    }
    /**
     * Writes out the value excaped as CDATA.
     *
     * @param textValue                         the text value to write
     */
    protected void writeEscapedCDATA(String textValue) {
        int start=0;
        int i=textValue.indexOf("]]>",start);
        m_out.print("<![CDATA[");
        while (i>=0 && start<textValue.length()) {
            m_out.print(textValue.substring(start,i));
            m_out.print("]]>]]&#x3e;<![CDATA[");
            start=i+3;
            i=textValue.indexOf("]]>",start);
        }
        m_out.print(textValue.substring(start));
        m_out.print("]]>");
    }
    /**
     * Reutrns the supplied value escaped according to XML standards.
     *
     * @param textValue                         the value to escape
     * @return                                  the escaped value
     */
    protected String escapeValueForXML(String textValue) {
        if (textValue.indexOf('<')==-1 && textValue.indexOf('&')==-1)
            return textValue;
        StringBuffer buffer=new StringBuffer(textValue);
        for (int i=buffer.length()-1;i>=0;i--) {
            char c=buffer.charAt(i);
            if (c=='<') {
                buffer.deleteCharAt(i);
                buffer.insert(i,"&lt;");
            }
            else if (c=='&') {
                buffer.deleteCharAt(i);
                buffer.insert(i,"&amp;");
            }
        }
        return buffer.toString();
    }
    /**
     * Returns the type of quotes that should be used for given value.
     *
     * @param textValue                         the value examined
     * @return                                  the type of quote (may be <code>USE_ANY_QUOTE</code>, '," or <code>USE_CDATA</code>)
     */
    protected char getValueQuoteType(String textValue) {
        char quote=USE_ANY_QUOTE;
        boolean hasBreaks=false;
        boolean whiteSpaceOnly=true;
        for (int i=0;i<textValue.length();i++) {
            char c=textValue.charAt(i);
            if (c=='\n')
                hasBreaks=true;
            if (c=='"' || c=='\'') {
                if (quote==USE_ANY_QUOTE)
                    quote=(c=='"') ? '\'' : '"';
                else if (c==quote)
                    return USE_CDATA;
            }
            if (!Character.isWhitespace(c))
                whiteSpaceOnly=false;
        }
        if (whiteSpaceOnly || hasBreaks)
            return USE_CDATA;
        return quote==USE_ANY_QUOTE ? '"' : quote;
    }
    /**
     * Returns <code>true</code> if supplied statement can be abbreviated as attribute of the rdf:Description element.
     *
     * @param statement                         the statement examined
     * @return                                  <code>true</code> if the statement can be abbreviated
     */
    protected boolean canAbbreviateLiteralValue(RDFStatement statement) {
        if (statement.m_isLiteral && statement.m_language==null && statement.m_datatype==null) {
            String value=statement.m_object;
            if (value.length()<MAX_ALLOWED_ABBREVIATED_LENGTH) {
                char c=getValueQuoteType(value);
                return c=='"' || c=='\'';
            }
        }
        return false;
    }
    /**
     * Returns <code>true</code> if supplied statement can be abbreviated as the type declaration.
     *
     * @param statement                         the statement examined
     * @return                                  <code>true</code> if the statement can be used for type declaration abbreviation
     * @throws IOException                      thrown if there is an error
     */
    protected boolean canAbbreviateStatementAsTypeDeclaration(RDFStatement statement) throws IOException {
        if (!statement.m_isLiteral && RDF_TYPE.equals(statement.m_predicate)) {
            String elementName=getElementText(statement.m_object);
            return isValidXMLName(elementName);
        }
        else
            return false;
    }
    /**
     * Returns <code>true</code> if the supplied name is a valid XML element/attribute name.
     *
     * @param objectName                        the name of the object
     * @return                                  <code>true</code> if given object can safely be serialized as an element or attribute
     */
    protected boolean isValidXMLName(String objectName) {
        for (int i=objectName.length()-1;i>=0;i--) {
            char c=objectName.charAt(i);
            if (!Character.isLetterOrDigit(c) && c!=':' && c!='-' && c!='_')
                return false;
        }
        return true;
    }
    /**
     * Returns the text that should be used for given element name by replacing the enement's namespace with the namespace prefix.
     *
     * @param elementName                       the name of the element
     * @return                                  the text to use for element
     * @throws IOException                      thrown if there is an error
     */
    protected String getElementText(String elementName) throws IOException {
        String namespace=RDFUtil.guessNamespace(elementName);
        String localName=RDFUtil.guessName(elementName);
        if (namespace==null || namespace.length()==0)
            return localName;
        String prefix=(String)m_namespaces.get(namespace);
        if (prefix==null)
            throw new IOException("Prefix for element '"+elementName+"' cannot be found.");
        return prefix+":"+localName;
    }
    /**
     * Notifies the serializer that supplied URI occurs in the model. The serializer will determine whether another namespace declaraion
     * is necessary.
     *
     * @param uri                               the URI that occurs in the model
     */
    public void collectNamespace(String uri) {
        String namespace=RDFUtil.guessNamespace(uri);
        if (!m_namespaces.containsKey(namespace)) {
            String prefix=(String)m_defaultNamespaces.get(namespace);
            if (prefix==null)
                prefix=(String)m_defaultNamespaces.get(namespace);
            if (prefix==null)
                do {
                    prefix=getNextNamespacePrefix();
                } while (m_namespaces.containsKey(prefix));
            m_namespaces.put(namespace,prefix);
        }
    }
    /**
     * Returns the next new namespace prefix.
     *
     * @return                                  the next new namespace prefix
     */
    protected String getNextNamespacePrefix() {
        StringBuffer buffer=new StringBuffer();
        int index=m_nextAutomaticPrefixIndex++;
        do {
            buffer.append((char)('a'+(index % 26)));
            index=index/26;
        } while (index!=0);
        return buffer.toString();
    }

    /**
     * A holder class for infromation about a statement.
     */
    protected static class RDFStatement {
        public String m_subject;
        public String m_predicate;
        public String m_object;
        public String m_language;
        public String m_datatype;
        public boolean m_isLiteral;

        public RDFStatement(String subject,String predicate,String object,String language,String datatype,boolean isLiteral) {
            m_subject=subject;
            m_predicate=predicate;
            m_object=object;
            m_language=language;
            m_datatype=datatype;
            m_isLiteral=isLiteral;
        }
    }
}
