package edu.unika.aifb.rdf.api.syntax;

import java.util.List;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.io.IOException;
import java.io.Writer;
import java.io.OutputStream;
import java.io.OutputStreamWriter;

import edu.unika.aifb.rdf.api.model.*;

/**
 * A serializer for RDF models.
 */
public class RDFSerializer extends RDFWriter {
    /**
     * Creates an instance of this class.
     */
    public RDFSerializer() {
    }
    /**
     * Serializes given RDF model.
     *
     * @param model                     the model to serialize
     * @param outputStream              the stream where serialization is performed
     * @param encoding                  tne encoding
     * @throws IOException              thrown if there is an error
     * @throws ModelException           thrown if there is an error accessing the RDF model
     */
    public void serialize(Model model,OutputStream outputStream,String encoding) throws IOException,ModelException {
        serialize(model,new OutputStreamWriter(outputStream,encoding),encoding);
    }
    /**
     * Serializes given RDF model.
     *
     * @param model                     the model to serialize
     * @param writer                    the writer where serialization is performed
     * @param encoding                  tne encoding
     * @throws IOException              thrown if there is an error
     * @throws ModelException           thrown if there is an error accessing the RDF model
     */
    public void serialize(Model model,Writer writer,String encoding) throws IOException,ModelException {
        try {
            // parepare the serializer to collect namespaces
            prepareNamespaceCollection();
            // get all statements in a list and sort them so that the statements with the same subject are grouped together
            List statements=new ArrayList();
            statements=new ArrayList(model.thisSize());
            for (Iterator iterator=model.thisIterator();iterator.hasNext();) {
                Statement statement=(Statement)iterator.next();
                statements.add(statement);
                if (RDF_TYPE.equals(statement.predicate().getURI()))
                    collectNamespace(((Resource)statement.object()).getURI());
                else if (RDFS_SUBCLASSOF.equals(statement.predicate()) || RDFS_SUBPROPERTYOF.equals(statement.predicate())) {
                    collectNamespace(statement.subject().getURI());
                    collectNamespace(((Resource)statement.object()).getURI());
                }
                collectNamespace(statement.predicate().getURI());
            }
            Collections.sort(statements,StatementComparator.INSTANCE);
            // start the serialization
            super.startSerialization(writer,model.getPhysicalURI(),model.getLogicalURI(),encoding);
            // print out the model inclusions
            Collection includedModels=model.getIncludedModels();
            if (!includedModels.isEmpty()) {
                m_out.println();
                Iterator iterator=includedModels.iterator();
                while (iterator.hasNext()) {
                    Model includedModel=(Model)iterator.next();
                    writeInclusion(includedModel.getLogicalURI(),includedModel.getPhysicalURI());
                }
            }
            // print the model attributes
            writeModelAttributes(model.getAttributes());
            // start the RDF block
            startRDFContents();
            // write out the statements
            for (int i=0;i<statements.size();i++) {
                Statement statement=(Statement)statements.get(i);
                Resource subject=statement.subject();
                Resource predicate=statement.predicate();
                RDFNode object=statement.object();
                if (object instanceof Resource)
                    writeStatement(subject.getURI(),predicate.getURI(),((Resource)object).getURI(),null,null,false);
                else {
                    Literal literal=(Literal)object;
                    writeStatement(subject.getURI(),predicate.getURI(),literal.getLabel(),literal.getLanguage(),literal.getDatatype(),true);
                }
            }
            // finish RDF block
            finishRDFContents();
        }
        finally {
            cleanUp();
        }
    }

    protected static class StatementComparator implements Comparator {
        protected static final StatementComparator INSTANCE=new StatementComparator();

        protected int compare(RDFNode r1,RDFNode r2) throws ModelException {
            return r1.getLabel().compareTo(r2.getLabel());
        }
        protected boolean isSystemNamespace(RDFNode node) throws ModelException {
            if (node instanceof Resource) {
                String uri=node.getLabel();
                return uri.startsWith(RDFNS) || uri.startsWith(RDFSNS);
            }
            else
                return false;
        }
        public int compare(Object o1,Object o2) {
            try {
                Statement statement1=(Statement)o1;
                Statement statement2=(Statement)o2;
                int result=compare(statement1.subject(),statement2.subject());
                if (result!=0)
                    return result;
                // sort them according to the predicate values
                result=compare(statement1.predicate(),statement2.predicate());
                if (result!=0)
                    return result;
                // finally compare the objects
                RDFNode object1=statement1.object();
                RDFNode object2=statement2.object();
                // to get nicer serialization, place rdf and rdfs objects first
                boolean isSystemNamespace1=isSystemNamespace(object1);
                boolean isSystemNamespace2=isSystemNamespace(object2);
                if (isSystemNamespace1 && !isSystemNamespace2)
                    return -1;
                if (!isSystemNamespace1 && isSystemNamespace2)
                    return 1;
                return compare(object1,object2);
            }
            catch (ModelException ignored) {
                return 0;
            }
        }
    }
}
