package edu.unika.aifb.rdf.api.model;

import java.util.Map;
import java.util.Iterator;
import java.util.Collection;

/**
 * This is a programming interface to an RDF model which is viewed as a set of statements. Models may offer transactional
 * behavior by implementing <code>commit()</code> and <code>rollabck()</code> methods. A model need not offer these capabilities.
 * In that case exceptions must be thrown in response to these calls.
 * <p>Each model has a logical URI that uniquely identifies the model. However, the model may be loaded from some other URI - this
 * URI is then called the physical URI. These two URIs may, but need not be the same.
 */
public interface Model extends Resource {
    /**
     * Returns <code>true</code> if this model is persistent. RDF files aren't persistent.
     *
     * @return              <code>true</code> if this model is persistent
     */
    boolean isPersistent() throws ModelException;
    /**
     * Returns the logical URI of the model.
     *
     * @return              logical URI of the model
     */
    String getLogicalURI() throws ModelException;
    /**
     * Sets the logical URI of the model.
     *
     * @param logicalURI    logical URI of the model
     */
    void setLogicalURI(String logicalURI) throws ModelException;
    /**
     * Returns the physical URI of the model.
     *
     * @return              physical URI of the model
     */
    String getPhysicalURI() throws ModelException;
    /**
     * Sets the physical URI of the model.
     *
     * @param physicalURI   physical URI of the model
     */
    void setPhysicalURI(String physicalURI) throws ModelException;
    /**
     * Returns a collection of included models (as <code>Model</code> objects).
     *
     * @return              collection of included models
     */
    Collection getIncludedModels() throws ModelException;
    /**
     * Returns a collection of all included models (as <code>Model</code> objects),
     * including models that are indirectly included.
     *
     * @return              collection of all included models
     */
    Collection getAllIncludedModels() throws ModelException;
    /**
     * Adds a model to the set of included models. This method checks whether supplied model was already included.
     * If so, the model is not included again.
     *
     * @param model         model to be included
     */
    void addIncludedModel(Model model) throws ModelException;
    /**
     * Removes a model from the set of included models.
     *
     * @param model         model to be removed
     */
    void removeIncludedModel(Model model) throws ModelException;
    /**
     * Whenever the set of includes has been changed, this method should return a different index. Thus the including
     * models may determine that the list of includes has changed.
     */
    int getInclusionIndex();
    /**
     * Number of triples in the model
     *
     * @return              number of triples in this model and all included models
     */
    int size() throws ModelException;
    /**
     * Number of triples in this model only.
     *
     * @return              number of triples in this model only
     */
    int thisSize() throws ModelException;
    /**
     * True if the model contains no triples.
     */
    boolean isEmpty() throws ModelException;
    /**
     * True if this model contains no triples (without taking into consideration included models).
     */
    boolean thisIsEmpty() throws ModelException;
    /**
     * Enumerate triples of this and all included models.
     */
    Iterator iterator() throws ModelException;
    /**
     * Enumerate triples of this model only.
     */
    Iterator thisIterator() throws ModelException;
    /**
     * Tests if the model contains the given triple.
     */
    boolean contains(Statement statement) throws ModelException;
    /**
     * Tests if the model contains the given triple, where each component may be <code>null</code>.
     */
    boolean contains(Resource subject,Resource predicate,RDFNode object) throws ModelException;
    /**
     * Tests if this model contains the given triple, wihtout taking into accound the included models.
     */
    boolean thisContains(Statement statement) throws ModelException;
    /**
     * Tests if this model contains the given triple, wihtout taking into accound the included models. Each component may be <code>null</code>.
     */
    boolean thisContains(Resource subject,Resource predicate,RDFNode object) throws ModelException;
    /**
     * Adds a new statement to the model.
     */
    void add(Statement statement) throws ModelException;
    /**
     * Removes the triple from the model.
     */
    void remove(Statement statement) throws ModelException;
    /**
     * Locates the model in which supplied statement is declared, or returns <code>null</code> if there is no such model.
     * If several models contain the same statement, the result is undefined.
     */
    Model findModel(Statement statement) throws ModelException;
    /**
     * General method to search for triples. <code>null</code> input for any parameter will match anything.
     */
    Model find(Resource subject,Resource predicate,RDFNode object) throws ModelException;
    /**
     * General method to search for triples in this model only, without taking into account included models.
     */
    Model thisFind(Resource subject,Resource predicate,RDFNode object) throws ModelException;
    /**
     * General method to search for triples.
     * <code>null</code> input for any parameter will match anything.
     */
    void find(Resource subject,Resource predicate,RDFNode object,Model result) throws ModelException;
    /**
     * General method to search for triples.
     * <code>null</code> input for any parameter will match anything.
     */
    void thisFind(Resource subject,Resource predicate,RDFNode object,Model result) throws ModelException;
    /**
     * Returns the node factory for this model
     */
    NodeFactory getNodeFactory() throws ModelException;
    /**
     * Returns a resource URI that is unique for this model.
     */
    String getUniqueResourceURI(String prefix) throws ModelException;
    /**
     * Checks whether this model supports transactions.
     */
    boolean supportsTransactions() throws ModelException;
    /**
     * Sets this model into autocommit mode where avery change is commited as it is performed.
     */
    void setAutocommit(boolean autocommit) throws ModelException;
    /**
     * If <code>true</code>, this model is in autocommit mode.
     */
    boolean isAutocommit() throws ModelException;
    /**
     * Commits all changes to the model performed after the last <code>commit()</code> or <code>rollabck()</code>.
     * If an exception is thrown during commit, the model is probably inconsistent.
     */
    void commit() throws ModelException;
    /**
     * Rolls back all changes to the model performed after the last <code>commit()</code> or <code>rollabck()</code>.
     * If an exception is thrown during commit, the model is probably inconsistent.
     */
    void rollback() throws ModelException;
    /**
     * Returns the value of the OI-model attribute with given key.
     */
    String getAttribute(String key) throws ModelException;
    /**
     * Sets the value of the OI-model attribute with given key.
     */
    void setAttribute(String key,String value) throws ModelException;
    /**
     * Returns the map of all key-value pairs of this OI-model.
     */
    Map getAttributes() throws ModelException;
}
