package edu.unika.aifb.kaon.query.compiler;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Stack;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

import edu.unika.aifb.kaon.datalog.program.*;

import edu.unika.aifb.kaon.query.ast.*;

/**
 * The compiler that takes an expression and compiles it into Datalog.
 */
public abstract class ExpressionToDatalogCompiler {
    /** The array of empty literals. */
    protected static final Literal[] EMPTY_LITERALS=new Literal[0];
    /** The map of concept functions known to the compiler. */
    protected static final Map s_conceptFunctions=new HashMap();
    static {
        addConceptFunctionInfo("SUBCONCEPTS","ConceptHierarchy_t",2,0,1);
        addConceptFunctionInfo("SUBCONCEPTS^","ConceptHierarchy",2,0,1);
        addConceptFunctionInfo("SUPERCONCEPTS","ConceptHierarchy_t",2,1,0);
        addConceptFunctionInfo("SUPERCONCEPTS^","ConceptHierarchy",2,1,0);
        addConceptFunctionInfo("INSTANCES","ConceptInstance_a",2,0,1);
        addConceptFunctionInfo("INSTANCES^","ConceptInstance",2,0,1);
        addConceptFunctionInfo("PARENT_CONCEPTS","ConceptInstance_a",2,1,0);
        addConceptFunctionInfo("PARENT_CONCEPTS^","ConceptInstance",2,1,0);
        addConceptFunctionInfo("PROPERTIES_FROM","PropertyDomain",4,1,0);
        addConceptFunctionInfo("PROPERTIES_TO","PropertyRange",2,1,0);
        addConceptFunctionInfo("DOMAIN_CONCEPTS","PropertyDomain",4,0,1);
        addConceptFunctionInfo("RANGE_CONCEPTS","PropertyRange",2,0,1);
        addConceptFunctionInfo("SUBPROPERTIES","PropertyHierarchy_t",2,0,1);
        addConceptFunctionInfo("SUBPROPERTIES^","PropertyHierarchy",2,0,1);
        addConceptFunctionInfo("SUPERPROPERTIES","PropertyHierarchy_t",2,1,0);
        addConceptFunctionInfo("SUPERPROPERTIES^","PropertyHierarchy",2,1,0);
    }
    protected static void addConceptFunctionInfo(String functionName,String predicateName,int predicateArity,int argumentPosition,int resultPosition) {
        ConceptFunctionInfo conceptFunctionInfo=new ConceptFunctionInfo(functionName,predicateName,predicateArity,argumentPosition,resultPosition);
        s_conceptFunctions.put(conceptFunctionInfo.m_functionName,conceptFunctionInfo);
    }

    /** The factory for the predicates in the datalog program. */
    protected PredicateFactory m_predicateFactory;
    /** The list of generated rules. */
    protected List m_rules;
    /** The counter of automatically generated expressions. */
    protected int m_lastAutomaticExpressionIndex;
    /** The map of property names to <code>Boolean</code> objects. If Boolean is TRUE, then the property is an attribute. */
    protected Map m_propertyTypes;

    /**
     * Creates an instance of this class.
     *
     * @param predicateFactory                      the factory for the program predicates
     */
    public ExpressionToDatalogCompiler(PredicateFactory predicateFactory) {
        m_predicateFactory=predicateFactory;
        m_rules=new ArrayList();
        m_propertyTypes=new HashMap();
    }
    /**
     * Returns the list of generated rules.
     *
     * @return                                      the list of generated rules
     */
    public List getRules() {
        return m_rules;
    }
    /**
     * Returns the program containing generated rules.
     *
     * @return                                      the program
     */
    public Program getProgram() {
        Rule[] ruleArray=new Rule[m_rules.size()];
        m_rules.toArray(ruleArray);
        return new Program(ruleArray);
    }
    /**
     * Compiles given expression into a rule and returns a predicate for invoking the rule.
     *
     * @param expression                            the expression being compiled
     * @param expressionName                        the name for the expression (if <code>null</code> a name will be generated automatically)
     * @return                                      the predicate that can be used to invoke the expression
     * @throws CompilationException                 thrown if there is an error
     */
    public Predicate compileExpression(Expression expression,String expressionName) throws CompilationException {
        CollectPropertyNamesVisitor visitor=new CollectPropertyNamesVisitor();
        expression.accept(visitor);
        Set propertiesToLoad=new HashSet();
        Set propertyNames=visitor.getPropertyNames();
        Iterator iterator=propertyNames.iterator();
        while (iterator.hasNext()) {
            String propertyName=(String)iterator.next();
            if (!m_propertyTypes.keySet().contains(propertyName))
                propertiesToLoad.add(propertyName);
        }
        if (!propertiesToLoad.isEmpty())
            loadPropertyTypes(propertiesToLoad);
        propertyNames.removeAll(m_propertyTypes.keySet());
        if (!propertyNames.isEmpty()) {
            StringBuffer buffer=new StringBuffer();
            buffer.append("Properties not in model: ");
            iterator=propertyNames.iterator();
            while (iterator.hasNext()) {
                String propertyName=(String)iterator.next();
                buffer.append('\n');
                buffer.append('\'');
                buffer.append(propertyName);
                buffer.append('\'');
            }
            throw new CompilationException(buffer.toString());
        }
        RuleBodyBuilder builder=new RuleBodyBuilder();
        return builder.generateRule(expression,expressionName);
    }
    /**
     * Generates a name for the expression.
     *
     * @return                                      a new name for the expression
     */
    protected String generateExpressionName() {
        return "Expression_"+(++m_lastAutomaticExpressionIndex);
    }
    /**
     * Converts the URI to a constant.
     *
     * @param entityURI                             the URI of the entity to be converted
     * @return                                      the constant
     */
    protected Constant entityURIConstant(String entityURI) {
        return new Constant(entityURI);
    }
    /**
     * Converts a literal to a constant.
     *
     * @param literal                               the literal to be converted
     * @return                                      the constant
     */
    protected Constant literalConstant(Object literal) {
        return new Constant(literal);
    }
    /**
     * Checks whether the property is the attribute.
     *
     * @param propertyName                          the name of the property
     * @return                                      <code>true</code> if the property is the attribute
     */
    protected boolean isPropertyAttribute(String propertyName) {
        return ((Boolean)m_propertyTypes.get(propertyName)).booleanValue();
    }
    /**
     * Retrieves the types of supplied properties.
     *
     * @param propertyNames                         the set of property names
     * @throws CompilationException                 thrown if there is an error
     */
    protected abstract void loadPropertyTypes(Set propertyNames) throws CompilationException;

    /**
     * The visitor for the expression that creates the array of rule's body literals for the expresion.
     */
    protected class RuleBodyBuilder implements ExpressionVisitor {
        /** The array of rule literals. */
        protected List m_literals;
        /** The stack of free terms. */
        protected Stack m_freeTermsStack;
        /** The index of the last variable. */
        protected int m_lastVariableIndex;

        /**
         * Creates an instance of this class.
         */
        public RuleBodyBuilder() {
            m_literals=new ArrayList();
            m_freeTermsStack=new Stack();
            pushFreeTerms();
        }
        /**
         * Generates a rule for given expression.
         *
         * @param expression                        the expression for which the rule will be generated
         * @param expressionName                    the name of the expression (may be <code>null</code> - then the name is invented)
         * @return                                  the head predicate of the rule
         */
        public Predicate generateRule(Expression expression,String expressionName) {
            expression.accept(this);
            return generateRule(expressionName);
        }
        /**
         * Generates a rule from the current state of the builder.
         *
         * @param expressionName                    the name of the expression (may be <code>null</code> - then the name is invented)
         * @return                                  the head predicate of the rule
         */
        public Predicate generateRule(String expressionName) {
            if (expressionName==null)
                expressionName=generateExpressionName();
            Term[] headTerms=getFreeTerms();
            Predicate headPredicate=m_predicateFactory.getPredicate(expressionName,headTerms.length);
            m_rules.add(new Rule(null,new Literal(headPredicate,true,headTerms),getLiterals()));
            return headPredicate;
        }
        /**
         * Returns the array of literals of the rule.
         *
         * @return                                  the array of literals of the rule
         */
        protected Literal[] getLiterals() {
            Literal[] literals=new Literal[m_literals.size()];
            m_literals.toArray(literals);
            return literals;
        }
        /**
         * Returns the array of the free terms of the rule.
         *
         * @return                                  the array of the free terms of the rule
         */
        protected Term[] getFreeTerms() {
            List freeTerms=getFreeTermsList();
            Term[] terms=new Term[freeTerms.size()];
            freeTerms.toArray(terms);
            return terms;
        }
        /**
         * Visits the primitive concept.
         *
         * @param expression                        the expression
         */
        public void visit(PrimitiveConcept expression) {
            m_literals.add(new Literal(m_predicateFactory.getPredicate("ConceptInstance_a",2),true,new Term[] { entityURIConstant(expression.getConceptURI()),getFreeTerm(0) }));
        }
        /**
         * Visits the nominal concept.
         *
         * @param expression                        the expression
         */
        public void visit(NominalConceptExpression expression) {
            if (!(expression.getInstanceIdentifiers().isEmpty() && expression.getConstants().isEmpty())) {
                Predicate nominalPredicate=m_predicateFactory.getPredicate(generateExpressionName(),1);
                m_literals.add(new Literal(nominalPredicate,true,new Term[] { getFreeTerm(0) }));
                Iterator iterator=expression.getInstanceIdentifiers().iterator();
                while (iterator.hasNext()) {
                    String instanceIdentifier=(String)iterator.next();
                    Constant identifierConstant=entityURIConstant(instanceIdentifier);
                    m_rules.add(new Rule(null,new Literal(nominalPredicate,true,new Term[] { identifierConstant }),EMPTY_LITERALS));
                }
                iterator=expression.getConstants().iterator();
                while (iterator.hasNext()) {
                    Object object=iterator.next();
                    Constant constant=literalConstant(object);
                    m_rules.add(new Rule(null,new Literal(nominalPredicate,true,new Term[] { constant }),EMPTY_LITERALS));
                }
            }
        }
        /**
         * Visits the concept whose instances are strings defined by a regular expression.
         *
         * @param expression                        the expression
         */
        public void visit(RegExpConceptExpression expression) {
            m_literals.add(new Literal(m_predicateFactory.getPredicate("RegExp",2),true,new Term[] { getFreeTerm(0),literalConstant(expression.getRegularExpression()), }));
        }
        /**
         * Visits the functional concept expression.
         *
         * @param expression                        the expression
         */
        public void visit(FunctionalConceptExpression expression) {
            ConceptFunctionInfo conceptFunctionInfo=(ConceptFunctionInfo)s_conceptFunctions.get(expression.getFunctionName());
            if (conceptFunctionInfo==null)
                conceptFunctionInfo=new ConceptFunctionInfo(expression.getFunctionName(),expression.getFunctionName(),2,0,1);
            Term argumentTerm=abbreviateAsConstant(expression.getConceptExpression());
            if (argumentTerm==null) {
                pushFreeTerms();
                expression.getConceptExpression().accept(this);
                argumentTerm=getFreeTerm(0);
                popFreeTerms();
            }
            Term resultTerm=getFreeTerm(0);
            Term[] bodyTerms=new Term[conceptFunctionInfo.m_predicateArity];
            bodyTerms[conceptFunctionInfo.m_argumentPosition]=argumentTerm;
            bodyTerms[conceptFunctionInfo.m_resultPosition]=resultTerm;
            for (int i=0;i<bodyTerms.length;i++)
                if (bodyTerms[i]==null)
                    bodyTerms[i]=newVariable();
            Predicate functionPredicate=m_predicateFactory.getPredicate(conceptFunctionInfo.m_predicateName,conceptFunctionInfo.m_predicateArity);
            m_literals.add(new Literal(functionPredicate,true,bodyTerms));
        }
        /**
         * Visits the ALL restriction.
         *
         * @param expression                        the expression
         */
        public void visit(AllRestriction expression) {
            if (!isCurrentConceptRangeRestricted())
                m_literals.add(new Literal(m_predicateFactory.getPredicate("ConceptInstance_a",2),true,new Term[] { entityURIConstant(PrimitiveConcept.ROOT),getFreeTerm(0) }));
            RuleBodyBuilder innerBuilder=new RuleBodyBuilder();
            innerBuilder.pushFreeTerms(innerBuilder.getFreeTerm(0));
            expression.getPropertyExpression().accept(innerBuilder);
            if (expression.getConceptExpression() instanceof PrimitiveConcept) {
                PrimitiveConcept primitiveConcept=(PrimitiveConcept)expression.getConceptExpression();
                innerBuilder.m_literals.add(new Literal(m_predicateFactory.getPredicate("ConceptInstance_a",2),false,new Term[] { entityURIConstant(primitiveConcept.getConceptURI()),innerBuilder.getFreeTerm(1) }));
            }
            else {
                RuleBodyBuilder innerConceptBuilder=new RuleBodyBuilder();
                Predicate headPredicate=innerConceptBuilder.generateRule(expression.getConceptExpression(),null);
                innerBuilder.m_literals.add(new Literal(headPredicate,false,new Term[] { innerBuilder.getFreeTerm(1) }));
            }
            innerBuilder.popFreeTerms();
            Predicate headPredicate=innerBuilder.generateRule(null);
            m_literals.add(new Literal(headPredicate,false,new Term[] { getFreeTerm(0) }));
        }
        /**
         * Visits the SOME restriction.
         *
         * @param expression                        the expression
         */
        public void visit(SomeRestriction expression) {
            Term secondTerm=abbreviateAsConstant(expression.getConceptExpression());
            pushFreeTerms(getFreeTerm(0),secondTerm);
            expression.getPropertyExpression().accept(this);
            if (secondTerm==null) {
                secondTerm=getFreeTerm(1);
                popFreeTerms();
                pushFreeTerms(secondTerm);
                expression.getConceptExpression().accept(this);
            }
            popFreeTerms();
        }
        /**
         * Visits the NOT expression.
         *
         * @param expression                        the expression
         */
        public void visit(NotExpression expression) {
            if (!isCurrentConceptRangeRestricted())
                m_literals.add(new Literal(m_predicateFactory.getPredicate("ConceptInstance_a",2),true,new Term[] { entityURIConstant(PrimitiveConcept.ROOT),getFreeTerm(0) }));
            if (expression.getConceptExpression() instanceof PrimitiveConcept) {
                PrimitiveConcept primitiveConcept=(PrimitiveConcept)expression.getConceptExpression();
                m_literals.add(new Literal(m_predicateFactory.getPredicate("ConceptInstance_a",2),false,new Term[] { entityURIConstant(primitiveConcept.getConceptURI()),getFreeTerm(0) }));
            }
            else {
                RuleBodyBuilder innerBuilder=new RuleBodyBuilder();
                Predicate headPredicate=innerBuilder.generateRule(expression.getConceptExpression(),null);
                m_literals.add(new Literal(headPredicate,false,new Term[] { getFreeTerm(0) }));
            }
        }
        /**
         * Visits the AND expression.
         *
         * @param expression                        the expression
         */
        public void visit(AndExpression expression) {
            expression.getConceptExpression1().accept(this);
            expression.getConceptExpression2().accept(this);
        }
        /**
         * Visits the OR expression.
         *
         * @param expression                        the expression
         */
        public void visit(OrExpression expression) {
            RuleBodyBuilder innerBuilder=new RuleBodyBuilder();
            Predicate headPredicate=innerBuilder.generateRule(expression.getConceptExpression1(),null);
            innerBuilder=new RuleBodyBuilder();
            innerBuilder.generateRule(expression.getConceptExpression2(),headPredicate.getSimpleName());
            m_literals.add(new Literal(headPredicate,true,new Term[] { getFreeTerm(0) }));
        }
        /**
         * Visits the property projection.
         *
         * @param expression                        the expression
         */
        public void visit(PropertyProjection expression) {
            if (expression.getProjectionPosition()==1)
                pushFreeTerms(getFreeTerm(0),null);
            else
                pushFreeTerms(null,getFreeTerm(0));
            expression.getPropertyExpression().accept(this);
            popFreeTerms();
        }
        /**
         * Visits the primitive property.
         *
         * @param expression                        the expression
         */
        public void visit(PrimitiveProperty expression) {
            String predicateName=isPropertyAttribute(expression.getPropertyURI()) ? "AttributeInstance_a" : "RelationInstance_a";
            m_literals.add(new Literal(m_predicateFactory.getPredicate(predicateName,3),true,new Term[] { entityURIConstant(expression.getPropertyURI()),getFreeTerm(0),getFreeTerm(1) }));
        }
        /**
         * Visits the functional property expression.
         *
         * @param expression                        the expression
         */
        public void visit(FunctionalPropertyExpression expression) {
            pushFreeTerms();
            expression.getPropertyExpression().accept(this);
            Term argumentTerm=getFreeTerm(expression.getApplicationPosition()-1);
            popFreeTerms();
            Term outerTerm=getFreeTerm(expression.getApplicationPosition()-1);
            m_literals.add(new Literal(m_predicateFactory.getPredicate(expression.getFunctionName(),2),true,new Term[] { argumentTerm,outerTerm }));
        }
        /**
         * Visits the inverse property.
         *
         * @param expression                        the expression
         */
        public void visit(InverseProperty expression) {
            pushFreeTerms(getFreeTerm(1),getFreeTerm(0));
            expression.getPropertyExpression().accept(this);
            popFreeTerms();
        }
        /**
         * Visits the path expression.
         *
         * @param expression                        the expression
         */
        public void visit(PathExpression expression) {
            Term firstTerm=getFreeTerm(0);
            Term secondTerm=getFreeTerm(1);
            Term middleTerm=newVariable();
            pushFreeTerms(firstTerm,middleTerm);
            expression.getPropertyExpression1().accept(this);
            popFreeTerms();
            if (!expression.isConceptExpressionRootConcept()) {
                pushFreeTerms(middleTerm);
                expression.getConceptExpression().accept(this);
                popFreeTerms();
            }
            pushFreeTerms(middleTerm,secondTerm);
            expression.getPropertyExpression2().accept(this);
            popFreeTerms();
        }
        /**
         * Visits the concept multiplication.
         *
         * @param expression                        the expression
         */
        public void visit(MultiplyExpression expression) {
            Term firstTerm=getFreeTerm(0);
            Term secondTerm=getFreeTerm(1);
            pushFreeTerms(firstTerm);
            expression.getConceptExpression1().accept(this);
            popFreeTerms();
            pushFreeTerms(secondTerm);
            expression.getConceptExpression2().accept(this);
            popFreeTerms();
        }
        /**
         * Visits the property selection.
         *
         * @param expression                        the expression
         */
        public void visit(PropertySelection expression) {
            pushFreeTerms(getFreeTerm(0),getFreeTerm(1));
            expression.getPropertyExpression().accept(this);
            Term selectionTerm=getFreeTerm(expression.getSelectionPosition()-1);
            popFreeTerms();
            pushFreeTerms(selectionTerm);
            expression.getConceptExpression().accept(this);
            popFreeTerms();
        }
        /**
         * Returns the list of free terms from the top of the stack.
         *
         * @return                                  the list of free terms from the top of the stack
         */
        protected List getFreeTermsList() {
            return (List)m_freeTermsStack.peek();
        }
        /**
         * Creates the new variable.
         *
         * @return                                  the new variable
         */
        protected Term newVariable() {
            return new Variable("X"+(++m_lastVariableIndex));
        }
        /**
         * Returns the free term with given index.
         *
         * @param index                             the index of the free term
         * @return                                  the free term with given index
         */
        protected Term getFreeTerm(int index) {
            List freeTerms=getFreeTermsList();
            while (freeTerms.size()<=index)
                freeTerms.add(null);
            Term term=(Term)freeTerms.get(index);
            if (term==null) {
                term=newVariable();
                freeTerms.set(index,term);
            }
            return term;
        }
        /**
         * Pushes the empty list of free terms onto the stack.
         */
        protected void pushFreeTerms() {
            m_freeTermsStack.push(new ArrayList());
        }
        /**
         * Pushes the free terms onto the stack.
         *
         * @param term0                             the first term to be pushed
         */
        protected void pushFreeTerms(Term term0) {
            List freeTerms=new ArrayList();
            freeTerms.add(term0);
            m_freeTermsStack.push(freeTerms);
        }
        /**
         * Pushes the free terms onto the stack.
         *
         * @param term0                             the first term to be pushed
         * @param term1                             the second term to be pushed
         */
        protected void pushFreeTerms(Term term0,Term term1) {
            List freeTerms=new ArrayList();
            freeTerms.add(term0);
            freeTerms.add(term1);
            m_freeTermsStack.push(freeTerms);
        }
        /**
         * Pops the free terms off the stack.
         */
        protected void popFreeTerms() {
            m_freeTermsStack.pop();
        }
        /**
         * Returns <code>true</code> if the current concept has been range-restricted.
         *
         * @return                                  <code>true</code> if the current concept has been range restricted
         */
        protected boolean isCurrentConceptRangeRestricted() {
            List freeTerms=getFreeTermsList();
            return !freeTerms.isEmpty() && freeTerms.get(0)!=null;
        }
        /**
         * Tries to abbreviate the concept expression as constant.
         *
         * @param conceptExpression                 the concept expression
         * @return                                  the constant (or <code>null</code> if abbreviation can't be done)
         */
        protected Constant abbreviateAsConstant(ConceptExpression conceptExpression) {
            if (conceptExpression instanceof NominalConceptExpression) {
                NominalConceptExpression nominalConceptExpression=(NominalConceptExpression)conceptExpression;
                if (nominalConceptExpression.getConstants().size()==1 && nominalConceptExpression.getInstanceIdentifiers().size()==0)
                    return literalConstant(nominalConceptExpression.getConstants().iterator().next());
                else if (nominalConceptExpression.getConstants().size()==0 && nominalConceptExpression.getInstanceIdentifiers().size()==1)
                    return entityURIConstant((String)nominalConceptExpression.getInstanceIdentifiers().iterator().next());
            }
            return null;
        }
    }

    /**
     * The information about concept functions.
     */
    protected static class ConceptFunctionInfo {
        /** The name of the function. */
        protected String m_functionName;
        /** The name of the predicate implementing the function. */
        protected String m_predicateName;
        /** The arity of the predicate. */
        protected int m_predicateArity;
        /** The position of the argument in the predicate. */
        protected int m_argumentPosition;
        /** The position of the result in the predicate. */
        protected int m_resultPosition;

        /**
         * Creates an instance of this class.
         *
         * @param functionName                      the name of the function
         * @param predicateName                     the name of the predicate implementing the function
         * @param predicateArity                    the arity of the predicate implementing the function
         * @param argumentPosition                  the position of the argument variable in the predicate
         * @param resultPosition                    the position of the result variable in the predicate
         */
        public ConceptFunctionInfo(String functionName,String predicateName,int predicateArity,int argumentPosition,int resultPosition) {
            m_functionName=functionName;
            m_predicateName=predicateName;
            m_predicateArity=predicateArity;
            m_argumentPosition=argumentPosition;
            m_resultPosition=resultPosition;
        }
    }
}
