package de.fzi.wim.oimodeler.ui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import de.fzi.wim.guibase.appdriver.AbstractModule;

/**
 * A general dialog for displaing the progress of some process.
 */
public class ProgressDlg extends JDialog {
    /** The OI-modeler module. */
    protected AbstractModule m_module;
    /** The label displaing the progress. */
    protected JLabel m_progress;
    /** The ID prefix of the dialog. */
    protected String m_prefixID;
    /** The thread for the background process. */
    protected Thread m_thread;

    /**
     * Creates an instance of this class.
     *
     * @param module                    the module
     * @param prefixID                  the prefix for the strings
     * @param canAbort                  if <code>true</code>, the operation can be aborted
     */
    public ProgressDlg(AbstractModule module,String prefixID,boolean canAbort) {
        super(module.getAppDriver().getMainFrameWindow(),module.getAppDriver().getLocalizationManager().getPhrase(prefixID+".title"),false);
        m_module=module;
        m_prefixID=prefixID;
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        m_progress=m_module.getAppDriver().getLocalizationManager().getLabel(m_prefixID+".initial");
        m_progress.setPreferredSize(new Dimension(400,m_progress.getPreferredSize().height));
        JPanel progressPane=new JPanel(new BorderLayout());
        progressPane.add(m_progress,BorderLayout.CENTER);
        progressPane.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        JPanel rootPane=new JPanel(new BorderLayout());
        rootPane.add(progressPane,BorderLayout.CENTER);
        if (canAbort) {
            JButton cancelButton=m_module.getAppDriver().getLocalizationManager().getButton("oimodeler.cancel");
            cancelButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    operationAborted();
                }
            });
            addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    operationAborted();
                }
            });
            JPanel buttonsPane=new JPanel(new FlowLayout(FlowLayout.RIGHT));
            buttonsPane.add(cancelButton);
            buttonsPane.setBorder(BorderFactory.createEmptyBorder(0,5,5,0));
            rootPane.add(buttonsPane,BorderLayout.SOUTH);
        }
        setContentPane(rootPane);
        pack();
        setLocationRelativeTo(m_module.getAppDriver().getMainFrameWindow());
    }
    /**
     * Sets the new prefix ID.
     *
     * @param prefixID                  the prefix ID for the dialog
     */
    public void setPrefixID(String prefixID) {
        m_prefixID=prefixID;
    }
    /**
     * Called when the cancel is pressed.
     */
    public void operationAborted() {
        m_thread.interrupt();
    }
    /**
     * Updates the progress display.
     *
     * @param phase                     the current phase
     * @param numberOfPhases            the total number of phases
     * @param stepsDone                 the number of steps that were preformed
     * @param totalSteps                the total number of steps to be performed
     */
    public void updateProgress(final int phase,final int numberOfPhases,final int stepsDone,final int totalSteps) {
        if (SwingUtilities.isEventDispatchThread())
            updateProgressInternal(phase,numberOfPhases,stepsDone,totalSteps);
        else
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    updateProgressInternal(phase,numberOfPhases,stepsDone,totalSteps);
                }
            });
    }
    /**
     * Updates the progress display on the main thread.
     *
     * @param phase                     the current phase
     * @param numberOfPhases            the total number of phases
     * @param stepsDone                 the number of steps that were preformed
     * @param totalSteps                the total number of steps to be performed
     */
    protected void updateProgressInternal(int phase,int numberOfPhases,int stepsDone,int totalSteps) {
        String text=m_module.getAppDriver().getLocalizationManager().format(m_prefixID+".phase_"+phase,new Object[] { new Integer(phase),new Integer(numberOfPhases),new Integer(stepsDone),new Integer(totalSteps) });
        m_progress.setText(text);
    }
    /**
     * Executes the supplied asynchronous action on a background thread.
     *
     * @param asynchronousAction        the action to be executed
     */
    public void executeAction(final AsynchronousAction asynchronousAction) {
        if (m_thread!=null)
            throw new IllegalStateException("Asynchronous action is already executing.");
        m_thread=new Thread() {
            public void run() {
                try {
                    asynchronousAction.run(ProgressDlg.this);
                    finishAction(asynchronousAction,true);
                }
                catch (InterruptedException error) {
                    finishAction(asynchronousAction,false);
                }
                catch (final Exception error) {
                    finishAction(asynchronousAction,false);
                    SwingUtilities.invokeLater(new Runnable() {
                        public void run() {
                            m_module.getAppDriver().displayErrorNotification(error);
                        }
                    });
                }
            }
        };
        m_thread.start();
        setVisible(true);
    }
    /**
     * Ends the action.
     *
     * @param asynchronousAction        the action to be executed
     * @param success                   <code>true</code> if the action is finished successfully
     */
    protected void finishAction(final AsynchronousAction asynchronousAction,final boolean success) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                dispose();
                asynchronousAction.completion(ProgressDlg.this,success);
                try {
                    m_thread.join();
                }
                catch (InterruptedException ignored) {
                }
                m_thread=null;
            }
        });
    }

    /**
     * Interface specifying the action to be performed.
     */
    public interface AsynchronousAction {
        void run(ProgressDlg progressDlg) throws Exception;
        void completion(ProgressDlg progressDlg,boolean success);
    }
}
