package de.fzi.wim.oimodeler.ui;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.Iterator;
import java.util.Collection;
import java.util.Arrays;
import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.NumberFormat;
import java.text.ParseException;
import java.awt.Insets;
import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.KeyEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.BorderFactory;
import javax.swing.KeyStroke;
import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JTextField;
import javax.swing.JFormattedTextField;
import javax.swing.JPasswordField;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JTabbedPane;
import javax.swing.JComboBox;
import javax.swing.JScrollPane;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.text.NumberFormatter;
import javax.swing.table.DefaultTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.appdriver.*;
import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.guibase.util.*;
import de.fzi.wim.guibase.configuration.*;
import de.fzi.wim.guibase.tables.*;
import de.fzi.wim.guibase.actions.*;

/**
 * Simple dialog for entering the data about a new OI-model.
 */
public class OpenOIModelDlg extends JDialog {
    /** The modus specifying that OI-model is being opened. */
    public static final int OPEN_MODUS=0;
    /** The modus specifying that OI-model is being created. */
    public static final int CREATE_MODUS=1;
    /** The modus specifying that OI-model is being copied. */
    public static final int COPY_MODUS=2;

    /** The module that opened this dialog. */
    protected Module m_module;
    /** The modus of this dialog. */
    protected int m_dialogModus;
    /** The logical URIs to be mapped to physical ones. */
    protected Collection m_logicalURIs;
    /** Tabbed pane for various types of servers. */
    protected JTabbedPane m_serverTypes;
    /** Set to <code>true</code> if dialog is contirmed. */
    protected boolean m_confirmed;
    /** The configuration for storing connection parameters. */
    protected Configuration m_connectionParameters;

    /**
     * Creates an instance of this class.
     *
     * @param module                                the module that opens this dialog
     * @param dialogModus                           the modus of the dialog
     * @param titleID                               the ID of the title
     * @param logicalURIs                           the logical URIs that must be mapped to physical ones
     */
    public OpenOIModelDlg(Module module,int dialogModus,String titleID,Collection logicalURIs) {
        super(module.getAppDriver().getMainFrameWindow(),module.getAppDriver().getLocalizationManager().getPhrase(titleID),true);
        m_module=module;
        m_dialogModus=dialogModus;
        m_logicalURIs=logicalURIs;
        m_connectionParameters=m_module.getAppDriver().getConfiguration().getSubConfiguration(titleID);
        LocalizationManager localizationManager=m_module.getAppDriver().getLocalizationManager();
        setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        m_serverTypes=new JTabbedPane();
        registerServerTypes();
        String selectedServerType=m_connectionParameters.getString("selectedServerType","--");
        for (int i=0;i<m_serverTypes.getComponentCount();i++) {
            String title=m_serverTypes.getTitleAt(i);
            int mnemonic=Mnemonics.getLabelMnemonic(title);
            if (mnemonic!=-1) {
                m_serverTypes.setTitleAt(i,Mnemonics.getLabelText(title));
                m_serverTypes.setMnemonicAt(i,mnemonic);
            }
            ServerType serverType=(ServerType)m_serverTypes.getComponent(i);
            if (selectedServerType.equals(serverType.getServerTypeName()))
                m_serverTypes.setSelectedIndex(i);
        }
        m_serverTypes.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        JButton okButton=localizationManager.getButton("oimodeler.OK");
        okButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                m_confirmed=true;
                ServerType serverType=(ServerType)m_serverTypes.getSelectedComponent();
                serverType.updateConfiguration();
                m_connectionParameters.setString("selectedServerType",serverType.getServerTypeName());
                m_module.getAppDriver().saveConfiguration();
                dispose();
            }
        });
        JButton cancelButton=localizationManager.getButton("oimodeler.cancel");
        cancelButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                dispose();
            }
        });
        JPanel buttonsPane=new JPanel(new FlowLayout(FlowLayout.RIGHT));
        buttonsPane.add(okButton);
        buttonsPane.add(cancelButton);
        buttonsPane.setBorder(BorderFactory.createEmptyBorder(0,5,5,0));
        JPanel rootPane=new JPanel(new BorderLayout());
        rootPane.add(m_serverTypes,BorderLayout.CENTER);
        rootPane.add(buttonsPane,BorderLayout.SOUTH);
        setContentPane(rootPane);
        getRootPane().setDefaultButton(okButton);
        pack();
        setLocationRelativeTo(m_module.getAppDriver().getMainFrameWindow());
    }
    /**
     * Registers all server types.
     */
    protected void registerServerTypes() {
        m_serverTypes.add(new RDFServerType());
        m_serverTypes.add(new RemoteEngineeringServerType());
        m_serverTypes.add(new DirectEngineeringServerType());
        m_serverTypes.add(new OtherServerType());
    }
    /**
     * Returns whether dialog was confirmed with OK.
     *
     * @return                      <code>true</code> if dialog has been confirmed
     */
    public boolean isConfirmed() {
        return m_confirmed;
    }
    /**
     * Returns the logical URI of the OI-model.
     *
     * @return                      the URI of the OI-model
     */
    public String getOIModelLogicalURI() {
        ServerType serverType=(ServerType)m_serverTypes.getSelectedComponent();
        return serverType.getOIModelLogicalURI();
    }
    /**
     * Returns the physical URI of the OI-model.
     *
     * @return                      the URI of the OI-model
     */
    public String getOIModelPhysicalURI() {
        ServerType serverType=(ServerType)m_serverTypes.getSelectedComponent();
        return serverType.getOIModelPhysicalURI();
    }
    /**
     * Returns the map of parameters for establishing the KAON connection.
     *
     * @return                      the KAON connection parameters
     */
    public Map getConnectionParameters() {
        ServerType serverType=(ServerType)m_serverTypes.getSelectedComponent();
        return serverType.getConnectionParameters();
    }
    /**
     * Returns the map of logical to physical URIs.
     *
     * @return                      the map of logical to physical URIs
     */
    public Map getLogicalToPhysicalURIsMap() {
        ServerType serverType=(ServerType)m_serverTypes.getSelectedComponent();
        Map map=new HashMap();
        Iterator iterator=m_logicalURIs.iterator();
        while (iterator.hasNext()) {
            String logicalURI=(String)iterator.next();
            String physicalURI=serverType.getPhysicalForLogicalURI(logicalURI);
            map.put(logicalURI,physicalURI);
        }
        return map;
    }

    /**
     * Interface for each type of the server.
     */
    protected interface ServerType {
        String getServerTypeName();
        String getOIModelLogicalURI();
        String getOIModelPhysicalURI();
        Map getConnectionParameters();
        void updateConfiguration();
        String getPhysicalForLogicalURI(String logicalURI);
    }

    /**
     * Server type for RDF models.
     */
    protected class RDFServerType extends JPanel implements ServerType {
        /** Editor for logical OI-model URI. */
        protected JComboBox m_oimodelLogicalURI;
        /** Editor for physical OI-model URI. */
        protected JTextField m_oimodelPhysicalURI;
        /** The URI mapper. */
        protected LogicalToPhysicalURIsMapping m_logicalToPhysicalURIsMapping;

        public RDFServerType() {
            super(new BorderLayout());
            Configuration configuration=getConfiguration();
            final LocalizationManager localizationManager=m_module.getAppDriver().getLocalizationManager();
            setName(localizationManager.getPhrase("oimodeler."+getServerTypeName()));
            JPanel physicalURIAndButton=null;
            if (m_dialogModus!=COPY_MODUS) {
                m_oimodelPhysicalURI=new JTextField();
                m_oimodelPhysicalURI.setPreferredSize(new Dimension(200,m_oimodelPhysicalURI.getPreferredSize().height));
                m_oimodelPhysicalURI.setText(configuration.getString("oimodelPhysicalURI"));
                JButton browseButton=m_module.getAppDriver().getLocalizationManager().getButton("oimodeler.browse");
                browseButton.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        JFileChooserEx fileChooser=new JFileChooserEx();
                        ExtensionFileFilter defaultFilter=new ExtensionFileFilter(localizationManager,"oimodeler.allOIModelerFiles",new String[] { "kaon" });
                        fileChooser.addChoosableFileFilter(defaultFilter);
                        Configuration configuration=getConfiguration();
                        String currentDirectory=configuration.getString("currentDirectory");
                        if (currentDirectory!=null)
                            fileChooser.setCurrentDirectory(new File(currentDirectory));
                        if (fileChooser.showOpenDialog(m_module.getAppDriver().getMainFrameWindow())==JFileChooserEx.APPROVE_OPTION) {
                            String uri=fileChooser.getSelectedFileEx().toURI().toString();
                            m_oimodelPhysicalURI.setText(uri);
                            configuration.setString("currentDirectory",fileChooser.getCurrentDirectory().getAbsolutePath());
                        }
                    }
                });
                physicalURIAndButton=new JPanel(new BorderLayout(10,0));
                physicalURIAndButton.add(m_oimodelPhysicalURI,BorderLayout.CENTER);
                physicalURIAndButton.add(browseButton,BorderLayout.EAST);
            }
            if (m_dialogModus==CREATE_MODUS) {
                LogicalURIComboBoxModel logicalURIComboBoxModel=new LogicalURIComboBoxModel(this);
                m_oimodelLogicalURI=new JComboBox(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setPreferredSize(new Dimension(200,m_oimodelLogicalURI.getPreferredSize().height));
                m_oimodelLogicalURI.setEditable(true);
                m_oimodelLogicalURI.addPopupMenuListener(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setSelectedItem(configuration.getString("oimodelLogicalURI"));
            }
            if (m_dialogModus==COPY_MODUS)
                m_logicalToPhysicalURIsMapping=new LogicalToPhysicalURIsMapping(m_logicalURIs);
            JPanel controls=new JPanel(new GridBagLayout());
            int index=0;
            if (physicalURIAndButton!=null)
                addPair(controls,index++,"oimodeler.oimodelPhysicalURI",physicalURIAndButton,true);
            if (m_oimodelLogicalURI!=null)
                addPair(controls,index++,"oimodeler.oimodelLogicalURI",m_oimodelLogicalURI,true);
            if (m_logicalToPhysicalURIsMapping!=null) {
                addPair(controls,index++,"oimodeler.logicalToPhysicalURIMappings",null,true);
                JScrollPane mappingsScrollPane=new JScrollPane(m_logicalToPhysicalURIsMapping);
                mappingsScrollPane.setPreferredSize(new Dimension(500,200));
                add(mappingsScrollPane,BorderLayout.CENTER);
            }
            add(controls,BorderLayout.NORTH);
            setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        }
        public String getServerTypeName() {
            return "rdfServerType";
        }
        public String getOIModelLogicalURI() {
            String logicalURI=(String)m_oimodelLogicalURI.getSelectedItem();
            if (logicalURI==null)
                return "";
            else
                return logicalURI.trim();
        }
        public String getOIModelPhysicalURI() {
            return m_oimodelPhysicalURI.getText().trim();
        }
        public Map getConnectionParameters() {
            Map parameters=new HashMap();
            parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.apionrdf.KAONConnectionImpl");
            return parameters;
        }
        protected Configuration getConfiguration() {
            return m_connectionParameters.getSubConfiguration(getServerTypeName());
        }
        public void updateConfiguration() {
            Configuration configuration=getConfiguration();
            if (m_oimodelPhysicalURI!=null)
                configuration.setString("oimodelPhysicalURI",getOIModelPhysicalURI());
            if (m_oimodelLogicalURI!=null)
                configuration.setString("oimodelLogicalURI",getOIModelLogicalURI());
        }
        public String getPhysicalForLogicalURI(String logicalURI) {
            return m_logicalToPhysicalURIsMapping.getPhysicalForLogicalURI(logicalURI);
        }
    }

    /**
     * Server type for remote enginnering server.
     */
    protected class RemoteEngineeringServerType extends JPanel implements ServerType {
        /** Editor for host name. */
        protected JTextField m_hostName;
        /** Editor for port. */
        protected JFormattedTextField m_port;
        /** Chooser for logical OI-model URI. */
        protected JComboBox m_oimodelLogicalURI;
        /** Editor for user name. */
        protected JTextField m_userName;
        /** Editor for password. */
        protected JPasswordField m_password;
        /** Check box specifying whether on-line updates are required. */
        protected JCheckBox m_trackUpdates;

        public RemoteEngineeringServerType() {
            super(new BorderLayout());
            Configuration configuration=getConfiguration();
            final LocalizationManager localizationManager=m_module.getAppDriver().getLocalizationManager();
            setName(localizationManager.getPhrase("oimodeler."+getServerTypeName()));
            m_hostName=new JTextField();
            m_hostName.setText(configuration.getString("hostName","localhost"));
            NumberFormat numberFormat=NumberFormat.getInstance();
            numberFormat.setGroupingUsed(false);
            m_port=new JFormattedTextField(new NumberFormatter(numberFormat));
            m_port.setPreferredSize(new Dimension(100,m_port.getPreferredSize().height));
            m_port.setValue(new Integer(configuration.getInt("port",1099)));
            if (m_dialogModus!=COPY_MODUS) {
                LogicalURIComboBoxModel logicalURIComboBoxModel=new LogicalURIComboBoxModel(this);
                m_oimodelLogicalURI=new JComboBox(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setPreferredSize(new Dimension(200,m_oimodelLogicalURI.getPreferredSize().height));
                m_oimodelLogicalURI.setEditable(true);
                m_oimodelLogicalURI.addPopupMenuListener(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setSelectedItem(configuration.getString("oimodelLogicalURI"));
            }
            m_userName=new JTextField();
            m_userName.setText(configuration.getString("userName"));
            m_password=new JPasswordField();
            m_trackUpdates=new JCheckBox();
            m_trackUpdates.setSelected(configuration.getBoolean("trackUpdates",false));
            Mnemonics.setText(m_trackUpdates,localizationManager.getPhrase("oimodeler.trackUpdates"));
            JPanel controls=new JPanel(new GridBagLayout());
            int index=0;
            addPair(controls,index++,"oimodeler.hostName",m_hostName,true);
            addPair(controls,index++,"oimodeler.port",m_port,false);
            if (m_oimodelLogicalURI!=null)
                addPair(controls,index++,"oimodeler.oimodelLogicalURI",m_oimodelLogicalURI,true);
            addPair(controls,index++,"oimodeler.userName",m_userName,true);
            addPair(controls,index++,"oimodeler.password",m_password,true);
            addPair(controls,index++,null,m_trackUpdates,true);
            add(controls,BorderLayout.NORTH);
            setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        }
        public String getServerTypeName() {
            return "engineeringServerType";
        }
        public String getOIModelLogicalURI() {
            String logicalURI=(String)m_oimodelLogicalURI.getSelectedItem();
            if (logicalURI==null)
                return "";
            else
                return logicalURI.trim();
        }
        public String getOIModelPhysicalURI() {
            return getServerURI()+"?"+getOIModelLogicalURI();
        }
        public Map getConnectionParameters() {
            Map parameters=new HashMap();
            parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.engineeringserver.client.RemoteKAONConnection");
            parameters.put("SERVER_URI",getServerURI());
            if (m_userName.getText().trim().length()!=0)
                parameters.put("PASSWORD",new String(m_password.getPassword()));
            if (m_trackUpdates.isSelected())
                parameters.put("CHANGE_LISTENER","swingthread");
            return parameters;
        }
        protected String getServerURI() {
            String userName=m_userName.getText().trim();
            return "jboss://"+(userName.length()==0 ? "" : userName+"@")+m_hostName.getText().trim()+":"+m_port.getValue().toString();
        }
        protected Configuration getConfiguration() {
            return m_connectionParameters.getSubConfiguration(getServerTypeName());
        }
        public void updateConfiguration() {
            Configuration configuration=getConfiguration();
            if (m_oimodelLogicalURI!=null)
                configuration.setString("oimodelLogicalURI",getOIModelLogicalURI());
            configuration.setString("hostName",m_hostName.getText().trim());
            configuration.setString("port",m_port.getValue().toString());
            configuration.setString("userName",m_userName.getText().trim());
            configuration.setBoolean("trackUpdates",m_trackUpdates.isSelected());
        }
        public String getPhysicalForLogicalURI(String logicalURI) {
            return getServerURI()+"?"+logicalURI;
        }
    }

    /**
     * Server type for direct engineering server.
     */
    protected class DirectEngineeringServerType extends JPanel implements ServerType {
        /** Combo box for logical OI-model URI. */
        protected JComboBox m_oimodelLogicalURI;
        /** Editor for user name. */
        protected JTextField m_userName;
        /** Editor for password. */
        protected JPasswordField m_password;
        /** Combo box for the driver type. */
        protected JComboBox m_driver;
        /** Editor for host name. */
        protected JTextField m_hostName;
        /** Label for port. */
        protected JLabel m_portLabel;
        /** Editor for port. */
        protected JFormattedTextField m_port;
        /** Label for database. */
        protected JLabel m_databaseLabel;
        /** Editor for database. */
        protected JTextField m_database;
        /** Label for connection string. */
        protected JLabel m_connectionStringLabel;
        /** Editor for connection string. */
        protected JTextField m_connectionString;
        /** Label for driver class. */
        protected JLabel m_driverClassLabel;
        /** Editor for driver class. */
        protected JTextField m_driverClass;

        public DirectEngineeringServerType() {
            super(new BorderLayout());
            Configuration configuration=getConfiguration();
            final LocalizationManager localizationManager=m_module.getAppDriver().getLocalizationManager();
            setName(localizationManager.getPhrase("oimodeler."+getServerTypeName()));
            if (m_dialogModus!=COPY_MODUS) {
                LogicalURIComboBoxModel logicalURIComboBoxModel=new LogicalURIComboBoxModel(this);
                m_oimodelLogicalURI=new JComboBox(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setPreferredSize(new Dimension(200,m_oimodelLogicalURI.getPreferredSize().height));
                m_oimodelLogicalURI.setEditable(true);
                m_oimodelLogicalURI.addPopupMenuListener(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setSelectedItem(configuration.getString("oimodelLogicalURI"));
            }
            m_userName=new JTextField();
            m_userName.setText(configuration.getString("userName"));
            m_password=new JPasswordField();
            m_hostName=new JTextField();
            m_hostName.setText(configuration.getString("hostName","localhost"));
            m_driver=new JComboBox();
            registerKnownDrivers();
            String driverID=configuration.getString("driver","--");
            for (int i=0;i<m_driver.getItemCount();i++) {
                DriverSpecification driverSpecification=(DriverSpecification)m_driver.getItemAt(i);
                if (driverID.equals(driverSpecification.getDriverID())) {
                    m_driver.setSelectedIndex(i);
                    break;
                }
            }
            NumberFormat numberFormat=NumberFormat.getInstance();
            numberFormat.setGroupingUsed(false);
            NumberFormatter numberFormatter=new NumberFormatter(numberFormat) {
                public Object stringToValue(String text) throws ParseException {
                    text=text.trim();
                    if (text.length()==0)
                        return null;
                    else
                        return super.stringToValue(text);
                }
            };
            m_port=new JFormattedTextField(numberFormatter);
            m_port.setPreferredSize(new Dimension(100,m_port.getPreferredSize().height));
            int port=configuration.getInt("port",-1);
            if (port!=-1)
                m_port.setValue(new Integer(port));
            m_database=new JTextField();
            m_database.setText(configuration.getString("database"));
            m_connectionString=new JTextField();
            m_connectionString.setText(configuration.getString("connectionString"));
            m_driverClass=new JTextField();
            m_driverClass.setText(configuration.getString("driverClass"));
            JPanel controls=new JPanel(new GridBagLayout());
            int index=0;
            if (m_oimodelLogicalURI!=null)
                addPair(controls,index++,"oimodeler.oimodelLogicalURI",m_oimodelLogicalURI,true);
            addPair(controls,index++,"oimodeler.userName",m_userName,true);
            addPair(controls,index++,"oimodeler.password",m_password,true);
            addPair(controls,index++,"oimodeler.hostName",m_hostName,true);
            addPair(controls,index++,"oimodeler.driver",m_driver,true);
            m_portLabel=addPair(controls,index++,"oimodeler.port",m_port,false);
            m_databaseLabel=addPair(controls,index++,"oimodeler.database",m_database,true);
            m_connectionStringLabel=addPair(controls,index++,"oimodeler.connectionString",m_connectionString,true);
            m_driverClassLabel=addPair(controls,index++,"oimodeler.driverClass",m_driverClass,true);
            add(controls,BorderLayout.NORTH);
            setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
            m_driver.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    ((DriverSpecification)m_driver.getSelectedItem()).updateControls();
                }
            });
            ((DriverSpecification)m_driver.getSelectedItem()).updateControls();
        }
        public String getServerTypeName() {
            return "directEngineeringServerType";
        }
        public String getOIModelLogicalURI() {
            String logicalURI=(String)m_oimodelLogicalURI.getSelectedItem();
            if (logicalURI==null)
                return "";
            else
                return logicalURI.trim();
        }
        public String getOIModelPhysicalURI() {
            return getServerURI()+"?"+getOIModelLogicalURI();
        }
        public Map getConnectionParameters() {
            DriverSpecification driverSpecification=(DriverSpecification)m_driver.getSelectedItem();
            URI serverURI=driverSpecification.getServerURI();
            Map parameters=new HashMap();
            parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.engineeringserver.client.DirectKAONConnection");
            parameters.put("SERVER_URI",serverURI.toString());
            String driverClass=driverSpecification.getDriverClass();
            if (driverClass!=null)
                parameters.put("JDBC_DRIVER",driverClass);
            String connectionString=driverSpecification.getConnectionString();
            if (connectionString!=null)
                parameters.put("DATABASE_CONNECTION_STRING",connectionString);
            if (serverURI.getUserInfo()==null)
                parameters.put("USER_NAME",m_userName.getText().trim());
            parameters.put("PASSWORD",new String(m_password.getPassword()));
            return parameters;
        }
        protected String getServerURI() {
            DriverSpecification driverSpecification=(DriverSpecification)m_driver.getSelectedItem();
            return driverSpecification.getServerURI().toString();
        }
        protected void registerKnownDrivers() {
            m_driver.addItem(new DriverSpecification("oimodeler.driver.MSSQL",true,true,false,false));
            m_driver.addItem(new DriverSpecification("oimodeler.driver.IBMDB2",true,true,false,false));
            m_driver.addItem(new DriverSpecification("oimodeler.driver.Oracle",true,true,false,false));
            m_driver.addItem(new DriverSpecification("oimodeler.driver.PostgreSQL",true,true,false,false));
            m_driver.addItem(new DriverSpecification("oimodeler.driver.other",false,false,true,true) {
                public String getConnectionString() {
                    return m_connectionString.getText().trim();
                }
                public String getDriverClass() {
                    return m_driverClass.getText().trim();
                }
            });
        }
        protected Configuration getConfiguration() {
            return m_connectionParameters.getSubConfiguration(getServerTypeName());
        }
        public void updateConfiguration() {
            Configuration configuration=getConfiguration();
            if (m_oimodelLogicalURI!=null)
                configuration.setString("oimodelLogicalURI",getOIModelLogicalURI());
            configuration.setString("userName",m_userName.getText().trim());
            configuration.setString("hostName",m_hostName.getText().trim());
            configuration.setString("driver",((DriverSpecification)m_driver.getSelectedItem()).getDriverID());
            configuration.setString("port",m_port.getValue()==null ? "" : m_port.getValue().toString());
            configuration.setString("database",m_database.getText().trim());
            configuration.setString("connectionString",m_connectionString.getText().trim());
            configuration.setString("driverClass",m_driverClass.getText().trim());
        }
        public String getPhysicalForLogicalURI(String logicalURI) {
            return getServerURI()+"?"+logicalURI;
        }

        protected class DriverSpecification {
            protected String m_driverID;
            protected String m_text;
            protected boolean m_showPort;
            protected boolean m_showDatabase;
            protected boolean m_showConnectionString;
            protected boolean m_showDriverClass;

            public DriverSpecification(String driverID,boolean showPort,boolean showDatabase,boolean showConnectionString,boolean showDriverClass) {
                m_driverID=driverID;
                m_text=m_module.getAppDriver().getLocalizationManager().getPhrase(driverID);
                m_showPort=showPort;
                m_showDatabase=showDatabase;
                m_showConnectionString=showConnectionString;
                m_showDriverClass=showDriverClass;
            }
            public void updateControls() {
                m_portLabel.setVisible(m_showPort);
                m_port.setVisible(m_showPort);
                m_databaseLabel.setVisible(m_showDatabase);
                m_database.setVisible(m_showDatabase);
                m_connectionStringLabel.setVisible(m_showConnectionString);
                m_connectionString.setVisible(m_showConnectionString);
                m_driverClassLabel.setVisible(m_showDriverClass);
                m_driverClass.setVisible(m_showDriverClass);
            }
            public String toString() {
                return m_text;
            }
            public String getDriverID() {
                return m_driverID;
            }
            public URI getServerURI() {
                try {
                    int port=-1;
                    if (m_showPort) {
                        Number portNumber=(Number)m_port.getValue();
                        if (portNumber!=null)
                            port=portNumber.intValue();
                    }
                    return new URI("direct",m_userName.getText().trim(),m_hostName.getText().trim(),port,getPath(),null,null);
                }
                catch (URISyntaxException cantHappen) {
                    return null;
                }
            }
            public String getConnectionString() {
                return null;
            }
            public String getDriverClass() {
                return null;
            }
            protected String getPath() {
                int lastDot=m_driverID.lastIndexOf('.');
                String path="/"+m_driverID.substring(lastDot+1);
                if (m_showDatabase)
                    path+="/"+m_database.getText().trim();
                return path;
            }
        }
    }

    /**
     * Server type for direct KAON connections.
     */
    protected class OtherServerType extends JPanel implements ServerType {
        /** Editor for logical OI-model URI. */
        protected JComboBox m_oimodelLogicalURI;
        /** Editor for physical OI-model URI. */
        protected JTextField m_oimodelPhysicalURI;
        /** Table for the connection parameters. */
        protected SmartTable m_parameters;
        /** The table model. */
        protected DefaultTableModel m_tableModel;
        /** The URI mapper. */
        protected LogicalToPhysicalURIsMapping m_logicalToPhysicalURIsMapping;

        public OtherServerType() {
            super(new BorderLayout());
            Configuration configuration=getConfiguration();
            final LocalizationManager localizationManager=m_module.getAppDriver().getLocalizationManager();
            setName(localizationManager.getPhrase("oimodeler."+getServerTypeName()));
            if (m_dialogModus!=COPY_MODUS) {
                m_oimodelPhysicalURI=new JTextField();
                m_oimodelPhysicalURI.setPreferredSize(new Dimension(200,m_oimodelPhysicalURI.getPreferredSize().height));
                m_oimodelPhysicalURI.setText(configuration.getString("oimodelPhysicalURI"));
            }
            if (m_dialogModus==CREATE_MODUS) {
                LogicalURIComboBoxModel logicalURIComboBoxModel=new LogicalURIComboBoxModel(this);
                m_oimodelLogicalURI=new JComboBox(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setPreferredSize(new Dimension(200,m_oimodelLogicalURI.getPreferredSize().height));
                m_oimodelLogicalURI.setEditable(true);
                m_oimodelLogicalURI.addPopupMenuListener(logicalURIComboBoxModel);
                m_oimodelLogicalURI.setSelectedItem(configuration.getString("oimodelLogicalURI"));
            }
            if (m_dialogModus==COPY_MODUS)
                m_logicalToPhysicalURIsMapping=new LogicalToPhysicalURIsMapping(m_logicalURIs);
            m_tableModel=new DefaultTableModel(new Object[][] { {"",""} },new Object[] { localizationManager.getPhrase("oimodeler.connectionKey"),localizationManager.getPhrase("oimodeler.connectionValue") });
            m_tableModel.addTableModelListener(new TableModelListener() {
                public void tableChanged(TableModelEvent e) {
                    if (e.getType()==TableModelEvent.UPDATE && e.getLastRow()+1==m_tableModel.getRowCount()) {
                        String value=(String)m_tableModel.getValueAt(e.getLastRow(),e.getColumn());
                        if (value!=null && value.length()!=0)
                            m_tableModel.addRow(new Object[] { "","" });
                    }
                }
            });
            m_parameters=new SmartTable(m_tableModel);
            m_parameters.setDefaultEditor(Object.class,new SmartCellEditor(new JTextField()));
            SmartAction deleteSelectedRows=new AbstractSmartAction("deleteSelectedRows") {
                public void actionPerformed(ActionEvent e) {
                    int[] selectedRows=m_parameters.getSelectedRows();
                    Arrays.sort(selectedRows);
                    for (int i=selectedRows.length-1;i>=0;i--)
                        if (selectedRows[i]+1==m_tableModel.getRowCount()) {
                            m_tableModel.setValueAt("",selectedRows[i],0);
                            m_tableModel.setValueAt("",selectedRows[i],1);
                        }
                        else
                            m_tableModel.removeRow(selectedRows[i]);
                }
            };
            m_parameters.getInputMap(SmartTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE,0),deleteSelectedRows.getActionID());
            m_parameters.getActionMap().put(deleteSelectedRows.getActionID(),deleteSelectedRows);
            Map connectionParameters=configuration.getMap("connectionParameters");
            if (connectionParameters!=null) {
                int index=0;
                Iterator keys=connectionParameters.keySet().iterator();
                while (keys.hasNext()) {
                    String key=(String)keys.next();
                    String value=(String)connectionParameters.get(key);
                    m_tableModel.insertRow(index,new Object[] { key,value });
                }
            }
            JPanel controls=new JPanel(new GridBagLayout());
            int index=0;
            if (m_oimodelPhysicalURI!=null)
                addPair(controls,index++,"oimodeler.oimodelPhysicalURI",m_oimodelPhysicalURI,true);
            if (m_oimodelLogicalURI!=null)
                addPair(controls,index++,"oimodeler.oimodelLogicalURI",m_oimodelLogicalURI,true);
            if (m_logicalToPhysicalURIsMapping!=null) {
                addPair(controls,index++,"oimodeler.logicalToPhysicalURIMappings",null,true);
                JScrollPane mappingsScrollPane=new JScrollPane(m_logicalToPhysicalURIsMapping);
                mappingsScrollPane.setPreferredSize(new Dimension(500,200));
                GridBagConstraints gbc=new GridBagConstraints(0,index++,2,1,0.0,0.0,GridBagConstraints.CENTER,GridBagConstraints.BOTH,new Insets(4,4,4,4),0,0);
                controls.add(mappingsScrollPane,gbc);
            }
            addPair(controls,index++,"oimodeler.connectionParameters",null,true);
            add(controls,BorderLayout.NORTH);
            JScrollPane parametersScrollPane=new JScrollPane(m_parameters);
            parametersScrollPane.setPreferredSize(new Dimension(300,200));
            add(parametersScrollPane,BorderLayout.CENTER);
            setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
        }
        public String getServerTypeName() {
            return "otherServerType";
        }
        public String getOIModelLogicalURI() {
            String logicalURI=(String)m_oimodelLogicalURI.getSelectedItem();
            if (logicalURI==null)
                return "";
            else
                return logicalURI.trim();
        }
        public String getOIModelPhysicalURI() {
            return m_oimodelPhysicalURI.getText().trim();
        }
        public Map getConnectionParameters() {
            Map parameters=new HashMap();
            for (int i=0;i<m_parameters.getRowCount();i++) {
                String key=((String)m_parameters.getValueAt(i,0)).trim();
                String value=((String)m_parameters.getValueAt(i,1)).trim();
                if (key.length()!=0)
                    parameters.put(key,value);
            }
            return parameters;
        }
        protected Configuration getConfiguration() {
            return m_connectionParameters.getSubConfiguration(getServerTypeName());
        }
        public void updateConfiguration() {
            Configuration configuration=getConfiguration();
            if (m_oimodelLogicalURI!=null)
                configuration.setString("oimodelLogicalURI",getOIModelLogicalURI());
            if (m_oimodelPhysicalURI!=null)
                configuration.setString("oimodelPhysicalURI",getOIModelPhysicalURI());
            configuration.setMap("connectionParameters",getConnectionParameters());
        }
        public String getPhysicalForLogicalURI(String logicalURI) {
            return m_logicalToPhysicalURIsMapping.getPhysicalForLogicalURI(logicalURI);
        }
    }

    protected JLabel addPair(JPanel panel,int row,String labelText,JComponent control,boolean fill) {
        JLabel label=null;
        GridBagConstraints gbc=new GridBagConstraints();
        gbc.insets=new Insets(4,4,4,4);
        gbc.gridy=row;
        if (labelText!=null) {
            gbc.anchor=GridBagConstraints.WEST;
            gbc.gridx=0;
            label=m_module.getAppDriver().getLocalizationManager().getLabel(labelText);
            panel.add(label,gbc);
        }
        if (control!=null) {
            gbc.gridx=1;
            gbc.weightx=1.0f;
            if (fill)
                gbc.fill=GridBagConstraints.HORIZONTAL;
            panel.add(control,gbc);
        }
        return label;
    }

    /**
     * The chooser for the logical OI-model URI.
     */
    protected class LogicalURIComboBoxModel extends AbstractListModel implements PopupMenuListener,ComboBoxModel {
        /** The server type. */
        protected ServerType m_serverType;
        /** Selected URI. */
        protected String m_selectedURI;
        /** All URIs in the model. */
        protected String[] m_logicalURIs;

        public LogicalURIComboBoxModel(ServerType serverType) {
            m_serverType=serverType;
            m_logicalURIs=new String[0];
        }
        public void refresh() {
            try {
                KAONConnection kaonConnection=KAONManager.getKAONConnection(m_serverType.getConnectionParameters());
                try {
                    Set logicalURIs=kaonConnection.getAllOIModelLogicalURIs();
                    m_logicalURIs=new String[logicalURIs.size()];
                    Iterator iterator=logicalURIs.iterator();
                    int index=0;
                    while (iterator.hasNext())
                        m_logicalURIs[index++]=(String)iterator.next();
                }
                finally {
                    kaonConnection.close();
                }
                Arrays.sort(m_logicalURIs);
            }
            catch (KAONException e) {
                m_logicalURIs=new String[0];
                m_module.getAppDriver().displayErrorNotification(e);
            }
            fireContentsChanged(this,-1,-1);
        }
        public void setSelectedItem(Object anObject) {
            if (m_selectedURI!=null && !m_selectedURI.equals(anObject) || m_selectedURI==null && anObject!=null) {
                m_selectedURI=(String)anObject;
                fireContentsChanged(this,-1,-1);
            }
        }
        public Object getSelectedItem() {
            return m_selectedURI;
        }
        public int getSize() {
            return m_logicalURIs.length;
        }
        public Object getElementAt(int index) {
            return m_logicalURIs[index];
        }
        public void popupMenuCanceled(PopupMenuEvent e) {
        }
        public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
        }
        public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
            refresh();
        }
    }

    /**
     * The table for mapping from logical to physical URIs.
     */
    protected class LogicalToPhysicalURIsMapping extends SmartTable {
        public LogicalToPhysicalURIsMapping(Collection logicalURIs) {
            LocalizationManager localizationManager=m_module.getAppDriver().getLocalizationManager();
            DefaultTableModel model=new DefaultTableModel(new Object[] { localizationManager.getPhrase("oimodeler.logicalToPhysicalURIMappings.logicalURI"),localizationManager.getPhrase("oimodeler.logicalToPhysicalURIMappings.physicalURI") },logicalURIs.size()) {
                public boolean isCellEditable(int rowIndex,int columnIndex) {
                    return columnIndex==1;
                }
                public Class getColumnClass(int columnIndex) {
                    return String.class;
                }
            };
            int row=0;
            Iterator iterator=logicalURIs.iterator();
            while (iterator.hasNext()) {
                String logicalURI=(String)iterator.next();
                model.setValueAt(logicalURI,row++,0);
            }
            setModel(model);
            addToolTipRenderers();
            setDefaultEditor(String.class,new SmartCellEditor(new JTextField()));
        }
        public String getPhysicalForLogicalURI(String logicalURI) {
            DefaultTableModel model=(DefaultTableModel)getModel();
            for (int i=0;i<model.getRowCount();i++) {
                String rowLogicalURI=(String)model.getValueAt(i,0);
                if (rowLogicalURI.equals(logicalURI)) {
                    String rowPhysicalURI=(String)model.getValueAt(i,1);
                    return rowPhysicalURI==null ? "" : rowPhysicalURI;
                }
            }
            return "";
        }
    }
}
