package de.fzi.wim.oimodeler.selection;

import java.util.Collection;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.EventListener;
import javax.swing.event.EventListenerList;

import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.oimodeler.ui.*;

/**
 * Represents the current selection of entities in the application.
 */
public class OIModelerSelectionModel {
    /** The flag specifying that the selection contains concepts. */
    public static final int CONTAINS_CONCEPTS=0x01;
    /** The flag specifying that the selection contains properties. */
    public static final int CONTAINS_PROPERTIES=0x02;
    /** The flag specifying that the selection contains instances. */
    public static final int CONTAINS_INSTANCES=0x04;

    /** The viewable for the OI-model. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The set of selected entities. */
    protected Set m_selectedEntities;
    /** The event listeners. */
    protected EventListenerList m_eventListenerList;
    /** The type of the selection. */
    protected int m_selectionType;

    /**
     * Creates a selection model.
     *
     * @param oimodelerViewable         the viewable for the OI-model
     */
    public OIModelerSelectionModel(OIModelerViewable oimodelerViewable) {
        m_oimodelerViewable=oimodelerViewable;
        m_eventListenerList=new EventListenerList();
        m_selectedEntities=new HashSet();
    }
    /**
     * Adds an OI-model selection listener to this model.
     *
     * @param listener                  the listener to add
     */
    public void addOIModelSelectionListener(OIModelerSelectionListener listener) {
        m_eventListenerList.add(OIModelerSelectionListener.class,listener);
    }
    /**
     * removes an OI-model selection listener to this model.
     *
     * @param listener                  the listener to remove
     */
    public void removeOIModelSelectionListener(OIModelerSelectionListener listener) {
        m_eventListenerList.remove(OIModelerSelectionListener.class,listener);
    }
    /**
     * Fires the notification that the selected entities have changed.
     */
    protected void fireSelectedEntitiesChanged() {
        m_oimodelerViewable.getSmartActionMap().updateActions();
        EventListener[] listeners=m_eventListenerList.getListeners(OIModelerSelectionListener.class);
        for (int i=0;i<listeners.length;i++)
            ((OIModelerSelectionListener)listeners[i]).selectedEntitiesChanged(this);
    }
    /**
     * Makes specified entites selected.
     *
     * @param entities                  the collection of entities that are selected
     */
    public void setSelectedEntities(Collection entities) {
        if (!m_selectedEntities.equals(entities)) {
            m_selectedEntities.clear();
            Iterator iterator=entities.iterator();
            while (iterator.hasNext()) {
                Object object=iterator.next();
                if (object instanceof Entity)
                    m_selectedEntities.add(object);
            }
            updateSelectionType();
            fireSelectedEntitiesChanged();
        }
    }
    /**
     * Clears the selection.
     */
    public void clearSelectedEntities() {
        m_selectedEntities.clear();
        updateSelectionType();
        fireSelectedEntitiesChanged();
    }
    /**
     * Returns the set of selected entities.
     *
     * @return                          the selected entities
     */
    public Set getSelectedEntities() {
        return m_selectedEntities;
    }
    /**
     * The type of the selection.
     *
     * @return                          the type of the selection
     */
    public int getSelectionType() {
        return m_selectionType;
    }
    /**
     * Updates the selection type.
     */
    protected void updateSelectionType() {
        m_selectionType=0;
        Iterator iterator=m_selectedEntities.iterator();
        while (iterator.hasNext()) {
            Object object=iterator.next();
            if (object instanceof Concept)
                m_selectionType|=CONTAINS_CONCEPTS;
            else if (object instanceof Property)
                m_selectionType|=CONTAINS_PROPERTIES;
            else if (object instanceof Instance)
                m_selectionType|=CONTAINS_INSTANCES;
        }
    }
}
