package de.fzi.wim.oimodeler.search;

import java.awt.Insets;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import javax.swing.AbstractAction;
import javax.swing.KeyStroke;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JTextPane;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JScrollPane;

import edu.unika.aifb.kaon.api.*;

import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.guibase.util.*;
import de.fzi.wim.guibase.menus.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.sidebar.*;
import de.fzi.wim.oimodeler.entitylist.*;
import de.fzi.wim.oimodeler.selection.*;

/**
 * The search pane for the OI-modeler.
 */
public class OIModelSearchPane extends JPanel implements OIModelerSideBar.SideBarElement {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** The search table. */
    protected OIModelEntityTable m_oimodelEntityTable;
    /** Search pane. */
    protected JTextPane m_searchTerm;
    /** Type of search. */
    protected JComboBox m_searchType;
    /** The search button. */
    protected JButton m_searchButton;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable                 the OI-modeler viewable
     */
    public OIModelSearchPane(OIModelerViewable oimodelerViewable) {
        super(new GridBagLayout());
        m_oimodelerViewable=oimodelerViewable;
        createComponents();
    }
    /**
     * Creates the components in this pane.
     */
    protected void createComponents() {
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        m_searchTerm=new JTextPane();
        m_searchTerm.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,KeyEvent.CTRL_DOWN_MASK),"DO_SEARCH");
        m_searchTerm.getActionMap().put("DO_SEARCH",new AbstractAction() {
            public void actionPerformed(ActionEvent e) {
                m_searchButton.doClick();
            }
        });
        m_searchType=new JComboBox();
        registerQueryTypes();
        m_searchButton=new JButton();
        m_searchButton.setIcon(localizationManager.getImageIcon("oimodeler.searchIcon"));
        Mnemonics.setText(m_searchButton,localizationManager.getPhrase("oimodeler.searchButton"));
        m_searchButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                executeQuery(m_searchType.getSelectedIndex(),m_searchTerm.getText());
            }
        });
        m_oimodelEntityTable=new OIModelEntityTable(m_oimodelerViewable,createSearchTableModel());
        new DoubleClickSelectionFocuser(m_oimodelerViewable,m_oimodelEntityTable);
        new JTablePopupMenuManipulator(m_oimodelEntityTable,m_oimodelerViewable.getModule().getAppDriver().createPopupMenu("popup.oimodeler.selection"));
        JLabel label=localizationManager.getLabel("oimodeler.searchTerm");
        label.setHorizontalAlignment(JLabel.LEFT);
        GridBagConstraints gbc=new GridBagConstraints(0,0,1,1,1.0,0.0,GridBagConstraints.CENTER,GridBagConstraints.HORIZONTAL,new Insets(5,5,0,5),0,0);
        gbc.weightx=0.9;
        gbc.gridwidth=1;
        add(m_searchType,gbc);
        gbc.weightx=0.1;
        gbc.gridx++;
        add(m_searchButton,gbc);
        gbc.gridx=0;
        gbc.gridy++;
        gbc.gridwidth=2;
        JScrollPane scrollPane=new JScrollPane(m_searchTerm,JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scrollPane.getViewport().setPreferredSize(new Dimension(1,100));
        gbc.fill=GridBagConstraints.BOTH;
        gbc.weighty=0.3;
        add(scrollPane,gbc);
        gbc.fill=GridBagConstraints.BOTH;
        gbc.weighty=0.7;
        gbc.gridy++;
        gbc.insets.bottom=5;
        add(new JScrollPane(m_oimodelEntityTable),gbc);
    }
    /**
     * Creates the model of the search pane.
     *
     * @return                                  the model for the search pane
     */
    protected OIModelSearchTableModel createSearchTableModel() {
        return new OIModelSearchTableModel(m_oimodelerViewable);
    }
    /**
     * Registes the types of the queries.
     */
    protected void registerQueryTypes() {
        addQueryType("oimodeler.searchType.query");
        addQueryType("oimodeler.searchType.anythingByKeyword");
        addQueryType("oimodeler.searchType.conceptsByKeyword");
        addQueryType("oimodeler.searchType.propertiesByKeyword");
        addQueryType("oimodeler.searchType.instancesByKeyword");
    }
    /**
     * Adds the type of the query to the pane.
     *
     * @param queryName                         the resource ID of the type of the query
     */
    protected void addQueryType(String queryName) {
        m_searchType.addItem(m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase(queryName));
    }
    /**
     * Executes the query.
     *
     * @param searchType                        the type of the search (i.e. the index of the search type in the combo box)
     * @param query                             the query
     */
    public void executeQuery(int searchType,String query) {
        if (searchType==0)
            executeQuery(m_searchTerm.getText());
        else
            searchLexicon(m_searchTerm.getText(),searchType==1 || searchType==2,searchType==1 || searchType==3,searchType==1 || searchType==4);
    }
    /**
     * Disposes of the inspector.
     */
    public void dispose() {
        if (m_oimodelEntityTable!=null) {
            m_oimodelEntityTable.dispose();
            m_oimodelEntityTable=null;
        }
    }
    /**
     * Searches the lexicon and produces a list of entries with given lexical value.
     *
     * @param lexicalValue                      the lexical value
     * @param showConcepts                      if <code>true</code> concepts will be shown
     * @param showProperties                    if <code>true</code> properties will be shown
     * @param showInstances                     if <code>true</code> instances will be shown
     */
    public void searchLexicon(String lexicalValue,boolean showConcepts,boolean showProperties,boolean showInstances) {
        try {
            ((OIModelSearchTableModel)m_oimodelEntityTable.getModel()).searchLexicon(lexicalValue,showConcepts,showProperties,showInstances);
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Executes the specified query on the OI-model.
     *
     * @param queryString                       the query to be executed
     */
    public void executeQuery(String queryString) {
        try {
            ((OIModelSearchTableModel)m_oimodelEntityTable.getModel()).executeQuery(queryString);
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Sets the current language URI.
     *
     * @param languageURI                       the language URI
     * @throws KAONException                    thrown if there is a problem
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        OIModelerViewable.setFont(languageURI,m_searchTerm);
        m_oimodelEntityTable.setLanguageURI(languageURI);
    }
    /**
     * Returns the label of the element.
     *
     * @return                                  the name of the element
     */
    public String getLabel() {
        return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.sidebar.item.search");
    }
    /**
     * Returns the description of the element.
     *
     * @return                                  the description of the element
     */
    public String getDescription() {
        return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.sidebar.item.search.description");
    }
    /**
     * Returns the icon of the element.
     *
     * @return                                  the icon of the element
     */
    public Icon getIcon() {
        return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.searchIcon");
    }
    /**
     * Returns the component.
     *
     * @return                                  the component
     */
    public JComponent getComponent() {
        return this;
    }
    /**
     * Called when the element is evicted from the sidebar.
     */
    public void removedFromSideBar() {
    }
    /**
     * Returns the table containing entities.
     *
     * @return                                  the table containing entities
     */
    public OIModelEntityTable getOIModelEntityTable() {
        return m_oimodelEntityTable;
    }
}
