package de.fzi.wim.oimodeler.propertyconcepts;

import java.util.Set;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
import java.util.Comparator;
import javax.swing.Icon;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

import de.fzi.wim.guibase.localization.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * The table model for displaying the concept list.
 */
public class PropertyConceptsTableModel extends AbstractTableModel implements OIModelListener {
    /** The OI-modeler viewable. */
    protected OIModelerViewable m_oimodelerViewable;
    /** Set to <code>true</code> if domain concepts are shown. */
    protected boolean m_showDomainConcepts;
    /** The property for which the list is shown. */
    protected Property m_property;
    /** The list of all elements in the table. */
    protected List m_elements;
    /** The URI of the language. */
    protected String m_languageURI;
    /** Change visitor. */
    protected ChangeVisitor m_changeVisitor;
    /** The icon for inherited concept. */
    protected Icon m_inheritedConcept;
    /** The icon for own concept. */
    protected Icon m_ownConcept;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelerViewable             the viewable
     * @param showDomainConcepts            <code>true</code> if this model will show domain concepts
     */
    public PropertyConceptsTableModel(OIModelerViewable oimodelerViewable,boolean showDomainConcepts) {
        m_languageURI=oimodelerViewable.getLanguageURI();
        m_oimodelerViewable=oimodelerViewable;
        m_showDomainConcepts=showDomainConcepts;
        m_elements=Collections.EMPTY_LIST;
        m_changeVisitor=new OIModelChangeVisitor();
        m_oimodelerViewable.getOIModel().addOIModelListener(this);
        m_inheritedConcept=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.inherited_concept");
        m_ownConcept=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getImageIcon("oimodeler.concept");
    }
    /**
     * Disposes of this model.
     */
    public void dispose() {
        if (m_oimodelerViewable!=null) {
            m_oimodelerViewable.getOIModel().removeOIModelListener(this);
            m_oimodelerViewable=null;
            m_property=null;
            m_elements.clear();
        }
    }
    /**
     * Returns <code>true</code> if this model shows domain concepts.
     *
     * @return                              <code>true</code> if this model shows domain concepts
     */
    public boolean getShowDomainConcepts() {
        return m_showDomainConcepts;
    }
    /**
     * Sets the language URI.
     *
     * @param languageURI                   the language URI
     * @throws KAONException                thrown if there is an error
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        m_languageURI=languageURI;
        Iterator iterator=m_elements.iterator();
        while (iterator.hasNext()) {
            TableRow tableRow=(TableRow)iterator.next();
            tableRow.update();
        }
        Collections.sort(m_elements,RowComparator.INSTANCE);
        fireTableDataChanged();
    }
    /**
     * Returns the property.
     *
     * @return                              the property for which the list is shown
     */
    public Property getProperty() {
        return m_property;
    }
    /**
     * Sets the property.
     *
     * @param property                      the property for which the list is shown
     */
    public void setProperty(final Property property) {
        try {
            m_property=property;
            if (m_property==null)
                m_elements=Collections.EMPTY_LIST;
            else  if ((m_property.getOIModel().getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)!=0) {
                m_elements=null;
                m_oimodelerViewable.getModule().getAppDriver().getThreadPool().executeTask(new Runnable() {
                    public void run() {
                        try {
                            final List elements=loadModel(property);
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    // if we are still supposed to load this element
                                    if (m_property==property) {
                                        m_elements=elements;
                                        fireTableDataChanged();
                                    }
                                }
                            });
                        }
                        catch (final KAONException e) {
                            SwingUtilities.invokeLater(new Runnable() {
                                public void run() {
                                    // if we are still supposed to load this element
                                    if (m_property==property) {
                                        m_elements=Collections.EMPTY_LIST;
                                        fireTableDataChanged();
                                    }
                                    m_oimodelerViewable.errorLoadingEntity(m_property,e);
                                }
                            });
                        }
                    }
                });
            }
            else
                m_elements=loadModel(property);
        }
        catch (KAONException e) {
            m_elements=Collections.EMPTY_LIST;
            m_oimodelerViewable.errorLoadingEntity(m_property,e);
        }
        catch (InterruptedException e) {
            m_elements=Collections.EMPTY_LIST;
            m_oimodelerViewable.errorLoadingEntity(m_property,e);
        }
        fireTableDataChanged();
    }
    /**
     * Loads the model.
     *
     * @param property                      the property
     * @return                              the new array of model elements
     * @throws KAONException                thrown if there is an error
     */
    protected List loadModel(Property property) throws KAONException {
        ViewFilter viewFilter=m_oimodelerViewable.getViewFilter();
        List elements=new ArrayList();
        synchronized (property.getOIModel().getKAONConnection()) {
            Set allConcepts=m_showDomainConcepts ? property.getAllDomainConcepts() : property.getAllRangeConcepts();
            Set ownConcepts=m_showDomainConcepts ? property.getDomainConcepts() : property.getRangeConcepts();
            property.getOIModel().loadObjects(allConcepts,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_LEXICON);
            Iterator iterator=allConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                if (viewFilter.showEntity(concept)) {
                    TableRow tableRow=new TableRow(concept,!ownConcepts.contains(concept));
                    elements.add(tableRow);
                    tableRow.update();
                }
            }
        }
        Collections.sort(elements,RowComparator.INSTANCE);
        return elements;
    }
    /**
     * Shows nothing.
     */
    public void showNothing() {
        setProperty(null);
    }
    /**
     * Returns the class of the column.
     *
     * @param columnIndex                   the index of the column
     * @return                              the column class
     */
    public Class getColumnClass(int columnIndex) {
        if (columnIndex==0)
            return Icon.class;
        else if (columnIndex==1)
            return String.class;
        else
            return Integer.class;
    }
    /**
     * Returns the number of columns.
     *
     * @return                              the number of columns
     */
    public int getColumnCount() {
        return m_showDomainConcepts ? 4 : 2;
    }
    /**
     * Returns the name of the column.
     *
     * @param columnIndex                   the name of the column
     * @return                              the column name
     */
    public String getColumnName(int columnIndex) {
        LocalizationManager localizationManager=m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager();
        switch (columnIndex) {
        case 0:
            return " ";
        case 1:
            return localizationManager.getPhrase("oimodeler.conceptName");
        case 2:
            return localizationManager.getPhrase("oimodeler.minimumCardinality");
        case 3:
        default:
            return localizationManager.getPhrase("oimodeler.maximumCardinality");
        }
    }
    /**
     * Returns the number of rows.
     *
     * @return                              the number of rows
     */
    public int getRowCount() {
        if (m_elements==null)
            return 1;
        else
            return m_elements.size();
    }
    /**
     * Returns the value at given position.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              the value at given row and column
     */
    public Object getValueAt(int rowIndex,int columnIndex) {
        if (m_elements==null) {
            if (columnIndex==1)
                return m_oimodelerViewable.getModule().getAppDriver().getLocalizationManager().getPhrase("oimodeler.loadingWait");
            else
                return "";
        }
        else {
            TableRow tableRow=(TableRow)m_elements.get(rowIndex);
            switch (columnIndex) {
            case 0:
                return tableRow.m_isInherited ? m_inheritedConcept : m_ownConcept;
            case 1:
                return tableRow.m_label;
            case 2:
                return tableRow.m_minimumCardinality;
            case 3:
            default:
                if (tableRow.m_maximumCardinality.intValue()==Integer.MAX_VALUE)
                    return null;
                else
                    return tableRow.m_maximumCardinality;
            }
        }
    }
    /**
     * Checks if the cell is editable.
     *
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     * @return                              <code>true</code> if the cell at given row and column is editable
     */
    public boolean isCellEditable(int rowIndex,int columnIndex) {
        return m_elements!=null && columnIndex>1;
    }
    /**
     * Sets the value at given position.
     *
     * @param aValue                        the value
     * @param rowIndex                      the row
     * @param columnIndex                   the column
     */
    public void setValueAt(Object aValue,int rowIndex,int columnIndex) {
        try {
            TableRow tableRow=(TableRow)m_elements.get(rowIndex);
            if (columnIndex==2) {
                int value;
                if (aValue instanceof Number)
                    value=((Number)aValue).intValue();
                else
                    value=0;
                m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetMinimumCardinality(m_property.getSourceOIModel(),null,m_property,tableRow.m_concept,value)));
            }
            else if (columnIndex==3) {
                int value;
                if (aValue instanceof Number)
                    value=((Number)aValue).intValue();
                else
                    value=Integer.MAX_VALUE;
                m_oimodelerViewable.changeOIModel(Collections.singletonList(new SetMaximumCardinality(m_property.getSourceOIModel(),null,m_property,tableRow.m_concept,value)));
            }
        }
        catch (KAONException e) {
            m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
        }
    }
    /**
     * Called when a bulk of change events is processed in the pool.
     *
     * @param oimodel                   OI-model that was changed
     * @param changeEvents              list of change events that occured
     */
    public void modelChanged(OIModel oimodel,List changeEvents) {
        if (m_property!=null && m_elements!=null)
            try {
                Iterator iterator=changeEvents.iterator();
                while (iterator.hasNext()) {
                    ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                    changeEvent.accept(m_changeVisitor);
                }
                Set instancesWithChangedLexicalEntries=LexiconUtil.getInstancesWithChangedLexicalEntries(changeEvents);
                for (int i=m_elements.size()-1;i>=0;i--) {
                    TableRow tableRow=(TableRow)m_elements.get(i);
                    if (instancesWithChangedLexicalEntries.contains(tableRow.m_concept.getSpanningInstance())) {
                        tableRow.update();
                        fireTableRowsUpdated(i,i);
                    }
                }
            }
            catch (KAONException e) {
                m_oimodelerViewable.getModule().getAppDriver().displayErrorNotification(e);
            }
    }
    /**
     * Called when model is refreshed.
     *
     * @param oimodel                   OI-model that was refreshed
     */
    public void modelRefreshed(OIModel oimodel) {
        if (m_property!=null)
            setProperty(m_property);
    }
    /**
     * Called when model is being deleted.
     *
     * @param oimodel                   OI-model that is being deleted
     */
    public void modelDeleted(OIModel oimodel) {
    }
    /**
     * Locates the table row index for given concept.
     *
     * @param concept                   the concept
     * @return                          the table row index for given concept (-1 if row doesn't exist)
     */
    public int getTableRowIndex(Concept concept) {
        for (int i=0;i<m_elements.size();i++) {
            TableRow tableRow=(TableRow)m_elements.get(i);
            if (tableRow.m_concept.equals(concept))
                return i;
        }
        return -1;
    }
    /**
     * Inserts a row into the table.
     *
     * @param tableRow                  the row object
     */
    protected void insertTableRow(TableRow tableRow) {
        for (int i=0;i<m_elements.size();i++) {
            TableRow current=(TableRow)m_elements.get(i);
            int result=RowComparator.INSTANCE.compare(current,tableRow);
            if (result>=0) {
                m_elements.add(i,tableRow);
                fireTableRowsInserted(i,i);
                return;
            }
        }
        int index=m_elements.size();
        m_elements.add(tableRow);
        fireTableRowsInserted(index,index);
    }
    /**
     * Returns the concept in given row.
     *
     * @param rowIndex                  the index of the row
     * @return                          the concept in given row
     */
    public Concept getConcept(int rowIndex) {
        return ((TableRow)m_elements.get(rowIndex)).m_concept;
    }

    /**
     * The row in the table.
     */
    protected class TableRow {
        protected Concept m_concept;
        protected boolean m_isInherited;
        protected String m_label;
        protected Integer m_minimumCardinality;
        protected Integer m_maximumCardinality;

        public TableRow(Concept concept,boolean isInherited) {
            m_concept=concept;
            m_isInherited=isInherited;
        }
        public void update() throws KAONException {
            m_label=m_concept.getLabel(m_languageURI);
            if (m_label==null)
                m_label=m_concept.getURI();
            if (m_showDomainConcepts) {
                m_minimumCardinality=new Integer(m_property.getMinimumCardinality(m_concept));
                m_maximumCardinality=new Integer(m_property.getMaximumCardinality(m_concept));
            }
        }
    }

    /**
     * The comparator for the rows.
     */
    protected static class RowComparator implements Comparator {
        public static final Comparator INSTANCE=new RowComparator();
        public int compare(Object o1,Object o2) {
            TableRow r1=(TableRow)o1;
            TableRow r2=(TableRow)o2;
            if (r1.m_isInherited && !r2.m_isInherited)
                return -1;
            if (!r1.m_isInherited && r2.m_isInherited)
                return 1;
            return r1.m_label.compareToIgnoreCase(r2.m_label);
        }
    }

    /**
     * The visitor for OI-model events.
     */
    protected class OIModelChangeVisitor extends NullChangeEventVisitor {
        public void visit(AddPropertyDomain event) throws KAONException {
            if (m_showDomainConcepts && m_property.equals(event.getProperty())) {
                TableRow tableRow=new TableRow(event.getConcept(),false);
                tableRow.update();
                insertTableRow(tableRow);
            }
        }
        public void visit(RemovePropertyDomain event) {
            if (m_showDomainConcepts && m_property.equals(event.getProperty())) {
                int index=getTableRowIndex(event.getConcept());
                if (index!=-1) {
                    m_elements.remove(index);
                    fireTableRowsDeleted(index,index);
                }
            }
        }
        public void visit(AddPropertyRange event) throws KAONException {
            if (!m_showDomainConcepts && m_property.equals(event.getProperty())) {
                TableRow tableRow=new TableRow(event.getConcept(),false);
                tableRow.update();
                insertTableRow(tableRow);
            }
        }
        public void visit(RemovePropertyRange event) {
            if (!m_showDomainConcepts && m_property.equals(event.getProperty())) {
                int index=getTableRowIndex(event.getConcept());
                if (index!=-1) {
                    m_elements.remove(index);
                    fireTableRowsDeleted(index,index);
                }
            }
        }
        public void visit(SetMinimumCardinality event) throws KAONException {
            if (m_showDomainConcepts && m_property.equals(event.getProperty())) {
                int tableRowIndex=getTableRowIndex(event.getConcept());
                if (tableRowIndex!=-1) {
                    TableRow tableRow=(TableRow)m_elements.get(tableRowIndex);
                    tableRow.update();
                    fireTableRowsUpdated(tableRowIndex,tableRowIndex);
                }
            }
        }
        public void visit(SetMaximumCardinality event) throws KAONException {
            if (m_showDomainConcepts && m_property.equals(event.getProperty())) {
                int tableRowIndex=getTableRowIndex(event.getConcept());
                if (tableRowIndex!=-1) {
                    TableRow tableRow=(TableRow)m_elements.get(tableRowIndex);
                    tableRow.update();
                    fireTableRowsUpdated(tableRowIndex,tableRowIndex);
                }
            }
        }
    }
}
