package de.fzi.wim.oimodeler.oimodelgraph.view;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Font;
import java.awt.FontMetrics;
import javax.swing.UIManager;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;

/**
 * The painter for nodes that are shown as strings only.
 */
public class StringNodePainter implements NodePainter {
    /** An instance. */
    public static final NodePainter INSTANCE=new StringNodePainter();
    /** The maximum width of text in the node. */
    protected static final int MAXIMUM_TEXT_WIDTH=100;

    /** The point of the current node. */
    protected Point m_nodePoint;
    /** The outer rectangle of the node. */
    protected Rectangle m_nodeOuter;
    /** The position of the label text. */
    public Point m_labelPosition;
    /** The adjusted label of the node. */
    public String m_adjustedNodeLabel;
    /** Set to <code>true</code> if the label has been adjusted. */
    public boolean m_labelAdjusted;

    /**
     * Creates an instance of this class.
     */
    public StringNodePainter() {
        m_nodeOuter=new Rectangle();
        m_labelPosition=new Point();
    }
    /**
     * Paints the supplied node.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics
     * @param node                  the node to paint
     */
    public void paintNode(JGraphPane graphPane,Graphics2D g,Node node) {
        loadNodeGeometry(graphPane,node);
        Color oldColor=g.getColor();
        g.setColor(graphPane.getBackground());
        g.fillRect(m_nodeOuter.x,m_nodeOuter.y,m_nodeOuter.width-1,m_nodeOuter.height-1);
        if (m_adjustedNodeLabel!=null) {
            Font oldFont=g.getFont();
            Color color;
            if (graphPane.isEnabled())
                color=UIManager.getColor("Label.foreground");
            else
                color=UIManager.getColor("Label.disabledForeground");
            g.setColor(color);
            g.drawString(m_adjustedNodeLabel,m_labelPosition.x,m_labelPosition.y);
            g.setFont(oldFont);
        }
        g.setColor(oldColor);
    }
    /**
     * Checks whether given point is inside the node.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param point                 the point
     * @return                      <code>true</code> if the point is in the node
     */
    public boolean isInNode(JGraphPane graphPane,Node node,Point point) {
        loadNodeGeometry(graphPane,node);
        if (m_nodeOuter.contains(point))
            return true;
        return false;
    }
    /**
     * Loads the geometry of given node.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     */
    public void loadNodeGeometry(JGraphPane graphPane,Node node) {
        m_nodePoint=graphPane.getScreenPointForNode(node);
        int labelWidth=1;
        int labelHeight=1;
        m_adjustedNodeLabel=node.getLabel();
        if (m_adjustedNodeLabel!=null) {
            FontMetrics fontMetrics=graphPane.getFontMetrics(graphPane.getFont());
            int stringWidth=fontMetrics.stringWidth(m_adjustedNodeLabel);
            if (stringWidth>=MAXIMUM_TEXT_WIDTH) {
                m_adjustedNodeLabel=EntityNodePainter.adjustTextWidth(m_adjustedNodeLabel,fontMetrics);
                m_labelAdjusted=true;
                stringWidth=fontMetrics.stringWidth(m_adjustedNodeLabel);
            }
            m_labelPosition.x=m_nodePoint.x-stringWidth/2;
            m_labelPosition.y=m_nodePoint.y+(fontMetrics.getAscent()-fontMetrics.getDescent())/2;
            labelWidth+=stringWidth;
            labelHeight+=fontMetrics.getAscent()+fontMetrics.getDescent();
        }
        else {
            labelWidth+=40;
            labelHeight+=20;
            m_labelPosition.x=m_nodePoint.x;
            m_labelPosition.y=m_nodePoint.y;
        }
        m_nodeOuter.setBounds(m_nodePoint.x-labelWidth/2,m_nodePoint.y-labelHeight/2,labelWidth,labelHeight);
    }
    /**
     * Returns the outer rectangle of the node on screen.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param nodeScreenRectangle   the rectangle receiving the node's coordinates
     */
    public void getNodeScreenBounds(JGraphPane graphPane,Node node,Rectangle nodeScreenRectangle) {
        loadNodeGeometry(graphPane,node);
        nodeScreenRectangle.setBounds(m_nodeOuter);
    }
    /**
     * Retruns the tool-tip for given point.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param point                 the point
     * @return                      the tool-tip at given point (or <code>null</code>)
     */
    public String getToolTipText(JGraphPane graphPane,Node node,Point point) {
        if (m_labelAdjusted && m_nodeOuter.contains(point))
            return node.getLabel();
        return null;
    }
}
