package de.fzi.wim.oimodeler.oimodelgraph.view;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.FontMetrics;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.controller.*;
import de.fzi.wim.guibase.graphview.view.*;

import de.fzi.wim.oimodeler.oimodelgraph.graph.*;

/**
 * The painter for OI-model nodes.
 */
public abstract class EntityNodePainter implements NodePainter {
    /** The color for the handle background while it is loading. */
    protected static final Color HANDLE_LOADING_BACKGROUND=new Color(240,240,240);
    /** The color for the handle border while it is loading. */
    protected static final Color HANDLE_LOADING_BORDER=new Color(200,200,200);
    /** The width of the handle in pixels. */
    protected static final int HANDLE_WIDTH=10;
    /** The numper of pixels that the handle lowers into the label box. */
    protected static final int HANDLE_DESCENT=5;
    /** The height of the text label border. */
    protected static final int TEXT_LABEL_BORDER_HEIGHT=3;
    /** The width of the text label border. */
    protected static final int TEXT_LABEL_BORDER_WIDTH=5;
    /** The maximum width of text in the node. */
    protected static final int MAXIMUM_TEXT_WIDTH=100;

    /** The point of the current node. */
    protected Point m_nodePoint;
    /** The outer rectangle of the node. */
    protected Rectangle m_nodeOuter;
    /** The label rectangle of the node. */
    public Rectangle m_nodeLabel;
    /** The position of the label text. */
    public Point m_labelPosition;
    /** The position of handles. */
    public Rectangle[] m_handles;
    /** The adjusted label of the node. */
    public String m_adjustedNodeLabel;
    /** Set to <code>true</code> if the label has been adjusted. */
    public boolean m_labelAdjusted;

    /**
     * Creates an instance of this class.
     */
    public EntityNodePainter() {
        m_nodeOuter=new Rectangle();
        m_nodeLabel=new Rectangle();
        m_labelPosition=new Point();
        int numberOfHandles=getUpperBorderNumberOfHandles()+getLowerBorderNumberOfHandles();
        m_handles=new Rectangle[numberOfHandles];
        for (int i=0;i<numberOfHandles;i++)
            m_handles[i]=new Rectangle();
    }
    /**
     * Paints the supplied node.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics
     * @param node                  the node to paint
     */
    public void paintNode(JGraphPane graphPane,Graphics2D g,Node node) {
        BasicOIModelGraphPane basicOIModelGraphPane=(BasicOIModelGraphPane)graphPane;
        EntityNode entityNode=(EntityNode)node;
        HighlightingManipulator highlightingManipulator=(HighlightingManipulator)graphPane.getManipulator(HighlightingManipulator.NAME);
        boolean isHighlighted=highlightingManipulator!=null && highlightingManipulator.getHighlightedNode()==entityNode;
        boolean isSelected;
        IncrementalSearchManipulator incrementalSearchManipulator=(IncrementalSearchManipulator)graphPane.getManipulator(IncrementalSearchManipulator.NAME);
        if (incrementalSearchManipulator!=null && incrementalSearchManipulator.getSearchActive())
            isSelected=incrementalSearchManipulator.getHighlightedNodes().contains(entityNode);
        else {
            SelectionManipulator selectionManipulator=(SelectionManipulator)graphPane.getManipulator(SelectionManipulator.NAME);
            isSelected=selectionManipulator!=null && selectionManipulator.getNodeSelectionModel().isNodeSelected(entityNode);
        }
        boolean isSelectionPrimary=basicOIModelGraphPane.isSelectionPrimary();
        boolean isShowBlackAndWhite=basicOIModelGraphPane.getShowBlackAndWhite();
        boolean isShowForCapture=basicOIModelGraphPane.getShowForScreenCapture();
        DraggingManipulator draggingManipulator=(DraggingManipulator)graphPane.getManipulator(DraggingManipulator.NAME);
        boolean isDragging=draggingManipulator!=null && draggingManipulator.getDraggedNode()==entityNode;
        boolean isInActiveOIModel=entityNode.isDeclaredIn(basicOIModelGraphPane.getOIModelGraphAnchor().getActiveOIModel());
        loadNodeGeometry(graphPane,node);
        paintNode(graphPane,g,node,isHighlighted,isSelected,isDragging,isSelectionPrimary,isShowBlackAndWhite,isShowForCapture,isInActiveOIModel,graphPane.isEnabled());
    }
    /**
     * Draws the handle with the down arrow.
     *
     * @param g                     the graphics for drawing the handle
     * @param rectangle             the rectangle of the handle
     * @param state                 the state
     */
    protected void paintDownArrowHandle(Graphics2D g,Rectangle rectangle,int state) {
        if (state!=EntityNode.LOADED) {
            paintHandleBox(g,rectangle,state);
            g.setColor(Color.black);
            int top=rectangle.y+3;
            int bottom=rectangle.y+rectangle.height-3;
            int horizontal1=rectangle.x+3;
            int horizontal2=rectangle.x+rectangle.width/2;
            int horizontal3=rectangle.x+rectangle.width-3;
            g.fillPolygon(new int[] { horizontal1,horizontal2,horizontal3 },new int[] { top,bottom,top },3);
        }
    }
    /**
     * Draws the handle with the up arrow.
     *
     * @param g                     the graphics for drawing the handle
     * @param rectangle             the rectangle of the handle
     * @param state                 the state
     */
    protected void paintUpArrowHandle(Graphics2D g,Rectangle rectangle,int state) {
        if (state!=EntityNode.LOADED) {
            paintHandleBox(g,rectangle,state);
            g.setColor(Color.black);
            int top=rectangle.y+3;
            int bottom=rectangle.y+rectangle.height-3;
            int horizontal1=rectangle.x+3;
            int horizontal2=rectangle.x+rectangle.width/2;
            int horizontal3=rectangle.x+rectangle.width-3;
            g.fillPolygon(new int[] { horizontal1,horizontal2,horizontal3 },new int[] { bottom,top,bottom },3);
        }
    }
    /**
     * Draws the handle with the right arrow.
     *
     * @param g                     the graphics for drawing the handle
     * @param rectangle             the rectangle of the handle
     * @param state                 the state
     */
    protected void paintRightArrowHandle(Graphics2D g,Rectangle rectangle,int state) {
        if (state!=EntityNode.LOADED) {
            paintHandleBox(g,rectangle,state);
            g.setColor(Color.black);
            int left=rectangle.x+3;
            int right=rectangle.x+rectangle.width-3;
            int vertical1=rectangle.y+3;
            int vertical2=rectangle.y+rectangle.height/2;
            int vertical3=rectangle.y+rectangle.height-3;
            g.fillPolygon(new int[] { left,right,left },new int[] { vertical1,vertical2,vertical3 },3);
        }
    }
    /**
     * Draws the handle with the left arrow.
     *
     * @param g                     the graphics for drawing the handle
     * @param rectangle             the rectangle of the handle
     * @param state                 the state
     */
    protected void paintLeftArrowHandle(Graphics2D g,Rectangle rectangle,int state) {
        if (state!=EntityNode.LOADED) {
            paintHandleBox(g,rectangle,state);
            g.setColor(Color.black);
            int left=rectangle.x+3;
            int right=rectangle.x+rectangle.width-3;
            int vertical1=rectangle.y+3;
            int vertical2=rectangle.y+rectangle.height/2;
            int vertical3=rectangle.y+rectangle.height-3;
            g.fillPolygon(new int[] { right,left,right },new int[] { vertical1,vertical2,vertical3 },3);
        }
    }
    /**
     * Draws the handle with the dotted line.
     *
     * @param g                     the graphics for drawing the handle
     * @param rectangle             the rectangle of the handle
     * @param state                 the state
     */
    protected void paintDottedLineHandle(Graphics2D g,Rectangle rectangle,int state) {
        if (state!=EntityNode.LOADED) {
            paintHandleBox(g,rectangle,state);
            g.setColor(Color.black);
            int middle=rectangle.x+rectangle.width/2;
            int vertical0=rectangle.y+3;
            int vertical1=rectangle.y+rectangle.height/2-3;
            int vertical2=rectangle.y+rectangle.height/2+3;
            int vertical3=rectangle.y+rectangle.height-3;
            g.drawLine(middle,vertical0,middle,vertical1);
            g.drawLine(middle,vertical2,middle,vertical3);
        }
    }
    /**
     * Draws the handle box.
     *
     * @param g                     the graphics for drawing the handle box
     * @param handleBox             the box for the handle
     * @param state                 the state
     */
    protected void paintHandleBox(Graphics2D g,Rectangle handleBox,int state) {
        g.setColor(state==EntityNode.LOADING ? HANDLE_LOADING_BACKGROUND : Color.lightGray);
        g.fillRect(handleBox.x,handleBox.y,handleBox.width-1,handleBox.height-1);
        g.setColor(state==EntityNode.LOADING ? HANDLE_LOADING_BORDER : Color.black);
        g.drawRect(handleBox.x,handleBox.y,handleBox.width-1,handleBox.height-1);
    }
    /**
     * Checks whether given point is inside the node.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param point                 the point
     * @return                      <code>true</code> if the point is in the node
     */
    public boolean isInNode(JGraphPane graphPane,Node node,Point point) {
        loadNodeGeometry(graphPane,node);
        if (m_nodeLabel.contains(point))
            return true;
        for (int i=0;i<m_handles.length;i++)
            if (m_handles[i].contains(point))
                return true;
        return false;
    }
    /**
     * Loads the geometry of given node.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     */
    public void loadNodeGeometry(JGraphPane graphPane,Node node) {
        m_nodePoint=graphPane.getScreenPointForNode(node);
        m_nodeLabel.width=1;
        m_nodeLabel.height=1;
        m_adjustedNodeLabel=node.getLabel();
        m_labelAdjusted=false;
        if (m_adjustedNodeLabel!=null) {
            FontMetrics fontMetrics=graphPane.getFontMetrics(graphPane.getFont());
            int stringWidth=fontMetrics.stringWidth(m_adjustedNodeLabel);
            if (stringWidth>=MAXIMUM_TEXT_WIDTH) {
                m_adjustedNodeLabel=adjustTextWidth(m_adjustedNodeLabel,fontMetrics);
                m_labelAdjusted=true;
                stringWidth=fontMetrics.stringWidth(m_adjustedNodeLabel);
            }
            m_labelPosition.x=m_nodePoint.x-stringWidth/2;
            m_labelPosition.y=m_nodePoint.y+(fontMetrics.getAscent()-fontMetrics.getDescent())/2;
            m_nodeLabel.width+=stringWidth+2*TEXT_LABEL_BORDER_WIDTH;
            m_nodeLabel.height+=fontMetrics.getAscent()+fontMetrics.getDescent()+2*TEXT_LABEL_BORDER_HEIGHT;
        }
        else {
            m_nodeLabel.width+=40;
            m_nodeLabel.height+=20;
            m_labelPosition.x=m_nodePoint.x;
            m_labelPosition.y=m_nodePoint.y;
        }
        int upperNumberOfHandles=getUpperBorderNumberOfHandles();
        int lowerNumberOfHandles=getLowerBorderNumberOfHandles();
        int minimumWidth=Math.max(upperNumberOfHandles*(HANDLE_WIDTH+3)+3,lowerNumberOfHandles*(HANDLE_WIDTH+3)+3);
        if (m_nodeLabel.width<minimumWidth)
            m_nodeLabel.width=minimumWidth;
        m_nodeLabel.x=m_nodePoint.x-m_nodeLabel.width/2;
        m_nodeLabel.y=m_nodePoint.y-m_nodeLabel.height/2;
        m_nodeOuter.setBounds(m_nodeLabel.x,m_nodeLabel.y-HANDLE_WIDTH+HANDLE_DESCENT,m_nodeLabel.width,m_nodeLabel.height+2*(HANDLE_WIDTH-HANDLE_DESCENT));
        if (upperNumberOfHandles>0) {
            m_handles[0].setBounds(m_nodeOuter.x+3,m_nodeOuter.y,HANDLE_WIDTH,HANDLE_WIDTH);
            Rectangle previous=m_handles[0];
            for (int i=1;i<upperNumberOfHandles;i++) {
                m_handles[i].setBounds(previous.x+previous.width+3,m_nodeOuter.y,HANDLE_WIDTH,HANDLE_WIDTH);
                previous=m_handles[i];
            }
        }
        if (lowerNumberOfHandles>0) {
            m_handles[upperNumberOfHandles].setBounds(m_nodeOuter.x+3,m_nodeOuter.y+m_nodeOuter.height-HANDLE_WIDTH,HANDLE_WIDTH,HANDLE_WIDTH);
            Rectangle previous=m_handles[upperNumberOfHandles];
            for (int i=1;i<lowerNumberOfHandles;i++) {
                m_handles[i+upperNumberOfHandles].setBounds(previous.x+previous.width+3,m_nodeOuter.y+m_nodeOuter.height-HANDLE_WIDTH,HANDLE_WIDTH,HANDLE_WIDTH);
                previous=m_handles[i];
            }
        }
    }
    /**
     * Returns the outer rectangle of the node on screen.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param nodeScreenRectangle   the rectangle receiving the node's coordinates
     */
    public void getNodeScreenBounds(JGraphPane graphPane,Node node,Rectangle nodeScreenRectangle) {
        loadNodeGeometry(graphPane,node);
        nodeScreenRectangle.setBounds(m_nodeOuter);
    }
    /**
     * Retruns the tool-tip for given point.
     *
     * @param graphPane             the graph pane
     * @param node                  the node
     * @param point                 the point
     * @return                      the tool-tip at given point (or <code>null</code>)
     */
    public String getToolTipText(JGraphPane graphPane,Node node,Point point) {
        loadNodeGeometry(graphPane,node);
        for (int i=m_handles.length-1;i>=0;--i)
            if (m_handles[i].contains(point)) {
                String phrase=getHandleToolTipPrefix()+i;
                return ((BasicOIModelGraphPane)graphPane).getOIModelGraphAnchor().getLocalizationManager().getPhrase(phrase);
            }
        if (m_labelAdjusted && m_nodeLabel.contains(point))
            return node.getLabel();
        return null;
    }
    /**
     * Returns the text adjusted to given width.
     *
     * @param text                  the text being adjusted
     * @param fontMetrics           the font metrics
     * @return                      the adjusted text
     */
    public static String adjustTextWidth(String text,FontMetrics fontMetrics) {
        int width=fontMetrics.stringWidth("...");
        int index=-1;
        do {
            index++;
            width+=fontMetrics.charWidth(text.charAt(index));
        } while (width<MAXIMUM_TEXT_WIDTH);
        return text.substring(0,index)+"...";
    }
    /**
     * Paints the node.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics
     * @param node                  the node to paint
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @param isSelectionPrimary    <code>true</code> if the selection is primary
     * @param isShowBlackAndWhite   <code>true</code> if the graph should be shown in black-and-white
     * @param isShowForCapture      <code>true</code> if the graph should be shown for screen capture
     * @param isInActiveOIModel     <code>true</code> if the painted node is in the active OI-model
     * @param isEnabled             <code>true</code> if the graph pane is enabled
     */
    protected abstract void paintNode(JGraphPane graphPane,Graphics2D g,Node node,boolean isHighlighted,boolean isSelected,boolean isDragging,boolean isSelectionPrimary,boolean isShowBlackAndWhite,boolean isShowForCapture,boolean isInActiveOIModel,boolean isEnabled);
    /**
     * Returns the number of handles in the upper border for this node type.
     *
     * @return                      the number of handles on the upper border for this node type
     */
    public abstract int getUpperBorderNumberOfHandles();
    /**
     * Returns the number of handles in the lower border for this node type.
     *
     * @return                      the number of handles on the lower border for this node type
     */
    public abstract int getLowerBorderNumberOfHandles();
    /**
     * Returns the name of the prefix for the handle tool-tip text.
     *
     * @return                      the name of the prefix for the handle tool-tip text
     */
    public abstract String getHandleToolTipPrefix();
}
