package de.fzi.wim.oimodeler.oimodelgraph.view;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Font;
import javax.swing.UIManager;

import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.view.*;

import de.fzi.wim.oimodeler.oimodelgraph.graph.*;

/**
 * The painter for concept nodes.
 */
public class ConceptNodePainter extends EntityNodePainter {
    /** The background color of the concept. */
    protected static final Color BACKGROUND_COLOR=new Color(242,237,65);
    /** The background color of the concept in non-acitve OI-model. */
    protected static final Color BACKGROUND_COLOR_NON_ACTIVE=new Color(250,248,180);
    /** An instance. */
    public static final NodePainter INSTANCE=new ConceptNodePainter();
    /** The index of the subconcepts handle. */
    public static final int SUBCONCEPTS_HANDLE=0;
    /** The index of the superconcepts handle. */
    public static final int SUPERCONCEPTS_HANDLE=1;
    /** The index of the spanning objects handle. */
    public static final int SPANNING_OBJECTS_HANDLE=2;
    /** The index of the properties to handle. */
    public static final int PROPERTIES_TO_HANDLE=3;
    /** The index of the properties from handle. */
    public static final int PROPERTIES_FROM_HANDLE=4;
    /** The index of the instances handle. */
    public static final int INSTANCES_HANDLE=5;

    /**
     * Paints the node.
     *
     * @param graphPane             the graph pane
     * @param g                     the graphics
     * @param node                  the node to paint
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @param isSelectionPrimary    <code>true</code> if the selection is primary
     * @param isShowBlackAndWhite   <code>true</code> if the graph should be shown in black-and-white
     * @param isShowForCapture      <code>true</code> if the graph should be shown for screen capture
     * @param isInActiveOIModel     <code>true</code> if the painted node is in the active OI-model
     * @param isEnabled             <code>true</code> if the graph pane is enabled
     */
    protected void paintNode(JGraphPane graphPane,Graphics2D g,Node node,boolean isHighlighted,boolean isSelected,boolean isDragging,boolean isSelectionPrimary,boolean isShowBlackAndWhite,boolean isShowForCapture,boolean isInActiveOIModel,boolean isEnabled) {
        ConceptNode conceptNode=(ConceptNode)node;
        Color oldColor=g.getColor();
        g.setColor(getBackgroundColor(isHighlighted,isSelected,isDragging,isSelectionPrimary,isShowBlackAndWhite,isShowForCapture,isInActiveOIModel,isEnabled));
        g.fillRect(m_nodeLabel.x,m_nodeLabel.y,m_nodeLabel.width-1,m_nodeLabel.height-1);
        if (m_adjustedNodeLabel!=null) {
            Font oldFont=g.getFont();
            g.setColor(getTextColor(isHighlighted,isSelected,isDragging,isSelectionPrimary,isShowBlackAndWhite,isInActiveOIModel,isEnabled));
            g.drawString(m_adjustedNodeLabel,m_labelPosition.x,m_labelPosition.y);
            g.setFont(oldFont);
        }
        g.setColor(getBorderColor(isHighlighted,isSelected,isDragging,isSelectionPrimary,isShowBlackAndWhite,isInActiveOIModel,isEnabled));
        g.drawRect(m_nodeLabel.x,m_nodeLabel.y,m_nodeLabel.width-1,m_nodeLabel.height-1);
        if (!isShowForCapture) {
            paintDownArrowHandle(g,m_handles[SUBCONCEPTS_HANDLE],conceptNode.getSubConceptsExpanded());
            paintUpArrowHandle(g,m_handles[SUPERCONCEPTS_HANDLE],conceptNode.getSuperConceptsExpanded());
            paintLeftArrowHandle(g,m_handles[PROPERTIES_TO_HANDLE],conceptNode.getPropertiesToExpanded());
            paintRightArrowHandle(g,m_handles[PROPERTIES_FROM_HANDLE],conceptNode.getPropertiesFromExpanded());
            paintDownArrowHandle(g,m_handles[INSTANCES_HANDLE],conceptNode.getInstancesExpanded());
            paintDottedLineHandle(g,m_handles[SPANNING_OBJECTS_HANDLE],conceptNode.getSpanningObjectsExpanded());
        }
        g.setColor(oldColor);
    }
    /**
     * Returns the border color of the node.
     *
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @param isSelectionPrimary    <code>true</code> if the selection is primary
     * @param isShowBlackAndWhite   <code>true</code> if the graph should be shown in black-and-white
     * @param isInActiveOIModel     <code>true</code> if the painted node is in the active OI-model
     * @param isEnabled             <code>true</code> if the graph pane is enabled
     * @return                      the border color
     */
    protected Color getBorderColor(boolean isHighlighted,boolean isSelected,boolean isDragging,boolean isSelectionPrimary,boolean isShowBlackAndWhite,boolean isInActiveOIModel,boolean isEnabled) {
        if (isShowBlackAndWhite)
            return Color.black;
        else
            return isHighlighted || isDragging ? Color.red : Color.black;
    }
    /**
     * Returns the background color of the node.
     *
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @param isSelectionPrimary    <code>true</code> if the selection is primary
     * @param isShowBlackAndWhite   <code>true</code> if the graph should be shown in black-and-white
     * @param isShowForCapture      <code>true</code> if the graph should be shown for screen capture
     * @param isInActiveOIModel     <code>true</code> if the painted node is in the active OI-model
     * @param isEnabled             <code>true</code> if the graph pane is enabled
     * @return                      the background color
     */
    protected Color getBackgroundColor(boolean isHighlighted,boolean isSelected,boolean isDragging,boolean isSelectionPrimary,boolean isShowBlackAndWhite,boolean isShowForCapture,boolean isInActiveOIModel,boolean isEnabled) {
        if (isShowBlackAndWhite)
            return Color.white;
        else if (isSelected && !isShowForCapture) {
            if (isSelectionPrimary)
                return UIManager.getColor("Tree.selectionBackground");
            else
                return Color.lightGray;
        }
        else if (isInActiveOIModel)
            return BACKGROUND_COLOR;
        else
            return BACKGROUND_COLOR_NON_ACTIVE;
    }
    /**
     * Returns the text color of the node
     *
     * @param isHighlighted         <code>true</code> if the node is highlighted
     * @param isSelected            <code>true</code> if the node is selected
     * @param isDragging            <code>true</code> if the node is being dragged
     * @param isSelectionPrimary    <code>true</code> if the selection is primary
     * @param isShowBlackAndWhite   <code>true</code> if the graph should be shown in black-and-white
     * @param isInActiveOIModel     <code>true</code> if the painted node is in the active OI-model
     * @param isEnabled             <code>true</code> if the graph pane is enabled
     * @return                      the background color
     */
    protected Color getTextColor(boolean isHighlighted,boolean isSelected,boolean isDragging,boolean isSelectionPrimary,boolean isShowBlackAndWhite,boolean isInActiveOIModel,boolean isEnabled) {
        if (isShowBlackAndWhite)
            return Color.black;
        else if (isSelected) {
            if (isSelectionPrimary)
                return UIManager.getColor("Tree.selectionForeground");
            else
                return Color.white;
        }
        else if (isEnabled)
            return UIManager.getColor("Label.foreground");
        else
            return UIManager.getColor("Label.disabledForeground");
    }
    /**
     * Returns the number of handles in the upper border for this node type.
     *
     * @return                      the number of handles on the upper border for this node type
     */
    public int getUpperBorderNumberOfHandles() {
        return 3;
    }
    /**
     * Returns the number of handles in the lower border for this node type.
     *
     * @return                      the number of handles on the lower border for this node type
     */
    public int getLowerBorderNumberOfHandles() {
        return 3;
    }
    /**
     * Returns the name of the prefix for the handle tool-tip text.
     *
     * @return                      the name of the prefix for the handle tool-tip text
     */
    public String getHandleToolTipPrefix() {
        return "oimodeler.tooltip.concept.";
    }
}
