package de.fzi.wim.oimodeler.oimodelgraph.view;

import java.util.Iterator;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.Collection;
import java.io.IOException;
import java.awt.Point;
import java.awt.geom.Point2D;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import javax.swing.JComponent;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

import de.fzi.wim.guibase.dnd.*;
import de.fzi.wim.guibase.selectionsource.*;
import de.fzi.wim.guibase.graphview.view.*;
import de.fzi.wim.guibase.graphview.graph.*;
import de.fzi.wim.guibase.graphview.controller.*;
import de.fzi.wim.guibase.graphview.selection.*;

import de.fzi.wim.oimodeler.ui.*;
import de.fzi.wim.oimodeler.oimodelgraph.*;
import de.fzi.wim.oimodeler.oimodelgraph.graph.*;

/**
 * A basic class for graph panes for rendering OI-models.
 */
public class BasicOIModelGraphPane extends JGraphPane implements SelectionSource,FocusListener {
    /** Set to <code>true</code> if the selection is primary. */
    protected boolean m_selectionPrimary;
    /** The selection source manager. */
    protected SelectionSourceManager m_selectionSourceManager;
    /** The node selection listener. */
    protected NodeSelectionListener m_nodeSelectionListener;
    /** Set to <code>true</code> if the graph should be shown in black-and-white. */
    protected boolean m_showBlackAndWhite;
    /** Set to <code>true</code> if the graph should be shown for screen capture. */
    protected boolean m_showForScreenCapture;

   /**
    * Creates a graph pane.
    *
    * @param oimodelGraph               the OI-model graph
    */
    public BasicOIModelGraphPane(OIModelGraph oimodelGraph) {
        super(oimodelGraph);
        m_showBlackAndWhite=false;
        m_showForScreenCapture=false;
        addFocusListener(this);
        new OIModelGraphPaneDropTarget();
        m_nodeSelectionListener=new NodeSelectionListener() {
            public void nodesAddedToSelection(NodeSelectionModel nodeSelectionModel,Collection nodes) {
                if (m_selectionSourceManager!=null)
                    m_selectionSourceManager.notifySelectionChanged(BasicOIModelGraphPane.this);
            }
            public void nodesRemovedFromSelection(NodeSelectionModel nodeSelectionModel,Collection nodes) {
                if (m_selectionSourceManager!=null)
                    m_selectionSourceManager.notifySelectionChanged(BasicOIModelGraphPane.this);
            }
            public void selectionCleared(NodeSelectionModel nodeSelectionModel) {
                if (m_selectionSourceManager!=null)
                    m_selectionSourceManager.notifySelectionChanged(BasicOIModelGraphPane.this);
            }
        };
    }
    /**
     * Sets the selection source manager.
     *
     * @param selectionSourceManager    the selection source manager
     */
    public void setSelectionSourceManager(SelectionSourceManager selectionSourceManager) {
        m_selectionSourceManager=selectionSourceManager;
        SelectionManipulator manipulator=(SelectionManipulator)getManipulator(SelectionManipulator.NAME);
        if (manipulator!=null) {
            NodeSelectionModel nodeSelectionModel=manipulator.getNodeSelectionModel();
            if (m_selectionSourceManager==null)
                nodeSelectionModel.removeNodeSelectionListener(m_nodeSelectionListener);
            else
                nodeSelectionModel.addNodeSelectionListener(m_nodeSelectionListener);
        }
    }
    /**
     * Returns <code>true</code> if the graph should be shown black-and-white.
     *
     * @return                          <code>true</code> if the graph should be shown black-and-white
     */
    public boolean getShowBlackAndWhite() {
        return m_showBlackAndWhite;
    }
    /**
     * Determines whether the graph should be shown in black-and-white.
     *
     * @param showBlackAndWhite         <code>true</code> if the graph should shown in black-and-white
     */
    public void setShowBlackAndWhite(boolean showBlackAndWhite) {
        m_showBlackAndWhite=showBlackAndWhite;
        repaint();
    }
    /**
     * Returns <code>true</code> if the graph should be shown for screen capture.
     *
     * @return                          <code>true</code> if the graph should be shown for screen capture
     */
    public boolean getShowForScreenCapture() {
        return m_showForScreenCapture;
    }
    /**
     * Determines whether the graph should be shown for screen capture.
     *
     * @param showForScreenCapture      <code>true</code> if the graph should be shown for screen capture
     */
    public void setShowForScreenCapture(boolean showForScreenCapture) {
        m_showForScreenCapture=showForScreenCapture;
        repaint();
    }
    /**
     * Sets the language of the pane.
     *
     * @param languageURI               the language URI
     * @throws KAONException            thrown if there is a problem
     */
    public void setLanguageURI(String languageURI) throws KAONException {
        ((OIModelGraph)getGraph()).setLanguageURI(languageURI);
        OIModelerViewable.setFont(languageURI,this);
    }
    /**
     * Returns the OI-model graph anchor.
     *
     * @return                          the OI-model graph anchor
     */
    public OIModelGraphAnchor getOIModelGraphAnchor() {
        return ((OIModelGraph)getGraph()).getOIModelGraphAnchor();
    }
    /**
     * Returns the painter for the node.
     *
     * @param node                      the node
     * @return                          the painter for the node
     */
    public NodePainter getPainterForNode(Node node) {
        if (node instanceof ConceptNode)
            return ConceptNodePainter.INSTANCE;
        else if (node instanceof PropertyNode)
            return PropertyNodePainter.INSTANCE;
        else if (node instanceof InstanceNode)
            return InstanceNodePainter.INSTANCE;
        else if (node instanceof PropertyInstancesNode)
            return StringNodePainter.INSTANCE;
        else if (node instanceof LiteralPropertyInstanceValueNode)
            return StringNodePainter.INSTANCE;
        else if (node instanceof FanOutNode)
            return FanOutNodePainter.INSTANCE;
        else
            return super.getPainterForNode(node);
    }
    /**
     * Returns the painter for the edge.
     *
     * @param edge                      the edge
     * @return                          the painter for the edge
     */
    public EdgePainter getPainterForEdge(Edge edge) {
        if ((edge instanceof ConceptHierarchyEdge) || (edge instanceof PropertyHierarchyEdge))
            return ArrowEdgePainter.INSTANCE;
        else if ((edge instanceof PropertyDomainEdge) || (edge instanceof PropertyRangeEdge))
            return DomainRangeEdgePainter.INSTANCE;
        else if (edge instanceof PropertyInstanceEdge)
            return PropertyInstanceEdgePainter.INSTANCE;
        else if (edge instanceof SpanningObjectEdge)
            return SpanningObjectEdgePainter.INSTANCE;
        else
            return super.getPainterForEdge(edge);
    }
    /**
     * Returns the current selection.
     *
     * @return                          the current selection
     */
    public Collection getSelection() {
        Collection selection=new HashSet();
        SelectionManipulator manipulator=(SelectionManipulator)getManipulator(SelectionManipulator.NAME);
        if (manipulator!=null) {
            Iterator nodes=manipulator.getNodeSelectionModel().getSelectedNodes().iterator();
            while (nodes.hasNext()) {
                Node node=(Node)nodes.next();
                if (node instanceof EntityNode) {
                    EntityNode entityNode=(EntityNode)node;
                    selection.add(entityNode.getEntity());
                }
            }
        }
        return selection;
    }
    /**
     * Notifies the source that his selection is currently the primary selection.
     */
    public void notifySelectionPrimary() {
        m_selectionPrimary=true;
        repaint();
    }
    /**
     * Notifies the source that his selection is currently the secondary selection.
     */
    public void notifySelectionSecondary() {
        m_selectionPrimary=false;
        repaint();
    }
    /**
     * Return <code>true</code> if the selection is primary.
     *
     * @return                          <code>true</code> if the selection is primary
     */
    public boolean isSelectionPrimary() {
        return m_selectionPrimary;
    }
    /**
     * The component of the selection source.
     *
     * @return                                  the component of the selection source
     */
    public JComponent getComponent() {
        return this;
    }
    /**
     * Called when this component gains focus.
     *
     * @param e                                 the event
     */
    public void focusGained(FocusEvent e) {
        if (m_selectionSourceManager!=null)
            m_selectionSourceManager.setActiveSelectionSource(this);
    }
    /**
     * Called when this component looses focus.
     *
     * @param e                                 the event
     */
    public void focusLost(FocusEvent e) {
    }

    /**
     * Drop target for the graph pane.
     */
    public class OIModelGraphPaneDropTarget implements DropTargetListener {
        /** Drop target for the table. */
        protected DropTarget m_dropTarget;

        /**
         * Attachs this manputlator to the enclosing pane.
         */
        public OIModelGraphPaneDropTarget() {
            m_dropTarget=new SmartDropTarget(BasicOIModelGraphPane.this,this);
        }
        /**
         * Called when a drag operation has encountered the DropTarget.
         *
         * @param dtde                              the event
         */
        public void dragEnter(DropTargetDragEvent dtde) {
            processDragEvent(dtde);
        }
        /**
         * The drag operation has departed the DropTarget without dropping.
         *
         * @param dte                               the event
         */
        public void dragExit(DropTargetEvent dte) {
        }
        /**
         * Called when a drag operation is ongoing on the DropTarget.
         *
         * @param dtde                              the event
         */
        public void dragOver(DropTargetDragEvent dtde) {
            processDragEvent(dtde);
        }
        /**
         * The drag operation has terminated with a drop on this DropTarget.
         *
         * @param dtde                              the event
         */
        public void drop(DropTargetDropEvent dtde) {
            HighlightingManipulator highlightingManipulator=(HighlightingManipulator)getManipulator(HighlightingManipulator.NAME);
            highlightingManipulator.setHighlightedNode(null);
            highlightingManipulator.setHighlightedEdge(null);
            Collection collection;
            try {
                collection=(Collection)dtde.getTransferable().getTransferData(CollectionSelection.s_objectCollectionFlavor);
            }
            catch (IOException ignored) {
                dtde.rejectDrop();
                return;
            }
            catch (UnsupportedFlavorException ignored) {
                dtde.rejectDrop();
                return;
            }
            dtde.acceptDrop(dtde.getSourceActions());
            try {
                // it is important to get the node before elements are added, because I will always get the added element
                Node dropOverNode=getNodeAtPoint(dtde.getLocation());
                addElementsToGraph(collection,dtde.getLocation());
                if (dropOverNode instanceof ConceptNode)
                    dropOverConcept(((ConceptNode)dropOverNode).getConcept(),collection);
                else if (dropOverNode instanceof PropertyNode)
                    dropOverProperty(((PropertyNode)dropOverNode).getProperty(),collection);
            }
            catch (KAONException error) {
                getOIModelGraphAnchor().notifyError(error);
            }
        }
        /**
         * Called if the user has modified the current drop gesture.
         *
         * @param dtde                              the event
         */
        public void dropActionChanged(DropTargetDragEvent dtde) {
        }
        /**
         * Processes a drag event.
         *
         * @param dtde                              the event
         */
        protected void processDragEvent(DropTargetDragEvent dtde) {
            HighlightingManipulator highlightingManipulator=(HighlightingManipulator)getManipulator(HighlightingManipulator.NAME);
            if (dtde.isDataFlavorSupported(CollectionSelection.s_objectCollectionFlavor)) {
                dtde.acceptDrag(dtde.getSourceActions());
                Node dragOverNode=getNodeAtPoint(dtde.getLocation());
                highlightingManipulator.setHighlightedNode(dragOverNode);
                highlightingManipulator.setHighlightedEdge(null);
            }
            else {
                dtde.rejectDrag();
                highlightingManipulator.setHighlightedNode(null);
                highlightingManipulator.setHighlightedEdge(null);
            }
        }
        /**
         * Called when elements are dropped over a concept.
         *
         * @param concept                           the concept over which elements are dropped
         * @param collection                        the dropped elements
         */
        protected void dropOverConcept(Concept concept,Collection collection) {
            List events=new LinkedList();
            Iterator iterator=collection.iterator();
            while (iterator.hasNext()) {
                Object object=iterator.next();
                if (object instanceof Property)
                    events.add(new AddPropertyDomain(getOIModelGraphAnchor().getActiveOIModel(),null,(Property)object,concept));
                else if (object instanceof Concept)
                    events.add(new AddSubConcept(getOIModelGraphAnchor().getActiveOIModel(),null,concept,(Concept)object));
            }
            getOIModelGraphAnchor().changeOIModel(events);
        }
        /**
         * Called when elements are dropped over a property.
         *
         * @param property                          the property over which elements are dropped
         * @param collection                        the dropped elements
         */
        protected void dropOverProperty(Property property,Collection collection) {
            List events=new LinkedList();
            Iterator iterator=collection.iterator();
            while (iterator.hasNext()) {
                Object object=iterator.next();
                if (object instanceof Concept)
                    events.add(new AddPropertyDomain(getOIModelGraphAnchor().getActiveOIModel(),null,property,(Concept)object));
                else if (object instanceof Property)
                    events.add(new AddSubProperty(getOIModelGraphAnchor().getActiveOIModel(),null,property,(Property)object));
            }
            getOIModelGraphAnchor().changeOIModel(events);
        }
        /**
         * Called to add dropped elements to the graph.
         *
         * @param collection                        the dropped elements
         * @param point                             the location where elements were dropped
         * @throws KAONException                    thrown if there is a problem
         */
        protected void addElementsToGraph(Collection collection,Point point) throws KAONException {
            Point2D graphPoint=new Point2D.Double();
            screenToGraphPoint(point,graphPoint);
            OIModelGraph oimodelGraph=(OIModelGraph)getGraph();
            try {
                oimodelGraph.suspendEvents();
                Iterator iterator=collection.iterator();
                while (iterator.hasNext()) {
                    Object object=iterator.next();
                    if (object instanceof Entity) {
                        Entity entity=(Entity)object;

                        EntityNode entityNode=oimodelGraph.getNodeForEntity(entity);
                        if (entityNode==null) {
                            if (entity instanceof Concept)
                                entityNode=new ConceptNode((Concept)entity,null);
                            else if (entity instanceof Property)
                                entityNode=new PropertyNode((Property)entity,null);
                            else if (entity instanceof Instance)
                                entityNode=new InstanceNode((Instance)entity,null);
                            if (entityNode!=null) {
                                entityNode.setLocation(graphPoint.getX(),graphPoint.getY());
                                oimodelGraph.add(entityNode);
                            }
                        }
                        else
                            entityNode.setLocation(graphPoint.getX(),graphPoint.getY());
                    }
                }
                oimodelGraph.notifyUpdated();
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
}
