package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Set;
import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.graphview.graph.*;

import de.fzi.wim.oimodeler.viewfilter.*;

/**
 * A node in the OI-model graph representing a property.
 */
public class PropertyNode extends EntityNode {
    /** Set to <code>true</code> if the property is an attribute. */
    protected boolean m_isAttribute;
    /** The expansion state of the subproperties. */
    protected int m_subPropertiesExpanded;
    /** The expansion state of the superproperties. */
    protected int m_superPropertiesExpanded;
    /** The expansion state of the domain concepts. */
    protected int m_domainConceptsExpanded;
    /** The expansion state of the range concepts. */
    protected int m_rangeConceptsExpanded;

    /**
     * Creates a node for given entity.
     *
     * @param property                      the property of the node
     * @param originalNode                  the original node (may be <code>null</code>)
     * @throws KAONException                thrown if there is an error with accessing the ontology
     */
    public PropertyNode(Property property,Node originalNode) throws KAONException {
        super(property,originalNode);
        updateIsAttribute();
    }
    /**
     * Returns <code>true</code> if the property is an attribute.
     *
     * @return                              <code>true</code> if the property is an attribute
     */
    public boolean isAttribute() {
        return m_isAttribute;
    }
    /**
     * Updates the attribute status of the node.
     *
     * @throws KAONException                thrown if there is an error with accessing the ontology
     */
    public void updateIsAttribute() throws KAONException {
        m_isAttribute=getProperty().isAttribute();
    }
    /**
     * Returns the property of this node.
     *
     * @return                              the property of this node
     */
    public Property getProperty() {
        return (Property)m_entity;
    }
    /**
     * Expands the spanning objects obf this entity.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSpanningObjects(final OIModelGraph oimodelGraph) throws KAONException {
        m_spanningObjectsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Instance m_spanningInstance;
            public void backgroundJob() throws KAONException {
                m_spanningInstance=getProperty().getSpanningInstance();
                if ((m_spanningInstance.getLoadedState() & OIModel.LOAD_INSTANCE_BASICS)==0)
                    oimodelGraph.getOIModel().loadObjects(Collections.singleton(m_spanningInstance),OIModel.LOAD_LEXICON | OIModel.LOAD_INSTANCE_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(PropertyNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                if (m_spanningInstance.isInOIModel()) {
                                    InstanceNode spanningInstanceNode=(InstanceNode)oimodelGraph.getNodeForEntity(m_spanningInstance);
                                    if (spanningInstanceNode==null) {
                                        spanningInstanceNode=new InstanceNode(m_spanningInstance,PropertyNode.this);
                                        oimodelGraph.add(spanningInstanceNode);
                                    }
                                    if (SpanningObjectEdge.getEdge(oimodelGraph,PropertyNode.this,spanningInstanceNode)==null)
                                        oimodelGraph.add(new SpanningObjectEdge(PropertyNode.this,spanningInstanceNode));
                                }
                                m_spanningObjectsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_spanningObjectsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion state of the subproperties.
     *
     * @return                              the expansion state of the subproperties
     */
    public int getSubPropertiesExpanded() {
        return m_subPropertiesExpanded;
    }
    /**
     * Expands the subproperties in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSubProperties(final OIModelGraph oimodelGraph) throws KAONException {
        m_subPropertiesExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_subProperties;
            public void backgroundJob() throws KAONException {
                m_subProperties=getProperty().getSubProperties();
                oimodelGraph.getOIModel().loadObjects(m_subProperties,OIModel.LOAD_LEXICON | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(PropertyNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                double edgeLength=PropertyHierarchyEdge.UNIT_LENGTH;
                                int count=0;
                                Iterator iterator=m_subProperties.iterator();
                                while (iterator.hasNext()) {
                                    Property subProperty=(Property)iterator.next();
                                    if (viewFilter.showEntity(subProperty)) {
                                        PropertyNode subPropertyNode=(PropertyNode)oimodelGraph.getNodeForEntity(subProperty);
                                        if (subPropertyNode==null) {
                                            subPropertyNode=new PropertyNode(subProperty,PropertyNode.this);
                                            oimodelGraph.add(subPropertyNode);
                                        }
                                        if (PropertyHierarchyEdge.getEdge(oimodelGraph,PropertyNode.this,subPropertyNode)==null)
                                            oimodelGraph.add(new PropertyHierarchyEdge(PropertyNode.this,subPropertyNode,edgeLength));
                                        count++;
                                        if (count % 6==0)
                                            edgeLength+=PropertyHierarchyEdge.UNIT_LENGTH;
                                    }
                                }
                                m_subPropertiesExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_subPropertiesExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion state of the superproperties.
     *
     * @return                              the expansion state of the superproperties
     */
    public int getSuperPropertiesExpanded() {
        return m_superPropertiesExpanded;
    }
    /**
     * Expands the superproperties in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandSuperProperties(final OIModelGraph oimodelGraph) throws KAONException {
        m_superPropertiesExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_superProperties;
            public void backgroundJob() throws KAONException {
                m_superProperties=getProperty().getSuperProperties();
                oimodelGraph.getOIModel().loadObjects(m_superProperties,OIModel.LOAD_LEXICON | OIModel.LOAD_PROPERTY_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(PropertyNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                double edgeLength=PropertyHierarchyEdge.UNIT_LENGTH;
                                int count=0;
                                Iterator iterator=m_superProperties.iterator();
                                while (iterator.hasNext()) {
                                    Property superProperty=(Property)iterator.next();
                                    if (viewFilter.showEntity(superProperty)) {
                                        PropertyNode superPropertyNode=(PropertyNode)oimodelGraph.getNodeForEntity(superProperty);
                                        if (superPropertyNode==null) {
                                            superPropertyNode=new PropertyNode(superProperty,PropertyNode.this);
                                            oimodelGraph.add(superPropertyNode);
                                        }
                                        if (PropertyHierarchyEdge.getEdge(oimodelGraph,superPropertyNode,PropertyNode.this)==null)
                                            oimodelGraph.add(new PropertyHierarchyEdge(superPropertyNode,PropertyNode.this,edgeLength));
                                        count++;
                                        if (count % 6==0)
                                            edgeLength+=PropertyHierarchyEdge.UNIT_LENGTH;
                                    }
                                }
                                m_superPropertiesExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_superPropertiesExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion state of the domain concepts.
     *
     * @return                              the expansion state of the domain concepts
     */
    public int getDomainConceptsExpanded() {
        return m_domainConceptsExpanded;
    }
    /**
     * Expands the domain concepts in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandDomainConcepts(final OIModelGraph oimodelGraph) throws KAONException {
        m_domainConceptsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_domainConcepts;
            public void backgroundJob() throws KAONException {
                m_domainConcepts=getProperty().getDomainConcepts();
                oimodelGraph.getOIModel().loadObjects(m_domainConcepts,OIModel.LOAD_LEXICON | OIModel.LOAD_CONCEPT_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(PropertyNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                Iterator iterator=m_domainConcepts.iterator();
                                while (iterator.hasNext()) {
                                    Concept concept=(Concept)iterator.next();
                                    if (viewFilter.showEntity(concept)) {
                                        ConceptNode conceptNode=(ConceptNode)oimodelGraph.getNodeForEntity(concept);
                                        if (conceptNode==null) {
                                            conceptNode=new ConceptNode(concept,PropertyNode.this);
                                            oimodelGraph.add(conceptNode);
                                        }
                                        if (PropertyDomainEdge.getEdge(oimodelGraph,PropertyNode.this,conceptNode)==null)
                                            oimodelGraph.add(new PropertyDomainEdge(PropertyNode.this,conceptNode));
                                    }
                                }
                                m_domainConceptsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_domainConceptsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Returns the expansion state of the range concepts.
     *
     * @return                              the expansion state of the range concepts
     */
    public int getRangeConceptsExpanded() {
        return m_rangeConceptsExpanded;
    }
    /**
     * Expands the range concepts in the context of the given graph.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void expandRangeConcepts(final OIModelGraph oimodelGraph) throws KAONException {
        m_rangeConceptsExpanded=LOADING;
        doAction(oimodelGraph,new BackgroundAction() {
            protected Set m_rangeConcepts;
            public void backgroundJob() throws KAONException {
                m_rangeConcepts=getProperty().getRangeConcepts();
                oimodelGraph.getOIModel().loadObjects(m_rangeConcepts,OIModel.LOAD_LEXICON | OIModel.LOAD_CONCEPT_BASICS);
            }
            public void completionJob() throws KAONException {
                synchronized (oimodelGraph) {
                    synchronized (oimodelGraph.getOIModel().getKAONConnection()) {
                        if (oimodelGraph.contains(PropertyNode.this))
                            try {
                                oimodelGraph.suspendEvents();
                                ViewFilter viewFilter=oimodelGraph.getOIModelGraphAnchor().getViewFilter();
                                Iterator iterator=m_rangeConcepts.iterator();
                                while (iterator.hasNext()) {
                                    Concept concept=(Concept)iterator.next();
                                    if (viewFilter.showEntity(concept)) {
                                        ConceptNode conceptNode=(ConceptNode)oimodelGraph.getNodeForEntity(concept);
                                        if (conceptNode==null) {
                                            conceptNode=new ConceptNode(concept,PropertyNode.this);
                                            oimodelGraph.add(conceptNode);
                                        }
                                        if (PropertyRangeEdge.getEdge(oimodelGraph,PropertyNode.this,conceptNode)==null)
                                            oimodelGraph.add(new PropertyRangeEdge(PropertyNode.this,conceptNode));
                                    }
                                }
                                m_rangeConceptsExpanded=LOADED;
                                oimodelGraph.notifyUpdated();
                            }
                            finally {
                                oimodelGraph.resumeEvents();
                            }
                    }
                }
            }
            public void errorJob() {
                synchronized (oimodelGraph) {
                    m_rangeConceptsExpanded=NOT_LOADED;
                    oimodelGraph.notifyUpdated();
                }
            }
        });
        oimodelGraph.notifyUpdated();
    }
    /**
     * Resets the node's expanded states.
     */
    public void resetExpandedState() {
        super.resetExpandedState();
        m_subPropertiesExpanded=NOT_LOADED;
        m_superPropertiesExpanded=NOT_LOADED;
        m_domainConceptsExpanded=NOT_LOADED;
        m_rangeConceptsExpanded=NOT_LOADED;
    }
}
