package de.fzi.wim.oimodeler.oimodelgraph.graph;

import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.Collection;
import java.util.Random;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

import de.fzi.wim.guibase.graphview.graph.*;

/**
 * A node in the OI-model graph representing a set of property instances of the same property.
 */
public class PropertyInstancesNode extends DefaultNode {
    /** The label. */
    protected String m_label;
    /** The property for this node. */
    protected Property m_property;
    /** The source instance for this node. */
    protected Instance m_sourceInstance;
    /** The map of the nodes pointing out of this node indexed by the value. */
    protected Map m_pointedNodes;

    /**
     * Creates a node for given entity.
     *
     * @param property                      the property of the node
     * @param sourceInstance                the source instnace
     * @param originalNode                  the original node (may be <code>null</code>)
     */
    public PropertyInstancesNode(Property property,Instance sourceInstance,Node originalNode) {
        m_property=property;
        m_sourceInstance=sourceInstance;
        if (originalNode!=null) {
            Random random=new Random();
            setLocation(originalNode.getX()+random.nextInt(60)-30,originalNode.getY()+random.nextInt(60)-30);
        }
        m_pointedNodes=new HashMap();
    }
    /**
     * Returns the property of this node.
     *
     * @return                              the property of this node
     */
    public Property getProperty() {
        return m_property;
    }
    /**
     * Returns the source instance of this node.
     *
     * @return                              the source instance of this node
     */
    public Instance getSourceInstance() {
        return m_sourceInstance;
    }
    /**
     * Returns the set of values from this node.
     *
     * @return                              the set of values from this node
     */
    public Set getTargetValues() {
        return m_pointedNodes.keySet();
    }
    /**
     * Returns the collection of nodes pointed by this node.
     *
     * @return                              the collection of nodes pointed by this node
     */
    public Collection getTargetNodes() {
        return m_pointedNodes.values();
    }
    /**
     * Updates the label for given language.
     *
     * @param oimodelGraph                  the graph
     * @throws KAONException                thrown if there is a problem with accessing the ontology
     */
    public void updateLabel(OIModelGraph oimodelGraph) throws KAONException {
        m_label=m_property.getLabel(oimodelGraph.getLanguageURI());
        if (m_label==null)
            m_label=oimodelGraph.replaceURIPrefix(m_property.getURI());
    }
    /**
     * Return the label of this node.
     *
     * @return                               the label of this node
     */
    public String getLabel() {
        return m_label;
    }
    /**
     * Adds a property instance to this node.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstance              the property instance
     * @throws KAONException                thrown if there is a problem
     */
    public void addPropertyInstance(OIModelGraph oimodelGraph,PropertyInstance propertyInstance) throws KAONException {
        synchronized (oimodelGraph) {
            try {
                oimodelGraph.suspendEvents();
                Object value=propertyInstance.getTargetValue();
                if (!m_pointedNodes.containsKey(value)) {
                    Node pointedNode;
                    if (value instanceof Instance) {
                        pointedNode=oimodelGraph.getNodeForEntity((Instance)value);
                        if (pointedNode==null) {
                            pointedNode=new InstanceNode((Instance)value,this);
                            oimodelGraph.add(pointedNode);
                        }
                    }
                    else {
                        pointedNode=new LiteralPropertyInstanceValueNode(propertyInstance,this);
                        oimodelGraph.add(pointedNode);
                    }
                    m_pointedNodes.put(value,pointedNode);
                    oimodelGraph.add(new PropertyInstanceEdge(this,pointedNode));
                }
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Removes a property instance to this node.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstance              the property instance
     * @return                              <code>true</code> if this is the last property instance
     * @throws KAONException                thrown if there is a problem
     */
    public boolean removePropertyInstance(OIModelGraph oimodelGraph,PropertyInstance propertyInstance) throws KAONException {
        synchronized (oimodelGraph) {
            try {
                oimodelGraph.suspendEvents();
                Object value=propertyInstance.getTargetValue();
                Node pointedNode=(Node)m_pointedNodes.remove(value);
                if (pointedNode!=null) {
                    if (pointedNode instanceof LiteralPropertyInstanceValueNode)
                        oimodelGraph.remove(pointedNode);
                    oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(this,pointedNode));
                }
                return m_pointedNodes.isEmpty();
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
    /**
     * Changes a property instance at this node.
     *
     * @param oimodelGraph                  the graph
     * @param propertyInstance              the property instance
     * @param newTargetValue                the new target value of the instance
     * @throws KAONException                thrown if there is a problem
     */
    public void changePropertyInstance(OIModelGraph oimodelGraph,PropertyInstance propertyInstance,Object newTargetValue) throws KAONException {
        synchronized (oimodelGraph) {
            try {
                oimodelGraph.suspendEvents();
                Object value=propertyInstance.getTargetValue();
                PropertyInstance newPropertyInstance=propertyInstance.getOIModel().getPropertyInstance(propertyInstance.getProperty(),propertyInstance.getSourceInstance(),newTargetValue);
                if (!(value instanceof Instance) && !(newTargetValue instanceof Instance)) {
                    LiteralPropertyInstanceValueNode pointedNode=(LiteralPropertyInstanceValueNode)m_pointedNodes.remove(value);
                    if (pointedNode!=null) {
                        m_pointedNodes.put(newTargetValue,pointedNode);
                        pointedNode.setPropertyInstance(newPropertyInstance);
                        oimodelGraph.notifyUpdated();
                    }
                }
                else {
                    Node pointedNode=(Node)m_pointedNodes.remove(value);
                    if (pointedNode!=null) {
                        if (pointedNode instanceof LiteralPropertyInstanceValueNode)
                            oimodelGraph.remove(pointedNode);
                        oimodelGraph.remove(oimodelGraph.getEdgeBetweenNodes(this,pointedNode));
                    }
                    if (!m_pointedNodes.containsKey(newTargetValue)) {
                        Node newPointedNode;
                        if (newTargetValue instanceof Instance) {
                            newPointedNode=oimodelGraph.getNodeForEntity((Instance)newTargetValue);
                            if (newPointedNode==null) {
                                newPointedNode=new InstanceNode((Instance)newTargetValue,this);
                                oimodelGraph.add(newPointedNode);
                            }
                        }
                        else {
                            newPointedNode=new LiteralPropertyInstanceValueNode(newPropertyInstance,this);
                            oimodelGraph.add(newPointedNode);
                        }
                        m_pointedNodes.put(newTargetValue,newPointedNode);
                        oimodelGraph.add(new PropertyInstanceEdge(this,newPointedNode));
                    }
                }
            }
            finally {
                oimodelGraph.resumeEvents();
            }
        }
    }
}
